// 
// Notice Regarding Standards.  AMD does not provide a license or sublicense to
// any Intellectual Property Rights relating to any standards, including but not
// limited to any audio and/or video codec technologies such as MPEG-2, MPEG-4;
// AVC/H.264; HEVC/H.265; AAC decode/FFMPEG; AAC encode/FFMPEG; VC-1; and MP3
// (collectively, the "Media Technologies"). For clarity, you will pay any
// royalties due for such third party technologies, which may include the Media
// Technologies that are owed as a result of AMD providing the Software to you.
// 
// MIT license 
// 
// Copyright (c) 2016 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//

#include <iostream>
#include <vector>

#include "OGLApplication.h"
#include "Timer.h"

#ifdef _WIN32
    #define _CRT_NONSTDC_NO_DEPRECATE
    #define _CRT_SECURE_NO_WARNINGS
    #define glGetProcAddress wglGetProcAddress
#elif __linux__
    #include <X11/Xlib.h>
    #include <sys/mman.h>
    #include <stdio.h>
    #include <string.h>
    #include <GL/glx.h>
    #define glGetProcAddress glXGetProcAddress
#endif

#include <climits>
#include <cassert>


// initialize static variables
OGLApplication * OGLApplication::CurrentApp = 0;


const char * vertexShaderString = "\
    #version 450        \n\
    out vec2 v_texcoord;\n\
                        \n\
    void main()         \n\
    {                   \n\
        gl_Position = gl_ModelViewProjectionMatrix * gl_Vertex; \n\
        v_texcoord.x = gl_Vertex.x / 2.0 + 0.5;                 \n\
        v_texcoord.y = gl_Vertex.y / -2.0 + 0.5;                \n\
    }\
";

const char * yuv420_8_FragmentShaderString = " \
    #version 450                        \n\
    in vec2 v_texcoord;                 \n\
    uniform samplerBuffer textureBuffer;\n\
    uniform int bufferWidth;            \n\
    uniform int bufferHeight;           \n\
    float UV_420_8(int x, int y, int width, int height, bool v) \n\
    {                                                           \n\
        int offset = width * height;                            \n\
        if (v)                                                  \n\
        {                                                       \n\
            offset += (width * height) >> 2;                    \n\
        }                                                       \n\
        offset += (y >> 2) * width + (x >> 1);                  \n\
        return texelFetch(textureBuffer, offset).r;             \n\
    }                                                           \n\
    void main()                                                 \n\
    {                                                           \n\
        int x = int(abs(v_texcoord).x * bufferWidth);           \n\
        int y = int(abs(v_texcoord).y * bufferHeight);          \n\
        float Y, U, V;                                          \n\
        Y = texelFetch(textureBuffer, y * bufferWidth + x).r;   \n\
        U = UV_420_8(x, y, bufferWidth, bufferHeight, false);   \n\
        V = UV_420_8(x, y, bufferWidth, bufferHeight, true);    \n\
        Y = clamp(Y, 16.f/255.f, 235.f/255.f);                  \n\
        U = clamp(U, 16.f/255.f, 240.f/255.f);                  \n\
        V = clamp(V, 16.f/255.f, 240.f/255.f);                  \n\
        Y -= 16.f/255.f;                                        \n\
        U -= 0.5f;                                              \n\
        V -= 0.5f;                                              \n\
        float r = 1.164384f * Y                 + 1.79263f * V; \n\
        float g = 1.164384f * Y - 0.21321f * U - 0.53302f * V;  \n\
        float b = 1.164384f * Y + 2.11244f * U;                 \n\
        gl_FragColor = vec4(r, g, b, 1.0);                      \n\
    }";



OGLApplication::OGLApplication()
{
    CurrentApp = this;

    m_Application = nullptr;
    m_Settings = nullptr;
    m_IsInteropUsed = false;

    m_Window = 0;
    m_Width = 0;
    m_Height = 0;

    m_Program = 0;
    m_IsFirstFrame = true;

    m_readBuffer = 0;
    m_waitBuffer = 0;
}

OGLApplication::~OGLApplication()
{
    CurrentApp = nullptr;

    Term();
}

void OGLApplication::ProcessKeys(unsigned char key, int x, int y)
{
    if (CurrentApp != NULL)
    {
        int translatedCode;

        switch (key)
        {
        case 27:
            translatedCode = KEY_ESCAPE;
            break;
        case 32:
            translatedCode = KEY_SPACE;
            break;
        case 'f':
        case 'F':
            translatedCode = KEY_FULLSCREEN;
            break;
        case 'v':
        case 'V':
            translatedCode = KEY_FPSCONTROL;
            break;
        default:
            return;
        }

        CurrentApp->m_Application->HandleKey(translatedCode);
    }
}

void OGLApplication::Reshape(int w, int h)
{
    glViewport(0, 0, w, h);
}

void OGLApplication::Render()
{
    // if we don't have a pointer to the App, 
    // there's nothing to do...
    if (!CurrentApp)
        return;

    if (!CurrentApp->m_Application->HandleFrame())
    {
        // exit
        glutLeaveMainLoop();
        if (CurrentApp->m_Window != 0)
            CloseWindow();
        exit(0);
    }
}

void OGLApplication::CloseWindow()
{
    // When glutCloseFunc is activated, we don't need to call glutDestroyWindow
    if (CurrentApp && CurrentApp->m_Window != 0)
    {
        CurrentApp->m_Window = 0;
        CurrentApp->Stop();
    }
}

GLuint OGLApplication::CompileShader(GLenum type, const char * string) 
{
    GLuint shader = glCreateShader(type);
    if (shader == 0 || shader == GL_INVALID_ENUM)
        return 0;

    const GLchar *sources = (GLchar *)string;
    glShaderSource(shader, 1, &sources, NULL);
    glCompileShader(shader);

    GLint status = 0;
    glGetShaderiv(shader, GL_COMPILE_STATUS, &status);
    if (status == GL_FALSE) 
    {
        GLchar InfoLog[1024];
        glGetShaderInfoLog(shader, 1024, NULL, InfoLog);
        std::cout << "Error compiling shader:" << InfoLog << std::endl;
        glDeleteShader(shader);
        return 0;
    }

    return shader;
}



//
//
// Initialize
//
//

bool  OGLApplication::InitWindow(int /*width*/, int /*height*/)
{
    // we're sizing the window to the screen not to 
    // the max image resolution
#ifdef _WIN32
    const int width  = GetSystemMetrics(SM_CXSCREEN);
    const int height = GetSystemMetrics(SM_CYSCREEN);
#elif __linux__
    Display* disp = XOpenDisplay(NULL);
    Screen*  scrn = DefaultScreenOfDisplay(disp);
    int height = scrn->height;
    int width  = scrn->width;
#endif

    glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE);
    glutInitWindowSize(width, height);

    m_Window = glutCreateWindow("OpenGL");

    return true;
}

bool  OGLApplication::InitExtensions()
{
#ifdef _WIN32
    glCreateBuffers = (PFNGLCREATEBUFFERSPROC) glGetProcAddress("glCreateBuffers");
    glNamedBufferStorage = (PFNGLNAMEDBUFFERSTORAGEPROC) glGetProcAddress("glNamedBufferStorage");
    wglGetSwapIntervalEXT = (PFNWGLGETSWAPINTERVALEXTPROC) glGetProcAddress("wglGetSwapIntervalEXT");
    wglSwapIntervalEXT = (PFNWGLSWAPINTERVALEXTPROC) glGetProcAddress("wglSwapIntervalEXT");
#elif __linux__
    glCreateBuffers = (PFNGLCREATEBUFFERSPROC) glGetProcAddress((const GLubyte*)"glCreateBuffers");
    glNamedBufferStorage = (PFNGLNAMEDBUFFERSTORAGEPROC) glGetProcAddress((const GLubyte*)"glNamedBufferStorage");
    wglSwapIntervalEXT = (PFNWGLSWAPINTERVALEXTPROC) glGetProcAddress((const GLubyte*)"glXSwapIntervalSGI");
#endif

    return (glCreateBuffers != NULL) &&
           (glNamedBufferStorage != NULL) &&
#ifdef _WIN32
           (wglGetSwapIntervalEXT != NULL) &&
#endif
           (wglSwapIntervalEXT != NULL);
}

bool  OGLApplication::InitSsgExtensions()
{
#ifdef _WIN32
    glCreateFileAMD = (PFNGLCREATEFILEAMDPROC) glGetProcAddress("glCreateFileAMD");
    glReleaseFileAMD = (PFNGLRELEASEFILEAMDPROC) glGetProcAddress("glReleaseFileAMD");
    glGetFileParameteri64vAMD = (PFNGLGETFILEPARAMETERI64VAMDPROC) glGetProcAddress("glGetFileParameteri64vAMD");
    glReadFileAMD = (PFNGLREADFILEAMDPROC) glGetProcAddress("glReadFileAMD");
    glWriteFileAMD = (PFNGLWRITEFILEAMDPROC) glGetProcAddress("glWriteFileAMD");
#elif __linux__
    glCreateFileAMD = (PFNGLCREATEFILEAMDPROC) glGetProcAddress((const GLubyte*)"glCreateFileAMD");
    glReleaseFileAMD = (PFNGLRELEASEFILEAMDPROC) glGetProcAddress((const GLubyte*)"glReleaseFileAMD");
    glGetFileParameteri64vAMD = (PFNGLGETFILEPARAMETERI64VAMDPROC) glGetProcAddress((const GLubyte*)"glGetFileParameteri64vAMD");
    glReadFileAMD = (PFNGLREADFILEAMDPROC) glGetProcAddress((const GLubyte*)"glReadFileAMD");
    glWriteFileAMD = (PFNGLWRITEFILEAMDPROC) glGetProcAddress((const GLubyte*)"glWriteFileAMD");
#endif

    return (glCreateFileAMD != NULL) &&
           (glReleaseFileAMD != NULL) &&
           (glGetFileParameteri64vAMD != NULL) &&
           (glReadFileAMD != NULL) &&
           (glWriteFileAMD != NULL);
}

bool  OGLApplication::InitProgram()
{
    m_Program = glCreateProgram();

    GLuint vertShader = CompileShader(GL_VERTEX_SHADER, vertexShaderString);
    GLuint fragShader = CompileShader(GL_FRAGMENT_SHADER, yuv420_8_FragmentShaderString);

    glAttachShader(m_Program, vertShader);
    glAttachShader(m_Program, fragShader);
    glLinkProgram(m_Program);
    glValidateProgram(m_Program);

    GLint status = 0;
    glGetProgramiv(m_Program, GL_VALIDATE_STATUS, &status);
    if (status == GL_FALSE)
    {
        GLchar ErrorLog[1024] = { 0 };
        glGetProgramInfoLog(m_Program, sizeof(ErrorLog), NULL, ErrorLog);
        std::cout <<"Invalid shader program:" << ErrorLog << std::endl;
        return false;
    }

    // program
    glUseProgram(m_Program);
    m_TexBufferLocation = glGetUniformLocation(m_Program, "textureBuffer");
    assert(m_TexBufferLocation != -1);
    glUniform1i(m_TexBufferLocation, 0);

    m_BufWidthLocation  = glGetUniformLocation(m_Program, "bufferWidth");
    m_BufHeightLocation = glGetUniformLocation(m_Program, "bufferHeight");
    assert(m_BufWidthLocation != -1 && m_BufHeightLocation != -1);

    glUniform1i(m_BufWidthLocation, m_Width);
    glUniform1i(m_BufHeightLocation, m_Height);
    return true;
}

bool  OGLApplication::InitTextures()
{
    glGenTextures((int) m_Textures.size(), m_Textures.data());

    for (size_t i = 0; i < m_Textures.size(); i++)
    {
        glBindTexture(GL_TEXTURE_BUFFER, m_Textures[i]);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    }

    return true;
}

bool  OGLApplication::InitBuffers(int dataBufferSize)
{
    for (int i = 0; i < m_NumBuffers; i++)
    {
        glCreateBuffers(1, &m_Buffers[i]);
        GLenum err = glGetError();
        if (err != 0)
        {
            std::cout << "Warning: Unable to create buffer #" << i << std::endl;
            m_NumBuffers = i;
            return false;
        }

        // in the case of using OGL to display the buffers read through
        // OCL, the OCL buffer needs to be persistent for the SSG read
        // to work properly, but the OGL buffer doesn't need to also be
        // persistent - if it is persistent, it takes from the available
        // space for the OCL buffers and it's not really required so make 
        // sure to set the persistent flag only if OGL is used directly
        GLbitfield  flags = GL_MAP_READ_BIT | GL_MAP_WRITE_BIT;
        if (!m_IsInteropUsed)
            flags |= GL_MAP_PERSISTENT_BIT;
        glNamedBufferStorage(m_Buffers[i], dataBufferSize, NULL, flags);
        err = glGetError();
        if (err != 0)
        {
            std::cout << "Warning: Unable to allocate buffer #" << i << std::endl;
            m_NumBuffers = i;
            return false;
        }

        glBindTexture(GL_TEXTURE_BUFFER, m_Textures[i]);
        glTexBuffer(GL_TEXTURE_BUFFER, GL_LUMINANCE, m_Buffers[i]);
        err = glGetError();
        if (err != 0)
        {
            std::cout << "Warning: Unable to glTexBuffer #" << i << std::endl;
            m_NumBuffers = i;
            return false;
        }
    }

    return true;
}

bool OGLApplication::Init(IApplication * pApplication, int argc, char ** argv)
{
    m_Application = pApplication;
    m_Settings = pApplication->GetSettings();
    if (!m_Settings)
    {
        std::cout << "Error: No settings available!" << std::endl;
        return false;
    }

    // default values
    m_Paused = false;

    m_NumBuffers = m_Settings->NumBuffers();
    m_readBuffer = 0;
    m_waitBuffer = 0;

    // reserve enough space in the buffers that depend
    // on how many of them we have
    m_Buffers.resize(m_NumBuffers, 0);
    m_SyncObject.resize(m_NumBuffers, NULL);
    m_Textures.resize(m_NumBuffers, 0);

    //////////////////////////
    // initialize GLUT
    glutInit(&argc, argv);

    //////////////////////////
    // initialize display window
    m_Width = m_Settings->Width();
    m_Height = m_Settings->Height();
    if (!InitWindow(0, 0))
    {
        std::cout << "OpenGL: Unable to intialize display window!" << std::endl;
        return false;
    }
        
    //////////////////////////
    // set GLUT callbacks
    glutDisplayFunc(Render);
    glutIdleFunc(Render);
    glutKeyboardFunc(ProcessKeys);
    glutReshapeFunc(Reshape);
    glutCloseFunc(CloseWindow);

    //////////////////////////
    // initialize GLEW
    GLenum glew_status = glewInit();
    if (glew_status != GLEW_OK)
    {
        std::cout << "OpenGL: Unable to initialize GLEW!" << std::endl;
        return false;
    }

    //////////////////////////
    // check for a number of functions we need
    if (!InitExtensions())
    {
        std::cout << "OpenGL: Unable to intialize extension!" << std::endl;
        return false;
    }

    //////////////////////////
    // check for SSG extension support
    if (!InitSsgExtensions())
    {
        std::cout << "OpenGL: Unable to intialize SSG extension!" << std::endl;
        return false;
    }

    //////////////////////////
    // initialize program
    if (!InitProgram())
    {
        std::cout << "OpenGL: Unable to create program!" << std::endl;
        return false;
    }

    //////////////////////////
    // prepare textures
    if (!InitTextures())
    {
        std::cout << "OpenGL: Unable to initialize textures!" << std::endl;
        return false;
    }

    //////////////////////////
    // allocate buffers
    const long long  buffSize = m_Settings->FrameSize();
    if (buffSize > INT_MAX)
    {
        std::cout << "OpenGL: Buffer too large for allocation!" << std::endl;
        return false;
    }

    m_DataSize = (int) buffSize;
    if (!InitBuffers(m_DataSize))
    {
        std::cout << "OpenGL: Unable to initialize all required buffers!" << std::endl;
        return false;
    }

    //////////////////////////
    // set VSync
#ifdef _WIN32
    if (!wglSwapIntervalEXT(m_Settings->VSync() ? 1 : 0))
    {
        std::cout << "OpenGL: Failed to set the correct VSync mode!" << std::endl;
        return false;
    }
#elif __linux__
    wglSwapIntervalEXT(m_Settings->VSync() ? 1 : 0);
#endif

    return true;
}

void OGLApplication::Term()
{
    // delete buffers
    if (!m_Buffers.empty())
        glDeleteBuffers((int) m_Buffers.size(), m_Buffers.data());
    m_Buffers.clear();

    // delete textures
    if (!m_Textures.empty())
        glDeleteTextures((int) m_Textures.size(), m_Textures.data());
    m_Textures.clear();

    // terminate program
    if (m_Program != 0)
    {
        glDeleteProgram(m_Program);
        m_Program = 0;
    }

    // clean-up window
    if (m_Window != 0)
    {
        glutDestroyWindow(m_Window);
        m_Window = 0;
    }
}


//
//
// Control methods
//
//

void OGLApplication::Run()
{
    // main loop
    glutMainLoop();
}

void OGLApplication::Pause(bool pause)
{
    m_Paused = pause;
    if (!m_Paused)
        glutPostRedisplay();
}

void OGLApplication::Stop()
{
    glutLeaveMainLoop();

    // glutMainLoop never returns so on exit it will 
    // close the app which means we have to clean up
    // now, before we're done the main loop
    P2PCloseFile();
    Term();

    // once this function finishes, glutMainLoop will
    // do an exit(0)...
}

void OGLApplication::SetFullScreen(bool fullScreen)
{
   glutFullScreenToggle();
}



//
//
// Handle a frame
//
//

void OGLApplication::BeginFrame(bool clear)
{
    if (m_Settings->Sync())
    {
        glFinish();
    }

    if (clear)
    {
        glClear(GL_COLOR_BUFFER_BIT);
    }
}


void OGLApplication::DrawFrame()
{
    // Currently we are reading N'th frame to the buffer, and drawing the (N-1)'s frame
    if (m_SyncObject[m_waitBuffer] != NULL)
        glWaitSync(m_SyncObject[m_waitBuffer], 0, GL_TIMEOUT_IGNORED);

    // copy buffer to texture
    if (glGetError() == GL_NO_ERROR)
    {
        glBindTexture(GL_TEXTURE_BUFFER, m_Textures[m_waitBuffer]);

        // draw quad
        glBegin(GL_QUADS);
        {
            glTexCoord2f(0.0, 1.0);
            glVertex2f(-1.0, -1.0);

            glTexCoord2f(0.0, 0.0);
            glVertex2f(-1.0, 1.0);

            glTexCoord2f(1.0, 0.0);
            glVertex2f(1.0, 1.0);

            glTexCoord2f(1.0, 1.0);
            glVertex2f(1.0, -1.0);
        }
        glEnd();
    }
    else
    {
        std::cout << "Error: the file read is not completed" << std::endl;
    }

    m_waitBuffer = (m_waitBuffer + 1) % m_NumBuffers;
}

void OGLApplication::DrawStatus(char * text)
{
    glRasterPos2f(-1.0f, 0.975f);
    glutBitmapString(GLUT_BITMAP_HELVETICA_18, (const unsigned char *)text);

    GLenum err = glGetError();
    if (err != GL_NO_ERROR)
    {
        std::cout << "Error: draw text failed" << std::endl;
    }
}

void OGLApplication::DrawSeekbar(int frameNumber, int totalNumberOfFrames)
{
    // TODO: implement
}

void OGLApplication::EndFrame()
{
    glutSwapBuffers();

    if (!m_Paused)
        glutPostRedisplay();
}




//
//
// P2P file access through OpenGL extension
//
//

bool OGLApplication::P2PIsExtensionActive()
{
    return (glCreateFileAMD != NULL) &&
           (glReleaseFileAMD != NULL) &&
           (glGetFileParameteri64vAMD != NULL) &&
           (glReadFileAMD != NULL) &&
           (glWriteFileAMD != NULL);
}

bool OGLApplication::P2POpenFile(const char * filename)
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return false;

    m_OglSsgFileHandle = glCreateFileAMD((GLchar*)filename, GL_READ_ONLY);
    if (!m_OglSsgFileHandle)
        return false;

    P2PGetFileSize();
    P2PGetSectorSize();

    return true;
}

unsigned long long OGLApplication::P2PGetFileSize()
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return 0;

    // if file hasn't been opened, we don't have a handle
    // to pass and be able to get that information
    if (!m_OglSsgFileHandle)
        return 0;

    GLuint64 fileSize = 0;
    glGetFileParameteri64vAMD(m_OglSsgFileHandle, GL_FILE_SIZE_AMD, &fileSize);

    m_SsgFileSize = fileSize;
    return m_SsgFileSize;
}

unsigned long long OGLApplication::P2PGetSectorSize()
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return 0;

    // if file hasn't been opened, we don't have a handle
    // to pass and be able to get that information
    if (!m_OglSsgFileHandle)
        return 0;

    GLuint64 sectorSize = 0;
    glGetFileParameteri64vAMD(m_OglSsgFileHandle, GL_FILE_BLOCK_SIZE_AMD, &sectorSize);
    m_sectorSize = (unsigned int) sectorSize;

    return m_sectorSize;
}

bool OGLApplication::P2PReadFile(unsigned long long fileOffset, unsigned long long size)
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return false;

    // Multiple-read test
    const unsigned int  nrOfRegions = m_Settings->NumThreads();
    if (nrOfRegions == 0)
    {
        glReadFileAMD(m_Buffers[m_readBuffer], m_OglSsgFileHandle, 0, fileOffset, size, NULL);
    }
    else
    {
        const unsigned long long  regionSize  = ((size + m_sectorSize - 1) / m_sectorSize / nrOfRegions) * m_sectorSize;

        for (unsigned int i = 0; i < nrOfRegions; i++)
        {
            GLuint64  srcOffset  = fileOffset + i * regionSize;
            GLuint64  destOffset = i * regionSize;
            GLuint64  length     = (i < nrOfRegions - 1) ? regionSize
                                                         : size - i * regionSize;
            if (length > 0)
                glReadFileAMD(m_Buffers[m_readBuffer], m_OglSsgFileHandle, destOffset, srcOffset, length, &m_SyncObject[m_readBuffer]);
        }
    }

    m_readBuffer = (m_readBuffer + 1) % m_NumBuffers;

    // update the first frame(s) flag - if we sent requests
    // for all frames, now we can start waiting on them to 
    // finish to start displaying them...
    //   m_NumBuffers == 1    -->  m_readBuffer = 0
    //   m_NumBuffers == 2    -->  m_readBuffer = 0
    //   m_NumBuffers == 3    -->  m_readBuffer = 2
    //   m_NumBuffers == 4    -->  m_readBuffer = 3
    // it seems that for double buffering, it's better
    // to submit read(0) and read(1) first and 
    // then go with wait(0), present(0), read(0), even though 
    // present(0), read(0) should cause a bit of serialization 
    // as we can't start reading into the buffer that's in use 
    // till present is done with it
    if (m_IsFirstFrame && 
         (  ((m_NumBuffers != 2) && (m_readBuffer == m_NumBuffers - 1)) ||
            ((m_NumBuffers == 2) && (m_readBuffer == 0))  )  )
        m_IsFirstFrame = false;

    return true;
}

void OGLApplication::P2PCloseFile()
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return;

    // if we got a file handle, means we opened the file 
    // successfuly, so close the file and clear the handle...
    if (m_OglSsgFileHandle)
    {
        glReleaseFileAMD(m_OglSsgFileHandle);
        m_OglSsgFileHandle = 0;
    }
}
