
#include "stdafx.h"

#include <GL/glew.h>
#include <GL/wglew.h>

#include "GLShader.h"
#include "GLDOPPEngine.h"

#define DOPP2

#define GL_WAIT_FOR_PREVIOUS_VSYNC                        0x931C

typedef GLuint (APIENTRY* PFNWGLGETDESKTOPTEXTUREAMD)(void);
typedef void   (APIENTRY* PFNWGLENABLEPOSTPROCESSAMD)(bool enable);
typedef GLuint (APIENTRY* WGLGENPRESENTTEXTUREAMD)(void);
typedef GLboolean (APIENTRY* WGLDESKTOPTARGETAMD) (GLuint);
typedef GLuint (APIENTRY* PFNWGLPRESENTTEXTURETOVIDEOAMD)(GLuint presentTexture, const GLuint* attrib_list);

PFNWGLGETDESKTOPTEXTUREAMD      wglGetDesktopTextureAMD;
PFNWGLENABLEPOSTPROCESSAMD      wglEnablePostProcessAMD;
PFNWGLPRESENTTEXTURETOVIDEOAMD  wglPresentTextureToVideoAMD;
WGLDESKTOPTARGETAMD             wglDesktopTargetAMD;
WGLGENPRESENTTEXTUREAMD         wglGenPresentTextureAMD;


#define GET_PROC(xx)                                    \
{                                                       \
	void **x = (void**)&xx;								\
    *x = (void *) wglGetProcAddress(#xx);               \
    if (*x == NULL) {                                   \
		return false;                                   \
    }                                                   \
}



GLDOPPEngine::GLDOPPEngine()
{
    m_uiDesktopWidth    = 0;
    m_uiDesktopHeight   = 0;
   
    m_uiDesktopTexture  = 0;
    m_uiPresentTexture  = 0;

    m_uiFBO = 0;
    m_uiBaseMap = 0;

    m_pShader = NULL;

    m_uiVertexBuffer = 0;
    m_uiVertexArray = 0;

    m_bStartPostProcessing = false;
    m_bDoPresent           = true;
}



GLDOPPEngine::~GLDOPPEngine()
{
    wglEnablePostProcessAMD(false);

    glFinish();

    if (m_pShader)
    {
        delete m_pShader;
    }

    glDeleteTextures(1, &m_uiDesktopTexture);

    glDeleteRenderbuffers(1, &m_uiDepthRB);
    glDeleteFramebuffers(1,  &m_uiFBO);

    glDeleteBuffers(1, &m_uiVertexBuffer);
    glDeleteVertexArrays(1, &m_uiVertexArray);
}



bool GLDOPPEngine::initDOPP(unsigned int uiDesktop, bool bPresent)
{
    if (!setupDOPPExtension())
    {
        return false;
    }

    // Select Desktop to be processed. ID is the same as seen in CCC
    if (!wglDesktopTargetAMD(uiDesktop))
    {
        return false;
    }
    
    glActiveTexture(GL_TEXTURE1);

    // Get Desktop Texture. Instead of creating a regular texture we request the desktop texture
    m_uiDesktopTexture = wglGetDesktopTextureAMD();
    glBindTexture(GL_TEXTURE_2D, m_uiDesktopTexture);

    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    // Get size of the desktop. Usually this is the same values as returned by GetSystemMetrics(SM_CXSCREEN)
    // and GetSystemMetrics(SM_CYSCREEN). In some cases it might differ, e.g. if a rotated desktop is used.
    glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_WIDTH,  (GLint*)&m_uiDesktopWidth);
    glGetTexLevelParameteriv(GL_TEXTURE_2D, 0, GL_TEXTURE_HEIGHT, (GLint*)&m_uiDesktopHeight);

    glBindTexture(GL_TEXTURE_2D, 0);

    glFlush();

    // Create FBO that is used to generate the present texture. We ill render into this FBO 
    // in order to create the present texture
    glGenFramebuffers(1,  &m_uiFBO);
    glGenRenderbuffers(1, &m_uiDepthRB);

    // generate present texture
    m_uiPresentTexture = wglGenPresentTextureAMD();

    glBindTexture(GL_TEXTURE_2D, m_uiPresentTexture);
    
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	    
    glBindTexture(GL_TEXTURE_2D, 0);
                
    glBindRenderbuffer(GL_RENDERBUFFER, m_uiDepthRB);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH24_STENCIL8, m_uiDesktopWidth, m_uiDesktopHeight);

    glBindFramebuffer(GL_FRAMEBUFFER, m_uiFBO);

    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, m_uiPresentTexture, 0);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, m_uiDepthRB);

    GLenum FBStatus = glCheckFramebufferStatus(GL_FRAMEBUFFER);
        
    glBindRenderbuffer(GL_RENDERBUFFER, 0);
    glBindFramebuffer(GL_FRAMEBUFFER, 0);

    if (FBStatus != GL_FRAMEBUFFER_COMPLETE)
    {  
        return false;
    }

    m_bStartPostProcessing = bPresent; 
    m_bDoPresent           = bPresent;

    return true;
}


bool GLDOPPEngine::initEffect()
{
    if (m_pShader)
    {
        delete m_pShader;
    }

    m_pShader = new GLShader;

    // Load basic shader 
    if (!m_pShader->createVertexShaderFromFile("base.vp"))
    {
        return false;
    }

    if (!m_pShader->createFragmentShaderFromFile("base.fp"))
    {
        return false;
    }

    if (!m_pShader->buildProgram())
    {
        return false;
    }

    m_pShader->bind();

    m_uiBaseMap   = glGetUniformLocation(m_pShader->getProgram(), "baseMap");

    createQuad();

    return true;
}




void GLDOPPEngine::processDesktop()
{
    int pVP[4];

    glGetIntegerv(GL_VIEWPORT, pVP);

    // Bind FBO that has the present texture attached
    glBindFramebuffer(GL_FRAMEBUFFER, m_uiFBO);

    // Set viewport for this FBO
    glViewport(0,0, m_uiDesktopWidth, m_uiDesktopHeight);

    // Render to FBO
    updateTexture();

    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glBindTexture(GL_TEXTURE_2D, 0);

    if (m_bDoPresent)
    {
        const GLuint attrib[] = { GL_WAIT_FOR_PREVIOUS_VSYNC, 0 };

        // Set the new desktop texture
        wglPresentTextureToVideoAMD(m_uiPresentTexture, attrib);

        if (m_bStartPostProcessing)
        {
            m_bStartPostProcessing = false;

            wglEnablePostProcessAMD(true);
        }

        glFinish();
    }

    glBindFramebuffer(GL_FRAMEBUFFER, 0);

    // restore original viewport
    glViewport(pVP[0], pVP[1], pVP[2], pVP[3]);
}



void GLDOPPEngine::updateTexture()
{
    glDisable(GL_DEPTH_TEST);

    m_pShader->bind();

    glActiveTexture(GL_TEXTURE1);
    glBindTexture(GL_TEXTURE_2D, m_uiDesktopTexture);

    glUniform1i(m_uiBaseMap, 1);

    glBindVertexArray(m_uiVertexArray);
    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
    glBindVertexArray(0);

    glEnable(GL_DEPTH_TEST);

    m_pShader->unbind();
}



void GLDOPPEngine::createQuad()
{
    const float vec[] = { -1.0f, 1.0f, 0.0f, 1.0f,   -1.0f, -1.0f, 0.0f, 1.0f,   1.0f, 1.0f, 0.0f, 1.0f,    1.0f, -1.0f, 0.0f, 1.0f };
    const float tex[] = {  0.0f,  1.0f,               0.0f,  0.0f,               1.0f, 1.0f,                1.0f, 0.0f };
    
    glGenVertexArrays(1, &m_uiVertexArray);
    glBindVertexArray(m_uiVertexArray);

    glGenBuffers(1, &m_uiVertexBuffer);
    glBindBuffer(GL_ARRAY_BUFFER, m_uiVertexBuffer);

    glBufferData(GL_ARRAY_BUFFER, 24*sizeof(float), vec, GL_STATIC_DRAW);
    glBufferSubData(GL_ARRAY_BUFFER, 16*sizeof(float), 8*sizeof(float), tex);

    glEnableVertexAttribArray(0);
    glEnableVertexAttribArray(4);

    glVertexAttribPointer((GLuint)0, 4, GL_FLOAT, GL_FALSE, 0, 0); 
    glVertexAttribPointer((GLuint)4, 2, GL_FLOAT, GL_FALSE, 0, (void*)(16*sizeof(float)));

    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindVertexArray(0);
}


bool GLDOPPEngine::setupDOPPExtension()
{
    GET_PROC(wglGetDesktopTextureAMD);
    GET_PROC(wglEnablePostProcessAMD);
    GET_PROC(wglPresentTextureToVideoAMD);
    GET_PROC(wglDesktopTargetAMD);
    GET_PROC(wglGenPresentTextureAMD);
    
    return true;
}