
// DOPPEngineDlg.cpp : implementation file
//

#include "stdafx.h"
#include <string>

#include "DOPPEngine.h"
#include "DOPPEngineDlg.h"
#include "afxdialogex.h"

#include <GL/glew.h>
#include <GL/wglew.h>

#include "DOPPRotationDlg.h"
#include "DisplayManager.h"
#include "GLDOPPEngine.h"
#include "GLDOPPColorInvert.h"
#include "GLDOPPDistort.h"
#include "GLDOPPEdgeFilter.h"
#include "GLDOPPRotate.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif


static unsigned int g_uiWindowWidth  = 800;
static unsigned int g_uiWindowHeight = 600;


using namespace std;

LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

void MyDebugFunc(GLuint id, GLenum category, GLenum severity, GLsizei length, const GLchar* message, GLvoid* userParam)
{
    MessageBoxA(NULL, message, "GL Debug Message", MB_ICONWARNING | MB_OK);
}


CDOPPEngineDlg::CDOPPEngineDlg(CWnd* pParent ) : CDialogEx(CDOPPEngineDlg::IDD, pParent)
{
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);

    m_dwThreadId     =  0;
    m_bEngineRunning = false;
    m_hEngineThread  = NULL;

    m_pEffectComboBox  = NULL;
    m_pShowWinCheckBox = NULL;

    m_uiEffectSelection  = 0;
    m_uiDesktopSelection = 0;

    m_pDisplayManager = NULL;

    m_pRotationDlg = NULL;
}


CDOPPEngineDlg::~CDOPPEngineDlg()
{
    if (m_pDisplayManager)
        delete m_pDisplayManager;
}

void CDOPPEngineDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialogEx::DoDataExchange(pDX);
}

BEGIN_MESSAGE_MAP(CDOPPEngineDlg, CDialogEx)
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
    ON_COMMAND(IDC_START,  &CDOPPEngineDlg::OnStart)
    ON_COMMAND(IDC_STOP,   &CDOPPEngineDlg::OnStop)
    ON_COMMAND(IDC_EXIT,   &CDOPPEngineDlg::OnExit)
END_MESSAGE_MAP()


// CDOPPEngineDlg message handlers

BOOL CDOPPEngineDlg::OnInitDialog()
{
	CDialogEx::OnInitDialog();

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon

    m_pEffectComboBox = static_cast<CComboBox*>(GetDlgItem(IDC_COMBO_EFFECT));

    m_pEffectComboBox->Clear();
    m_pEffectComboBox->InsertString(0, _T("No Effect"));
    m_pEffectComboBox->InsertString(1, _T("Invert Colors"));
    m_pEffectComboBox->InsertString(2, _T("Edge Filter"));
    m_pEffectComboBox->InsertString(3, _T("Distort"));
    m_pEffectComboBox->InsertString(4, _T("Rotate"));

    m_pEffectComboBox->SetCurSel(0);
    m_uiEffectSelection = 0;

    m_pDisplayManager = new DisplayManager;

    m_pShowWinCheckBox = static_cast<CButton*>(GetDlgItem(IDC_CHECK_WINDOW));

    m_pShowWinCheckBox->SetCheck(BST_UNCHECKED);

    m_pDesktopComboBox = static_cast<CComboBox*>(GetDlgItem(IDC_COMBO_DESKTOP));

    m_pDesktopComboBox->Clear();

    unsigned int uiNumDesktops = m_pDisplayManager->enumDisplays();

    for (unsigned int i = 0; i < uiNumDesktops; ++i)
    {
        char buf[8];

        // Add 1 to the id to have the matching Desktop Id with CCC
        sprintf_s(buf," %d", (i + 1));

        m_pDesktopComboBox->InsertString(i, CA2CT(buf));
    }
    
    m_pDesktopComboBox->SetCurSel(0);

    m_uiDesktopSelection = 0;

    m_pRotationDlg = new DOPPRotationDlg;

	return TRUE;  // return TRUE  unless you set the focus to a control
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CDOPPEngineDlg::OnPaint()
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialogEx::OnPaint();
	}
}



// The system calls this function to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CDOPPEngineDlg::OnQueryDragIcon()
{
	return static_cast<HCURSOR>(m_hIcon);
}


void CDOPPEngineDlg::OnStart()
{
    if (!m_bEngineRunning)
    {
        m_uiEffectSelection  = m_pEffectComboBox->GetCurSel();
        m_uiDesktopSelection = m_pDesktopComboBox->GetCurSel();

        if (m_uiEffectSelection == ROTATE_DESKTOP && m_pRotationDlg)
        {
            if (m_pRotationDlg->DoModal() == IDOK)
            {
                m_uiRotationAngle = m_pRotationDlg->getAngle();
            }
            else
            {
                return;
            }
        }

        m_hEngineThread = CreateThread(NULL, 0, reinterpret_cast<LPTHREAD_START_ROUTINE>(&CDOPPEngineDlg::threadFunction), this, 0, &m_dwThreadId);

        m_bEngineRunning = true;
    }
}


void CDOPPEngineDlg::OnStop()
{
    if (m_bEngineRunning)
    {
        m_bEngineRunning = false;

        WaitForSingleObject(m_hEngineThread, INFINITE);
    }
}



void CDOPPEngineDlg::OnExit()
{
    // stop thread if required
    OnStop();

    OnOK();
}


DWORD CDOPPEngineDlg::threadFunction(void* pData)
{
    if (pData)
    {
        CDOPPEngineDlg* pInstance = reinterpret_cast<CDOPPEngineDlg*>(pData);

        pInstance->threadLoop();
    }

    return 0;
}


DWORD CDOPPEngineDlg::threadLoop()
{
    bool   bCapture = false;

    bCapture = (m_pShowWinCheckBox->GetCheck() == BST_CHECKED);

    // Create window with ogl context to load extensions
    if (!createGLWindow())
    {
        return 0;
    }

    GLDOPPEngine* pEngine = NULL;

    switch (m_uiEffectSelection)
    {
        case COLOR_INVERT:
            pEngine = new GLDOPPColorInvert;
            break;

        case EDGE_FILTER:
            pEngine = new GLDOPPEdgeFilter;
            break;

        case DISTORT_EFFECT:
            pEngine = new GLDOPPDistort;
            break;

        case ROTATE_DESKTOP:
            pEngine = new GLDOPPRotate;
            break;

        default:
            pEngine = new GLDOPPEngine;
            break;
    }

    // m_uiDesktopSelection is the id of the selected element in the Combo Box
    // Need to add 1 to get the dektop id as shown in CCC
    if (!pEngine->initDOPP(m_uiDesktopSelection + 1))
    {
        m_bEngineRunning = false;
    }

    if (m_bEngineRunning && !pEngine->initEffect())
    {
        m_bEngineRunning = false;
    }

    if (m_bEngineRunning && bCapture)
    {
        // Create off screen context
        showWindow();
    }

    if (m_bEngineRunning && m_uiEffectSelection == ROTATE_DESKTOP)
    {
        GLDOPPRotate* pRot = dynamic_cast<GLDOPPRotate*>(pEngine);

        pRot->setRotation((float)m_uiRotationAngle);
    }

    MSG mMsg;

    while(m_bEngineRunning)
    {
        pEngine->processDesktop();
        
        if (bCapture)
        {
            // Blit FBO of DOPPEngine into the window
            glViewport(0, 0, g_uiWindowWidth, g_uiWindowHeight);

            unsigned int uiFBO = pEngine->getPresentBuffer();

            glBindFramebuffer(GL_DRAW_FRAMEBUFFER, 0);
            glBindFramebuffer(GL_READ_FRAMEBUFFER, uiFBO);

            glBlitFramebuffer(0, pEngine->getDesktopHeight(), pEngine->getDesktopWidth(), 0, 0, 0, g_uiWindowWidth, g_uiWindowHeight, GL_COLOR_BUFFER_BIT, GL_LINEAR);

            glBindFramebuffer(GL_FRAMEBUFFER, 0);

            SwapBuffers(m_hDC);

            if (PeekMessage(&mMsg, NULL, 0, 0, PM_REMOVE))
		    {
			    if (mMsg.message == WM_QUIT)
			    {
				    m_bEngineRunning = false;
			    }
                else
			    {
				    TranslateMessage(&mMsg);
				    DispatchMessage(&mMsg);
			    }
		    }
        }
    }

    delete pEngine;

    deleteWindow();

    return 0;
}



bool CDOPPEngineDlg::createGLWindow()
{
    int			mPixelFormat;

    WNDCLASSEX		wndclass;
	const LPCWSTR   cClassName   = _T("OGL");
	const LPCWSTR	cWindowName  = _T("DOPP Capture");

	// Register WindowClass
	wndclass.cbSize         = sizeof(WNDCLASSEX);
	wndclass.style          = CS_OWNDC;
	wndclass.lpfnWndProc    = WndProc;
	wndclass.cbClsExtra     = 0;
	wndclass.cbWndExtra     = 0;
	wndclass.hInstance      = (HINSTANCE)GetModuleHandle(NULL);
	wndclass.hIcon		    = (HICON)LoadImage((HINSTANCE)AfxGetInstanceHandle(),  MAKEINTRESOURCE(IDR_MAINFRAME), IMAGE_ICON, LR_DEFAULTSIZE, LR_DEFAULTSIZE, NULL); 
	wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
	wndclass.hbrBackground  = NULL;
	wndclass.lpszMenuName   = NULL;
	wndclass.lpszClassName  = cClassName;
	wndclass.hIconSm		= (HICON)LoadImage((HINSTANCE)AfxGetInstanceHandle(),  MAKEINTRESOURCE(IDR_MAINFRAME), IMAGE_ICON, LR_DEFAULTSIZE, LR_DEFAULTSIZE, NULL); 


	if (!RegisterClassEx(&wndclass))
		return FALSE;
        
    DWORD dwExStyle = WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;	
    DWORD dwStyle   = WS_OVERLAPPEDWINDOW;  

	m_hWnd = CreateWindow(  cClassName,
						    cWindowName,
						    dwStyle,
						    0,
						    0,
						    g_uiWindowWidth,
						    g_uiWindowHeight,
						    NULL,
						    NULL,
						    (HINSTANCE)AfxGetInstanceHandle(),
						    NULL);

    
    if (!m_hWnd)
		return FALSE;


	static PIXELFORMATDESCRIPTOR pfd;

    memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize           = sizeof(PIXELFORMATDESCRIPTOR); 
	pfd.nVersion        = 1; 
	pfd.dwFlags         = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL  | PFD_DOUBLEBUFFER ;
	pfd.iPixelType      = PFD_TYPE_RGBA; 
	pfd.cColorBits      = 24; 
	pfd.cRedBits        = 8; 
	pfd.cGreenBits      = 8; 
	pfd.cBlueBits       = 8; 
	pfd.cAlphaBits      = 8;
	pfd.cDepthBits      = 24; 
	pfd.cStencilBits    = 8; 
	pfd.iLayerType      = PFD_MAIN_PLANE;

	m_hDC = ::GetDC(m_hWnd);

	if (!m_hDC)
		return FALSE;

	mPixelFormat = ChoosePixelFormat(m_hDC, &pfd);

    if (!mPixelFormat)
		return FALSE;

	SetPixelFormat(m_hDC, mPixelFormat, &pfd);

    m_hCtx = wglCreateContext(m_hDC);

	wglMakeCurrent(m_hDC, m_hCtx);

    if (glewInit() != GLEW_OK)
    {
        return false;
    }

    if (WGLEW_ARB_create_context)
	{
        wglMakeCurrent(NULL, NULL);
		wglDeleteContext(m_hCtx);

		int attribs[] = {
				 WGL_CONTEXT_MAJOR_VERSION_ARB, 4,
				 WGL_CONTEXT_MINOR_VERSION_ARB, 2,
                 WGL_CONTEXT_PROFILE_MASK_ARB , WGL_CONTEXT_CORE_PROFILE_BIT_ARB, 
#ifdef _DEBUG
				 WGL_CONTEXT_FLAGS_ARB, WGL_CONTEXT_DEBUG_BIT_ARB,
#endif
				 0}; 

		m_hCtx = wglCreateContextAttribsARB(m_hDC, 0, attribs);

		if (m_hCtx)
			wglMakeCurrent(m_hDC, m_hCtx);
		else
			return false;
	}

	if (!wglMakeCurrent(m_hDC, m_hCtx))
        return false;

    if (GLEW_AMD_debug_output)
		glDebugMessageCallbackAMD((GLDEBUGPROCAMD)&MyDebugFunc, NULL);

    return true;
}


void CDOPPEngineDlg::showWindow()
{
    ::ShowWindow(m_hWnd, SW_SHOW);

	::UpdateWindow(m_hWnd);
}


void CDOPPEngineDlg::deleteWindow()
{
    wglMakeCurrent(m_hDC, NULL);

    wglDeleteContext(m_hCtx);

    ::ReleaseDC(m_hWnd, m_hDC);

	::DestroyWindow(m_hWnd);

    ::UnregisterClass(_T("OGL"), NULL);
}



LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
    case WM_CHAR:
        {
            char c = (char)wParam;

			switch (c)
			{
				case VK_ESCAPE:
					PostQuitMessage(0);
					break;
			}

        }
        return 0;

	case WM_CREATE:
		return 0;

	case WM_SIZE:
		 g_uiWindowWidth  = LOWORD(lParam);
         g_uiWindowHeight = HIWORD(lParam);

         return 0;

	case WM_DESTROY:
		PostQuitMessage(0);
		return 0;

	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}

