//==============================================================================
// Copyright (c) 2015 Advanced Micro Devices, Inc. All rights reserved.
/// \author AMD Developer Tools Team
/// \file
/// \brief  This file contains functions to format HSA Trace string output
//==============================================================================

#include "HSAFunctionDefs.h"
#include "HSATraceStringUtils.h"

#include "AutoGenerated/HSARTModuleFuncTables.h"
#include "AutoGenerated/HSATraceStringOutput.h"

bool (*HSATraceStringUtils::pGetApiDisplayName)(HSA_API_Type type, std::string& apiDisplayName) = nullptr;

std::string HSATraceStringUtils::GetHSAAPINameString(HSA_API_Type type)
{
    if (nullptr != pGetApiDisplayName)
    {
        std::string translatedApiName = "";

        if (pGetApiDisplayName(type, translatedApiName))
        {
            return translatedApiName;
        }
    }

    switch (type)
    {
#define X(SYM) case HSA_API_Type_hsa_##SYM: return std::string("hsa_" #SYM);
            HSA_RUNTIME_API_TABLE
            HSA_EXT_FINALIZE_API_TABLE
            HSA_EXT_IMAGE_API_TABLE
            HSA_EXT_AMD_API_TABLE
            HSA_VEN_AMD_LOADER_API_TABLE
            HSA_VEN_AMD_AQL_PROFILE_API_TABLE

        default: return "";
#undef X
    }
}

std::string HSATraceStringUtils::GetStringString(const char* src, bool truncate, bool surroundWithDeref)
{
    if (NULL == src)
    {
        return "NULL";
    }

    std::string str(src);

    return GetStringString(src, str, truncate, surroundWithDeref);
}

std::string HSATraceStringUtils::GetStringString(const char* src, std::string& srcString, bool truncate, bool surroundWithDeref)
{
    if (NULL == src)
    {
        return "NULL";
    }

    std::string str = srcString;

    if (truncate && str.length() > 60)
    {
        str = str.substr(0, 60).append("...");
    }

    size_t found = str.find_first_of("\n\r\t\"");

    while (std::string::npos != found)
    {
        char subst[] = { '\\', '\0', '\0' };

        switch (str[found])
        {
            case '\n': subst[1] = 'n'; break;

            case '\r': subst[1] = 'r'; break;

            case '\t': subst[1] = 't'; break;

            case '\"': subst[1] = '\"'; break;

            default: ++found; continue;
        }

        str.replace(found, 1, subst);
        found += 2;
        found = str.find_first_of("\n\r\t\"", found);
    }

    str = SurroundWith(str, "\"", "\"");

    if (surroundWithDeref)
    {
        str = SurroundWithDeRef(str);
    }

    return str;
}

std::string HSATraceStringUtils::Get_bool_String(bool input)
{
    if (input)
    {
        return "true";
    }
    else
    {
        return "false";
    }
}

std::string HSATraceStringUtils::Get_bool_Ptr_String(bool* pInput, bool input)
{
    if (nullptr == pInput)
    {
        return "NULL";
    }

    return SurroundWithDeRef(Get_bool_String(input));
}

std::string HSATraceStringUtils::Get_uint8_t_String(uint8_t input)
{
    return StringUtils::ToString(static_cast<unsigned int>(input));
}

unsigned int HSATraceStringUtils::Get_hsa_agent_get_info_AttributeSize(hsa_agent_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // char[64]
        case HSA_AGENT_INFO_NAME:
        case HSA_AGENT_INFO_VENDOR_NAME:
            return sizeof(char[64]);

        // hsa_agent_feature_t
        case HSA_AGENT_INFO_FEATURE:
            return sizeof(hsa_agent_feature_t);

        // hsa_machine_model_t
        case HSA_AGENT_INFO_MACHINE_MODEL:
            return sizeof(hsa_machine_model_t);

        // hsa_profile_t
        case HSA_AGENT_INFO_PROFILE:
            return sizeof(hsa_profile_t);

        // hsa_default_float_rounding_mode_t
        case HSA_AGENT_INFO_DEFAULT_FLOAT_ROUNDING_MODE:
            return sizeof(hsa_default_float_rounding_mode_t);

        //case HSA_AGENT_INFO_BASE_PROFILE_DEFAULT_FLOAT_ROUNDING_MODES:
        //case HSA_AGENT_INFO_FAST_F16_OPERATION:

        // uint32_t
        case HSA_AGENT_INFO_WAVEFRONT_SIZE:
        case HSA_AGENT_INFO_WORKGROUP_MAX_SIZE:
        case HSA_AGENT_INFO_GRID_MAX_SIZE:
        case HSA_AGENT_INFO_FBARRIER_MAX_SIZE:
        case HSA_AGENT_INFO_QUEUES_MAX:
        case HSA_AGENT_INFO_QUEUE_MIN_SIZE:
        case HSA_AGENT_INFO_QUEUE_MAX_SIZE:
        case HSA_AGENT_INFO_NODE:
            return sizeof(uint32_t);

        // uint16_t[3]
        case HSA_AGENT_INFO_WORKGROUP_MAX_DIM:
            return sizeof(uint16_t[3]);

        // hsa_dim3_t
        case HSA_AGENT_INFO_GRID_MAX_DIM:
            return sizeof(hsa_dim3_t);

        // hsa_queue_type_t
        case HSA_AGENT_INFO_QUEUE_TYPE:
            return sizeof(hsa_queue_type_t);

        // hsa_device_type_t
        case HSA_AGENT_INFO_DEVICE:
            return sizeof(hsa_device_type_t);

        // uint32_t[4]
        case HSA_AGENT_INFO_CACHE_SIZE:
            return sizeof(uint32_t[4]);

        // hsa_isa_t
        case HSA_AGENT_INFO_ISA:
            return sizeof(hsa_isa_t);

        // uint8_t[128]
        case HSA_AGENT_INFO_EXTENSIONS:
            return sizeof(uint8_t[128]);

        // uint16_t
        case HSA_AGENT_INFO_VERSION_MAJOR:
        case HSA_AGENT_INFO_VERSION_MINOR:
            return sizeof(uint16_t);

        default:
            return Get_hsa_amd_agent_get_info_AttributeSize(static_cast<hsa_amd_agent_info_t>(attribute));
    }
}

std::string HSATraceStringUtils::Get_hsa_agent_get_info_AttributeString(void* value, hsa_agent_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {
                // char*
                case HSA_AGENT_INFO_NAME:
                case HSA_AGENT_INFO_VENDOR_NAME:
                    ss << GetStringString(static_cast<char*>(value), false, false);
                    break;

                // hsa_agent_feature_t
                case HSA_AGENT_INFO_FEATURE:
                    ss << Get_hsa_agent_feature_t_String(*(static_cast<hsa_agent_feature_t*>(value)));
                    break;

                // hsa_machine_model_t
                case HSA_AGENT_INFO_MACHINE_MODEL:
                    ss << Get_hsa_machine_model_t_String(*(static_cast<hsa_machine_model_t*>(value)));
                    break;

                // hsa_profile_t
                case HSA_AGENT_INFO_PROFILE:
                    ss << Get_hsa_profile_t_String(*(static_cast<hsa_profile_t*>(value)));
                    break;

                // hsa_default_float_rounding_mode_t
                case HSA_AGENT_INFO_DEFAULT_FLOAT_ROUNDING_MODE:
                    ss << Get_hsa_default_float_rounding_mode_t_String(*(static_cast<hsa_default_float_rounding_mode_t*>(value)));
                    break;

                //case HSA_AGENT_INFO_BASE_PROFILE_DEFAULT_FLOAT_ROUNDING_MODES:
                //case HSA_AGENT_INFO_FAST_F16_OPERATION:

                // uint32_t
                case HSA_AGENT_INFO_WAVEFRONT_SIZE:
                case HSA_AGENT_INFO_WORKGROUP_MAX_SIZE:
                case HSA_AGENT_INFO_GRID_MAX_SIZE:
                case HSA_AGENT_INFO_FBARRIER_MAX_SIZE:
                case HSA_AGENT_INFO_QUEUES_MAX:
                case HSA_AGENT_INFO_QUEUE_MIN_SIZE:
                case HSA_AGENT_INFO_QUEUE_MAX_SIZE:
                case HSA_AGENT_INFO_NODE:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // hsa_dim3_t
                case HSA_AGENT_INFO_GRID_MAX_DIM:
                    ss << Get_hsa_dim3_t_String(*(static_cast<hsa_dim3_t*>(value)));
                    break;

                // hsa_queue_type_t
                case HSA_AGENT_INFO_QUEUE_TYPE:
                    ss << Get_hsa_queue_type_t_String(*(static_cast<hsa_queue_type_t*>(value)));
                    break;

                // hsa_device_type_t
                case HSA_AGENT_INFO_DEVICE:
                    ss << Get_hsa_device_type_t_String(*(static_cast<hsa_device_type_t*>(value)));
                    break;

                // uint16_t
                case HSA_AGENT_INFO_VERSION_MAJOR:
                case HSA_AGENT_INFO_VERSION_MINOR:
                    ss << (*(static_cast<uint16_t*>(value)));
                    break;

                // uint16_t[3]
                case HSA_AGENT_INFO_WORKGROUP_MAX_DIM:
                {
                    std::ostringstream tempss;

                    for (int i = 0; i < 3; i++)
                    {
                        tempss << (static_cast<uint16_t*>(value))[i];

                        if (2 != i)
                        {
                            tempss << ",";
                        }
                    }

                    ss << SurroundWithList(tempss.str());
                    break;
                }

                // uint32_t[4]
                case HSA_AGENT_INFO_CACHE_SIZE:
                {
                    std::ostringstream tempss;

                    for (int i = 0; i < 4; i++)
                    {
                        tempss << (static_cast<uint32_t*>(value))[i];

                        if (3 != i)
                        {
                            tempss << ",";
                        }
                    }

                    ss << SurroundWithList(tempss.str());
                    break;
                }

                // hsa_isa_t
                case HSA_AGENT_INFO_ISA:
                    ss << Get_hsa_isa_t_String(*(static_cast<hsa_isa_t*>(value)));
                    break;

                // uint8_t[128] -- not yet implemented
                case HSA_AGENT_INFO_EXTENSIONS:

                default:
                    ss << Get_hsa_amd_agent_get_info_AttributeString(value, static_cast<hsa_amd_agent_info_t>(attribute), retVal);
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_amd_agent_get_info_AttributeSize(hsa_amd_agent_info_t attribute)
{
    switch (attribute)
    {
        // char[64]
        case HSA_AMD_AGENT_INFO_PRODUCT_NAME:
            return sizeof(char[64]);

        // uint32_t
        case HSA_AMD_AGENT_INFO_CHIP_ID:
        case HSA_AMD_AGENT_INFO_CACHELINE_SIZE:
        case HSA_AMD_AGENT_INFO_COMPUTE_UNIT_COUNT:
        case HSA_AMD_AGENT_INFO_MAX_CLOCK_FREQUENCY:
        case HSA_AMD_AGENT_INFO_DRIVER_NODE_ID:
        case HSA_AMD_AGENT_INFO_MAX_ADDRESS_WATCH_POINTS:
        case HSA_AMD_AGENT_INFO_BDFID:
        case HSA_AMD_AGENT_INFO_MEMORY_WIDTH:
        case HSA_AMD_AGENT_INFO_MEMORY_MAX_FREQUENCY:
        case HSA_AMD_AGENT_INFO_MAX_WAVES_PER_CU:
        case HSA_AMD_AGENT_INFO_NUM_SIMDS_PER_CU:
        case HSA_AMD_AGENT_INFO_NUM_SHADER_ENGINES:
        case HSA_AMD_AGENT_INFO_NUM_SHADER_ARRAYS_PER_SE:
            return sizeof(uint32_t);

        default:
            return sizeof(int);
    }
}
std::string HSATraceStringUtils::Get_hsa_amd_agent_get_info_AttributeString(void* value, hsa_amd_agent_info_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {
                // char*
                case HSA_AMD_AGENT_INFO_PRODUCT_NAME:
                    ss << GetStringString(static_cast<char*>(value), false, false);
                    break;

                // uint32_t
                case HSA_AMD_AGENT_INFO_CHIP_ID:
                case HSA_AMD_AGENT_INFO_CACHELINE_SIZE:
                case HSA_AMD_AGENT_INFO_COMPUTE_UNIT_COUNT:
                case HSA_AMD_AGENT_INFO_MAX_CLOCK_FREQUENCY:
                case HSA_AMD_AGENT_INFO_DRIVER_NODE_ID:
                case HSA_AMD_AGENT_INFO_MAX_ADDRESS_WATCH_POINTS:
                case HSA_AMD_AGENT_INFO_BDFID:
                case HSA_AMD_AGENT_INFO_MEMORY_WIDTH:
                case HSA_AMD_AGENT_INFO_MEMORY_MAX_FREQUENCY:
                case HSA_AMD_AGENT_INFO_MAX_WAVES_PER_CU:
                case HSA_AMD_AGENT_INFO_NUM_SIMDS_PER_CU:
                case HSA_AMD_AGENT_INFO_NUM_SHADER_ENGINES:
                case HSA_AMD_AGENT_INFO_NUM_SHADER_ARRAYS_PER_SE:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return ss.str();
    }
}


unsigned int HSATraceStringUtils::Get_hsa_system_get_info_AttributeSize(hsa_system_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // uint16_t
        case HSA_SYSTEM_INFO_VERSION_MAJOR:
        case HSA_SYSTEM_INFO_VERSION_MINOR:
            return sizeof(uint16_t);

        // uint64_t
        case HSA_SYSTEM_INFO_TIMESTAMP:
        case HSA_SYSTEM_INFO_TIMESTAMP_FREQUENCY:
        case HSA_SYSTEM_INFO_SIGNAL_MAX_WAIT:
            return sizeof(uint64_t);

        // hsa_endianness_t
        case HSA_SYSTEM_INFO_ENDIANNESS:
            return sizeof(hsa_endianness_t);

        // hsa_machine_model_t
        case HSA_SYSTEM_INFO_MACHINE_MODEL:
            return sizeof(hsa_machine_model_t);

        // uint8_t[128]
        case HSA_SYSTEM_INFO_EXTENSIONS:
            return sizeof(uint8_t[128]);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_system_get_info_AttributeString(void* value, hsa_system_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {
                // uint16_t
                case HSA_SYSTEM_INFO_VERSION_MAJOR:
                case HSA_SYSTEM_INFO_VERSION_MINOR:
                    ss << (*(static_cast<uint16_t*>(value)));
                    break;

                // uint64_t
                case HSA_SYSTEM_INFO_TIMESTAMP:
                case HSA_SYSTEM_INFO_TIMESTAMP_FREQUENCY:
                case HSA_SYSTEM_INFO_SIGNAL_MAX_WAIT:
                    ss << (*(static_cast<uint64_t*>(value)));
                    break;

                // hsa_endianness_t
                case HSA_SYSTEM_INFO_ENDIANNESS:
                    ss << Get_hsa_endianness_t_String(*(static_cast<hsa_endianness_t*>(value)));
                    break;

                // hsa_machine_model_t
                case HSA_SYSTEM_INFO_MACHINE_MODEL:
                    ss << Get_hsa_machine_model_t_String(*(static_cast<hsa_machine_model_t*>(value)));
                    break;

                // uint8_t[128]
                case HSA_SYSTEM_INFO_EXTENSIONS:

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_region_get_info_AttributeSize(hsa_region_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // hsa_region_segment_t
        case HSA_REGION_INFO_SEGMENT:
            return sizeof(hsa_region_segment_t);

        // uint32_t
        case HSA_REGION_INFO_GLOBAL_FLAGS: // this is actually a bitfield of hsa_region_global_flag_t
            return sizeof(uint32_t);

        // size_t
        case HSA_REGION_INFO_SIZE:
        case HSA_REGION_INFO_ALLOC_MAX_SIZE:
        case HSA_REGION_INFO_RUNTIME_ALLOC_GRANULE:
        case HSA_REGION_INFO_RUNTIME_ALLOC_ALIGNMENT:
            return sizeof(size_t);

        // bool
        case HSA_REGION_INFO_RUNTIME_ALLOC_ALLOWED:
            return sizeof(bool);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_region_get_info_AttributeString(void* value, hsa_region_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // hsa_region_segment_t
                case HSA_REGION_INFO_SEGMENT:
                    ss << Get_hsa_region_segment_t_String(*(static_cast<hsa_region_segment_t*>(value)));
                    break;

                // uint32_t
                case HSA_REGION_INFO_GLOBAL_FLAGS: // this is actually a bitfield of hsa_region_global_flag_t
                    ss << Get_hsa_region_global_flag_t_String(*(static_cast<hsa_region_global_flag_t*>(value)));
                    break;

                // size_t
                case HSA_REGION_INFO_SIZE:
                case HSA_REGION_INFO_ALLOC_MAX_SIZE:
                case HSA_REGION_INFO_RUNTIME_ALLOC_GRANULE:
                case HSA_REGION_INFO_RUNTIME_ALLOC_ALIGNMENT:
                    ss << (*(static_cast<size_t*>(value)));
                    break;

                // bool
                case HSA_REGION_INFO_RUNTIME_ALLOC_ALLOWED:
                    ss << (*(static_cast<bool*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_isa_get_info_AttributeSize(hsa_isa_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // uint32_t
        case HSA_ISA_INFO_NAME_LENGTH:
        case HSA_ISA_INFO_CALL_CONVENTION_COUNT:
        case HSA_ISA_INFO_CALL_CONVENTION_INFO_WAVEFRONT_SIZE:
        case HSA_ISA_INFO_CALL_CONVENTION_INFO_WAVEFRONTS_PER_COMPUTE_UNIT:
            return sizeof(uint32_t);

        // char*
        case HSA_ISA_INFO_NAME: //TODO: Fix this to make caller read the length from HSA_ISA_INFO_NAME_LENGTH
            return sizeof(char*);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_isa_get_info_AttributeString(void* value, hsa_isa_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // uint32_t
                case HSA_ISA_INFO_NAME_LENGTH:
                case HSA_ISA_INFO_CALL_CONVENTION_COUNT:
                case HSA_ISA_INFO_CALL_CONVENTION_INFO_WAVEFRONT_SIZE:
                case HSA_ISA_INFO_CALL_CONVENTION_INFO_WAVEFRONTS_PER_COMPUTE_UNIT:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // char*
                case HSA_ISA_INFO_NAME:
                    ss << GetStringString(static_cast<char*>(value), false, false);
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_code_object_get_info_AttributeSize(hsa_code_object_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // char[64]
        case HSA_CODE_OBJECT_INFO_VERSION:
            return sizeof(char[64]);

        // hsa_code_object_type_t
        case HSA_CODE_OBJECT_INFO_TYPE:
            return sizeof(hsa_code_object_type_t);

        // hsa_isa_t
        case HSA_CODE_OBJECT_INFO_ISA:
            return sizeof(hsa_isa_t);

        // hsa_machine_model_t
        case HSA_CODE_OBJECT_INFO_MACHINE_MODEL:
            return sizeof(hsa_machine_model_t);

        // hsa_profile_t
        case HSA_CODE_OBJECT_INFO_PROFILE:
            return sizeof(hsa_profile_t);

        // hsa_default_float_rounding_mode_t
        case HSA_CODE_OBJECT_INFO_DEFAULT_FLOAT_ROUNDING_MODE:
            return sizeof(hsa_default_float_rounding_mode_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_code_object_get_info_AttributeString(void* value, hsa_code_object_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // char[64]
                case HSA_CODE_OBJECT_INFO_VERSION:
                    ss << GetStringString(((char*)value), false, false);
                    break;

                // hsa_code_object_type_t
                case HSA_CODE_OBJECT_INFO_TYPE:
                    ss << HSATraceStringUtils::Get_hsa_code_object_type_t_String(*(static_cast<hsa_code_object_type_t*>(value)));
                    break;

                // hsa_isa_t
                case HSA_CODE_OBJECT_INFO_ISA:
                    ss << HSATraceStringUtils::Get_hsa_isa_t_String(*(static_cast<hsa_isa_t*>(value)));
                    break;

                // hsa_machine_model_t
                case HSA_CODE_OBJECT_INFO_MACHINE_MODEL:
                    ss << HSATraceStringUtils::Get_hsa_machine_model_t_String(*(static_cast<hsa_machine_model_t*>(value)));
                    break;

                // hsa_profile_t
                case HSA_CODE_OBJECT_INFO_PROFILE:
                    ss << HSATraceStringUtils::Get_hsa_profile_t_String(*(static_cast<hsa_profile_t*>(value)));
                    break;

                // hsa_default_float_rounding_mode_t
                case HSA_CODE_OBJECT_INFO_DEFAULT_FLOAT_ROUNDING_MODE:
                    ss << HSATraceStringUtils::Get_hsa_default_float_rounding_mode_t_String(*(static_cast<hsa_default_float_rounding_mode_t*>(value)));
                    break;


                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_code_symbol_get_info_AttributeSize(hsa_code_symbol_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // hsa_symbol_kind_t
        case HSA_CODE_SYMBOL_INFO_TYPE:
            return sizeof(hsa_symbol_kind_t);

        // uint32_t
        case HSA_CODE_SYMBOL_INFO_NAME_LENGTH:
        case HSA_CODE_SYMBOL_INFO_MODULE_NAME_LENGTH:
        case HSA_CODE_SYMBOL_INFO_VARIABLE_ALIGNMENT:
        case HSA_CODE_SYMBOL_INFO_VARIABLE_SIZE:
        case HSA_CODE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_SIZE:
        case HSA_CODE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_ALIGNMENT:
        case HSA_CODE_SYMBOL_INFO_KERNEL_GROUP_SEGMENT_SIZE:
        case HSA_CODE_SYMBOL_INFO_KERNEL_PRIVATE_SEGMENT_SIZE:
        case HSA_CODE_SYMBOL_INFO_INDIRECT_FUNCTION_CALL_CONVENTION:
            return sizeof(uint32_t);

        // char*
        case HSA_CODE_SYMBOL_INFO_NAME: //TODO: Fix this to make caller read the length from HSA_ISA_INFO_NAME_LENGTH
        case HSA_CODE_SYMBOL_INFO_MODULE_NAME:
            return sizeof(char*);

        // hsa_symbol_linkage_t
        case HSA_CODE_SYMBOL_INFO_LINKAGE:
            return sizeof(hsa_symbol_linkage_t);

        // hsa_variable_allocation_t
        case HSA_CODE_SYMBOL_INFO_VARIABLE_ALLOCATION:
            return sizeof(hsa_variable_allocation_t);

        // hsa_variable_segment_t
        case HSA_CODE_SYMBOL_INFO_VARIABLE_SEGMENT:
            return sizeof(hsa_variable_segment_t);

        // bool
        case HSA_CODE_SYMBOL_INFO_VARIABLE_IS_CONST:
        case HSA_CODE_SYMBOL_INFO_KERNEL_DYNAMIC_CALLSTACK:
            return sizeof(bool);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_code_symbol_get_info_AttributeString(void* value, hsa_code_symbol_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // hsa_symbol_t
                case HSA_CODE_SYMBOL_INFO_TYPE:
                    ss << HSATraceStringUtils::Get_hsa_symbol_kind_t_String(*(static_cast<hsa_symbol_kind_t*>(value)));
                    break;

                // uint32_t
                case HSA_CODE_SYMBOL_INFO_NAME_LENGTH:
                case HSA_CODE_SYMBOL_INFO_MODULE_NAME_LENGTH:
                case HSA_CODE_SYMBOL_INFO_VARIABLE_ALIGNMENT:
                case HSA_CODE_SYMBOL_INFO_VARIABLE_SIZE:
                case HSA_CODE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_SIZE:
                case HSA_CODE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_ALIGNMENT:
                case HSA_CODE_SYMBOL_INFO_KERNEL_GROUP_SEGMENT_SIZE:
                case HSA_CODE_SYMBOL_INFO_KERNEL_PRIVATE_SEGMENT_SIZE:
                case HSA_CODE_SYMBOL_INFO_INDIRECT_FUNCTION_CALL_CONVENTION:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // char*
                case HSA_CODE_SYMBOL_INFO_NAME: //TODO: Fix this to make caller read the length from HSA_ISA_INFO_NAME_LENGTH
                case HSA_CODE_SYMBOL_INFO_MODULE_NAME:
                    ss << GetStringString(static_cast<char*>(value), false);
                    break;

                // hsa_symbol_linkage_t
                case HSA_CODE_SYMBOL_INFO_LINKAGE:
                    ss << HSATraceStringUtils::Get_hsa_symbol_linkage_t_String(*(static_cast<hsa_symbol_linkage_t*>(value)));
                    break;

                // hsa_variable_allocation_t
                case HSA_CODE_SYMBOL_INFO_VARIABLE_ALLOCATION:
                    ss << HSATraceStringUtils::Get_hsa_variable_allocation_t_String(*(static_cast<hsa_variable_allocation_t*>(value)));
                    break;

                // hsa_variable_segment_t
                case HSA_CODE_SYMBOL_INFO_VARIABLE_SEGMENT:
                    ss << HSATraceStringUtils::Get_hsa_variable_segment_t_String(*(static_cast<hsa_variable_segment_t*>(value)));
                    break;

                // bool
                case HSA_CODE_SYMBOL_INFO_VARIABLE_IS_CONST:
                case HSA_CODE_SYMBOL_INFO_KERNEL_DYNAMIC_CALLSTACK:
                    ss << (*(static_cast<bool*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_executable_get_info_AttributeSize(hsa_executable_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // hsa_profile_t
        case HSA_EXECUTABLE_INFO_PROFILE:
            return sizeof(hsa_profile_t);

        // hsa_executable_state_t
        case HSA_EXECUTABLE_INFO_STATE:
            return sizeof(hsa_executable_state_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_executable_get_info_AttributeString(void* value, hsa_executable_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // hsa_profile_t
                case HSA_EXECUTABLE_INFO_PROFILE:
                    ss << HSATraceStringUtils::Get_hsa_profile_t_String(*(static_cast<hsa_profile_t*>(value)));
                    break;

                // hsa_executable_state_t
                case HSA_EXECUTABLE_INFO_STATE:
                    ss << HSATraceStringUtils::Get_hsa_executable_state_t_String(*(static_cast<hsa_executable_state_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_executable_symbol_get_info_AttributeSize(hsa_executable_symbol_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // hsa_symbol_kind_t
        case HSA_EXECUTABLE_SYMBOL_INFO_TYPE:
            return sizeof(hsa_symbol_kind_t);

        // uint32_t
        case HSA_EXECUTABLE_SYMBOL_INFO_NAME_LENGTH:
        case HSA_EXECUTABLE_SYMBOL_INFO_MODULE_NAME_LENGTH:
        case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_ALIGNMENT:
        case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_SIZE:
        case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_SIZE:
        case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_ALIGNMENT:
        case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_GROUP_SEGMENT_SIZE:
        case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_PRIVATE_SEGMENT_SIZE:
        case HSA_EXECUTABLE_SYMBOL_INFO_INDIRECT_FUNCTION_CALL_CONVENTION:
            return sizeof(uint32_t);

        // char*
        case HSA_EXECUTABLE_SYMBOL_INFO_NAME: //TODO: Fix this to make caller read the length from HSA_ISA_INFO_NAME_LENGTH
        case HSA_EXECUTABLE_SYMBOL_INFO_MODULE_NAME:
            return sizeof(char*);

        // hsa_agent_t
        case HSA_EXECUTABLE_SYMBOL_INFO_AGENT:
            return sizeof(hsa_agent_t);

        // uint64_t
        case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_ADDRESS:
        case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_OBJECT:
        case HSA_EXECUTABLE_SYMBOL_INFO_INDIRECT_FUNCTION_OBJECT: // TODO: the size of this one depends on the machine model.....
            return sizeof(uint64_t);

        // hsa_variable_allocation_t
        case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_ALLOCATION:
            return sizeof(hsa_variable_allocation_t);

        // hsa_variable_segment_t
        case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_SEGMENT:
            return sizeof(hsa_variable_segment_t);

        // bool
        case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_IS_CONST:
        case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_DYNAMIC_CALLSTACK:
            return sizeof(bool);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_executable_symbol_get_info_AttributeString(void* value, hsa_executable_symbol_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {
                // hsa_symbol_t
                case HSA_EXECUTABLE_SYMBOL_INFO_TYPE:
                    ss << HSATraceStringUtils::Get_hsa_symbol_kind_t_String(*(static_cast<hsa_symbol_kind_t*>(value)));
                    break;

                // uint32_t
                case HSA_EXECUTABLE_SYMBOL_INFO_NAME_LENGTH:
                case HSA_EXECUTABLE_SYMBOL_INFO_MODULE_NAME_LENGTH:
                case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_ALIGNMENT:
                case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_SIZE:
                case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_SIZE:
                case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_KERNARG_SEGMENT_ALIGNMENT:
                case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_GROUP_SEGMENT_SIZE:
                case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_PRIVATE_SEGMENT_SIZE:
                case HSA_EXECUTABLE_SYMBOL_INFO_INDIRECT_FUNCTION_CALL_CONVENTION:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // char*
                case HSA_EXECUTABLE_SYMBOL_INFO_NAME: //TODO: Fix this to make caller read the length from HSA_ISA_INFO_NAME_LENGTH
                case HSA_EXECUTABLE_SYMBOL_INFO_MODULE_NAME:
                    ss << GetStringString(static_cast<char*>(value), false);
                    break;

                // hsa_agent_t
                case HSA_EXECUTABLE_SYMBOL_INFO_AGENT:
                    ss << HSATraceStringUtils::Get_hsa_agent_t_String(*(static_cast<hsa_agent_t*>(value)));
                    break;

                // uint64_t
                case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_ADDRESS:
                case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_OBJECT:
                case HSA_EXECUTABLE_SYMBOL_INFO_INDIRECT_FUNCTION_OBJECT: // TODO: the size of this one depends on the machine model.....
                    ss << (*(static_cast<uint64_t*>(value)));
                    break;

                // hsa_variable_allocation_t
                case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_ALLOCATION:
                    ss << HSATraceStringUtils::Get_hsa_variable_allocation_t_String(*(static_cast<hsa_variable_allocation_t*>(value)));
                    break;

                // hsa_variable_segment_t
                case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_SEGMENT:
                    ss << HSATraceStringUtils::Get_hsa_variable_segment_t_String(*(static_cast<hsa_variable_segment_t*>(value)));
                    break;

                // bool
                case HSA_EXECUTABLE_SYMBOL_INFO_VARIABLE_IS_CONST:
                case HSA_EXECUTABLE_SYMBOL_INFO_KERNEL_DYNAMIC_CALLSTACK:
                    ss << (*(static_cast<bool*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_ext_program_get_info_AttributeSize(hsa_ext_program_info_t attribute)
{
    // TODO: add support for attributes defined by extensions

    switch (attribute)
    {
        // hsa_machine_model_t
        case HSA_EXT_PROGRAM_INFO_MACHINE_MODEL:
            return sizeof(hsa_machine_model_t);

        // hsa_profile_t
        case HSA_EXT_PROGRAM_INFO_PROFILE:
            return sizeof(hsa_profile_t);

        // hsa_default_float_rounding_mode_t
        case HSA_EXT_PROGRAM_INFO_DEFAULT_FLOAT_ROUNDING_MODE:
            return sizeof(hsa_default_float_rounding_mode_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_ext_program_get_info_AttributeString(void* value, hsa_ext_program_info_t attribute, hsa_status_t retVal)
{
    // TODO: add support for attributes defined by extensions

    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // hsa_machine_model_t
                case HSA_EXT_PROGRAM_INFO_MACHINE_MODEL:
                    ss << HSATraceStringUtils::Get_hsa_machine_model_t_String(*(static_cast<hsa_machine_model_t*>(value)));
                    break;

                // hsa_profile_t
                case HSA_EXT_PROGRAM_INFO_PROFILE:
                    ss << HSATraceStringUtils::Get_hsa_profile_t_String(*(static_cast<hsa_profile_t*>(value)));
                    break;

                // hsa_default_float_rounding_mode_t
                case HSA_EXT_PROGRAM_INFO_DEFAULT_FLOAT_ROUNDING_MODE:
                    ss << HSATraceStringUtils::Get_hsa_default_float_rounding_mode_t_String(*(static_cast<hsa_default_float_rounding_mode_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_amd_memory_pool_get_info_AttributeSize(hsa_amd_memory_pool_info_t attribute)
{
    switch (attribute)
    {
        // hsa_amd_segment_t
        case HSA_AMD_MEMORY_POOL_INFO_SEGMENT:
            return sizeof(hsa_amd_segment_t);

        // uint32_t
        case HSA_AMD_MEMORY_POOL_INFO_GLOBAL_FLAGS:
            return sizeof(uint32_t);

        // size_t
        case HSA_AMD_MEMORY_POOL_INFO_SIZE:
        case HSA_AMD_MEMORY_POOL_INFO_RUNTIME_ALLOC_GRANULE:
        case HSA_AMD_MEMORY_POOL_INFO_RUNTIME_ALLOC_ALIGNMENT:
            return sizeof(size_t);

        // bool
        case HSA_AMD_MEMORY_POOL_INFO_RUNTIME_ALLOC_ALLOWED:
        case HSA_AMD_MEMORY_POOL_INFO_ACCESSIBLE_BY_ALL:
            return sizeof(bool);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_amd_memory_pool_get_info_AttributeString(void* value, hsa_amd_memory_pool_info_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // hsa_amd_segment_t
                case HSA_AMD_MEMORY_POOL_INFO_SEGMENT:
                    ss << HSATraceStringUtils::Get_hsa_amd_segment_t_String(*(static_cast<hsa_amd_segment_t*>(value)));
                    break;

                // uint32_t
                case HSA_AMD_MEMORY_POOL_INFO_GLOBAL_FLAGS:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // size_t
                case HSA_AMD_MEMORY_POOL_INFO_SIZE:
                case HSA_AMD_MEMORY_POOL_INFO_RUNTIME_ALLOC_GRANULE:
                case HSA_AMD_MEMORY_POOL_INFO_RUNTIME_ALLOC_ALIGNMENT:
                    ss << StringUtils::ToString(*(static_cast<size_t*>(value)));
                    break;

                // bool
                case HSA_AMD_MEMORY_POOL_INFO_RUNTIME_ALLOC_ALLOWED:
                case HSA_AMD_MEMORY_POOL_INFO_ACCESSIBLE_BY_ALL:
                    ss << (*(static_cast<bool*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_amd_agent_memory_pool_get_info_AttributeSize(hsa_amd_agent_memory_pool_info_t attribute)
{
    switch (attribute)
    {
        // hsa_amd_memory_pool_access_t
        case HSA_AMD_AGENT_MEMORY_POOL_INFO_ACCESS:
            return sizeof(hsa_amd_memory_pool_access_t);

        // uint32_t
        case HSA_AMD_AGENT_MEMORY_POOL_INFO_NUM_LINK_HOPS:
            return sizeof(uint32_t);

        // hsa_amd_memory_pool_link_info_t
        case HSA_AMD_AGENT_MEMORY_POOL_INFO_LINK_INFO:
            return sizeof(hsa_amd_memory_pool_link_info_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_amd_agent_memory_pool_get_info_AttributeString(void* value, hsa_amd_agent_memory_pool_info_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // hsa_amd_memory_pool_access_t
                case HSA_AMD_AGENT_MEMORY_POOL_INFO_ACCESS:
                    ss << HSATraceStringUtils::Get_hsa_amd_memory_pool_access_t_String(*(static_cast<hsa_amd_memory_pool_access_t*>(value)));
                    break;

                // uint32_t
                case HSA_AMD_AGENT_MEMORY_POOL_INFO_NUM_LINK_HOPS:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // hsa_amd_memory_pool_link_info_t
                case HSA_AMD_AGENT_MEMORY_POOL_INFO_LINK_INFO:
                    ss << HSATraceStringUtils::Get_hsa_amd_memory_pool_link_info_t_String(*(static_cast<hsa_amd_memory_pool_link_info_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_cache_get_info_AttributeSize(hsa_cache_info_t attribute)
{
    switch (attribute)
    {
        // uint32_t
        case HSA_CACHE_INFO_NAME_LENGTH:
        case HSA_CACHE_INFO_SIZE:
            return sizeof(uint32_t);

        // char*
        case HSA_CACHE_INFO_NAME: //TODO: Fix this to make caller read the length from HSA_ISA_INFO_NAME_LENGTH
            return sizeof(char*);

        // uint8_t
        case HSA_CACHE_INFO_LEVEL:
            return sizeof(hsa_amd_memory_pool_link_info_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_cache_get_info_AttributeString(void* value, hsa_cache_info_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // uint32_t
                case HSA_CACHE_INFO_NAME_LENGTH:
                case HSA_CACHE_INFO_SIZE:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // char*
                case HSA_CACHE_INFO_NAME:
                    ss << GetStringString(static_cast<char*>(value), false, false);
                    break;

                // uint8_t
                case HSA_CACHE_INFO_LEVEL:
                    ss << Get_uint8_t_String(*(static_cast<uint8_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_wavefront_get_info_AttributeSize(hsa_wavefront_info_t attribute)
{
    switch (attribute)
    {
        // uint32_t
        case HSA_WAVEFRONT_INFO_SIZE:
            return sizeof(uint32_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_wavefront_get_info_AttributeString(void* value, hsa_wavefront_info_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // uint32_t
                case HSA_WAVEFRONT_INFO_SIZE:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_ven_amd_aqlprofile_get_info_AttributeSize(hsa_ven_amd_aqlprofile_info_type_t attribute)
{
    switch (attribute)
    {
        // uint32_t
        case HSA_VEN_AMD_AQLPROFILE_INFO_COMMAND_BUFFER_SIZE:
        case HSA_VEN_AMD_AQLPROFILE_INFO_PMC_DATA_SIZE:
            return sizeof(uint32_t);

        case HSA_VEN_AMD_AQLPROFILE_INFO_PMC_DATA:
        case HSA_VEN_AMD_AQLPROFILE_INFO_SQTT_DATA:
            return sizeof(uint64_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_ven_amd_aqlprofile_get_info_AttributeString(void* value, hsa_ven_amd_aqlprofile_info_type_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {

                // uint32_t
                case HSA_VEN_AMD_AQLPROFILE_INFO_COMMAND_BUFFER_SIZE:
                case HSA_VEN_AMD_AQLPROFILE_INFO_PMC_DATA_SIZE:
                    ss << (*(static_cast<uint32_t*>(value)));
                    break;

                // uint64_t
                case HSA_VEN_AMD_AQLPROFILE_INFO_PMC_DATA:
                case HSA_VEN_AMD_AQLPROFILE_INFO_SQTT_DATA:
                    ss << (*(static_cast<uint64_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}

unsigned int HSATraceStringUtils::Get_hsa_ven_amd_loader_loaded_code_object_get_info_AttributeSize(hsa_ven_amd_loader_loaded_code_object_info_t attribute)
{
    switch (attribute)
    {
        // hsa_executable_t
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_EXECUTABLE:
            return sizeof(hsa_executable_t);

        // hsa_ven_amd_loader_loaded_code_object_kind_t
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_KIND:
            return sizeof(hsa_ven_amd_loader_loaded_code_object_kind_t);

        // hsa_agent_t
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_AGENT:
            return sizeof(hsa_agent_t);

        // hsa_ven_amd_loader_code_object_storage_type_t
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_TYPE:
            return sizeof(hsa_ven_amd_loader_code_object_storage_type_t);

        // uint64_t
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_MEMORY_BASE:
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_MEMORY_SIZE:
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_LOAD_BASE:
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_LOAD_SIZE:
            return sizeof(uint64_t);

        // int
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_FILE:
            return sizeof(int);

        // int64_t
        case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_LOAD_DELTA:
            return sizeof(int64_t);

        default:
            return 0;
    }
}

std::string HSATraceStringUtils::Get_hsa_ven_amd_loader_loaded_code_object_get_info_AttributeString(void* value, hsa_ven_amd_loader_loaded_code_object_info_t attribute, hsa_status_t retVal)
{
    if (NULL == value)
    {
        return "NULL";
    }
    else
    {
        std::ostringstream ss;

        if (HSA_STATUS_SUCCESS == retVal)
        {
            switch (attribute)
            {
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_EXECUTABLE:
                    ss << HSATraceStringUtils::Get_hsa_executable_t_String(*(static_cast<hsa_executable_t*>(value)));
                    break;

                // hsa_ven_amd_loader_loaded_code_object_kind_t
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_KIND:
                    ss << HSATraceStringUtils::Get_hsa_ven_amd_loader_loaded_code_object_kind_t_String(*(static_cast<hsa_ven_amd_loader_loaded_code_object_kind_t*>(value)));
                    break;

                // hsa_agent_t
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_AGENT:
                    ss << HSATraceStringUtils::Get_hsa_agent_t_String(*(static_cast<hsa_agent_t*>(value)));
                    break;

                // hsa_ven_amd_loader_code_object_storage_type_t
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_TYPE:
                    ss << HSATraceStringUtils::Get_hsa_ven_amd_loader_code_object_storage_type_t_String(*(static_cast<hsa_ven_amd_loader_code_object_storage_type_t*>(value)));
                    break;

                // uint64_t
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_MEMORY_BASE:
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_MEMORY_SIZE:
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_LOAD_BASE:
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_LOAD_SIZE:
                    ss << (*(static_cast<uint64_t*>(value)));
                    break;

                // int
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_CODE_OBJECT_STORAGE_FILE:
                    ss << (*(static_cast<int*>(value)));
                    break;

                // int64_t
                case HSA_VEN_AMD_LOADER_LOADED_CODE_OBJECT_INFO_LOAD_DELTA:
                    ss << (*(static_cast<int64_t*>(value)));
                    break;

                default:
                    ss << StringUtils::ToString(*(static_cast<int*>(value)));
                    break;
            }
        }

        return SurroundWithDeRef(ss.str());
    }
}
