
#include "stdafx.h"
#include <string>

#include "AMDGenlockConfigManager.h"
#include "AMDGenlockMonitor.h"
#include "AMDGenlockMonitorDlg.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#endif


BEGIN_MESSAGE_MAP(AMDGenlockMonitorDlg, CDialogEx)
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
    ON_WM_TIMER()
    ON_COMMAND(ID_FILE_LOADCONFIG,  &AMDGenlockMonitorDlg::OnLoadConfig)
    ON_COMMAND(ID_FILE_SAVECONFIG,  &AMDGenlockMonitorDlg::OnSaveConfig)
    ON_COMMAND(ID_FILE_INFO,        &AMDGenlockMonitorDlg::OnInfo)
    ON_COMMAND(ID_FILE_EXIT,        &AMDGenlockMonitorDlg::OnExit)
    ON_BN_CLICKED(IDC_START_BTN,    &AMDGenlockMonitorDlg::OnStart)
    ON_BN_CLICKED(IDC_STOP_BTN,     &AMDGenlockMonitorDlg::OnStop)
END_MESSAGE_MAP()


using namespace std;

AMDGenlockMonitorDlg::AMDGenlockMonitorDlg(CWnd* pParent /*=NULL*/)
	: CDialogEx                 (AMDGenlockMonitorDlg::IDD, pParent),
      m_pGenlockManager         (NULL),
      m_bDoConfigCheck          (false),
      m_bConfigurationLost      (false),
      m_uiNumRestoreTries       (0)
{
	m_hIcon = AfxGetApp()->LoadIcon(IDI_ICON1);
}


void AMDGenlockMonitorDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialogEx::DoDataExchange(pDX);
}



BOOL AMDGenlockMonitorDlg::OnInitDialog()
{
    string  strStatus;
    CMenu   menu;

	CDialogEx::OnInitDialog();

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon

    menu.LoadMenu(IDR_MENU1);

    SetMenu(&menu);

    // Create GenlockManager to observe the configuration state
    m_pGenlockManager = new AMDGenlockConfigManager;

    if (!m_pGenlockManager->init(true))
    {
        MessageBox("Failed to int AMD Genlock Configuration Manager.", "Error", MB_OK | MB_ICONERROR );
        OnExit();
    }

    if (m_pGenlockManager->enumDisplays() == 0)
    {
        MessageBox("No Displays detected.", "Error", MB_OK | MB_ICONERROR );
        OnExit();
    }

    if (m_pGenlockManager->hasTimingServer() || m_pGenlockManager->getNumTimingClients() > 0)
    {
        m_bDoConfigCheck = true;
    }
    

    // Create timer to regularly read the genlock configuration state
    ::SetTimer(m_hWnd, 1, 1000, NULL);

	return TRUE;  // return TRUE  unless you set the focus to a control
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void AMDGenlockMonitorDlg::OnPaint()
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialogEx::OnPaint();
	}
}

// The system calls this function to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR AMDGenlockMonitorDlg::OnQueryDragIcon()
{
	return static_cast<HCURSOR>(m_hIcon);
}



// This function will be called on each timer event. The interval is defined
// in initDialg
void AMDGenlockMonitorDlg::OnTimer(UINT nIDEvent)
{
    string strStatus;

    // check if a valid configuration is present hat can be monitored
    if (m_bDoConfigCheck)
    {
        if (m_bConfigurationLost)
        {
            // We have currently no more TS or TC running, check if the external references
            // that are needed are present.
            if (m_pGenlockManager->referencePresent())
            {
                if (m_pGenlockManager->restoreConfiguration())
                {
                    // successfully restored config
                    m_bConfigurationLost = false;

                    m_uiNumRestoreTries = 0;
                }
                else
                {
                    // failed to restore
                    ++m_uiNumRestoreTries;

                    if (m_uiNumRestoreTries >= MAX_RESTORE_TRIES)
                    {
                        m_bDoConfigCheck    = false;
                        m_uiNumRestoreTries = 0;

                        MessageBox("Failed to restore configuration. Stopping monitoring", "Error", MB_OK | MB_ICONERROR );
                    }
                }
            }
            else
            {
                // No external refernce present. Need to wait until it is present
                // before restoring the configuration
                SetDlgItemText(IDC_STATUS, "Waiting for external refernce.");
            }
        }
        else if (!m_pGenlockManager->configurationPresent())
        {
            m_bConfigurationLost = true;
        }
        else
        {
            SetDlgItemText(IDC_STATUS, "Monitoring local configuration.");
        }
    }
    else
    {
        // Nothing to monitor
        SetDlgItemText(IDC_STATUS, "Nothing to monitor.");
    }
}


void AMDGenlockMonitorDlg::OnLoadConfig()
{
    CFileDialog dlgFileLoad(TRUE, _T("gc"), NULL, 0, _T("AMD Genlock Configuration (*.gc)|*.gc|All files (*.*)|*.*||"), this, 0);
    
    if (dlgFileLoad.DoModal() == IDOK && m_pGenlockManager)
    {
        string strPathName;

        strPathName = dlgFileLoad.GetPathName();

        if (!m_pGenlockManager->loadConfiguration(strPathName.c_str()))
        {
            MessageBox("Failed to load genlock configuration.", "Error", MB_OK | MB_ICONERROR);
            OnStop();
        }
        else
        {
            // Only restore configuration if all external references are present
            if (m_pGenlockManager->referencePresent())
            {
                if (!m_pGenlockManager->restoreConfiguration())
                {
                    OnStop();
                    MessageBox("Failed to apply genlock configuration.", "Error", MB_OK | MB_ICONERROR);
                }
                else 
                {
                    // start monitoring
                    OnStart();
                }
            }
            else
            {
                OnStop();
                MessageBox("Missing external reference.", "Error", MB_OK | MB_ICONERROR);
            }
        }
    }
}


void AMDGenlockMonitorDlg::OnSaveConfig()
{
    CFileDialog dlgFileSave(FALSE, _T("gc"), NULL, 0, _T("AMD Genlock Configuration (*.gc)|*.gc|All files (*.*)|*.*||"), this, 0);
    
    if (dlgFileSave.DoModal() == IDOK && m_pGenlockManager)
    {
        string strPathName;

        strPathName = dlgFileSave.GetPathName();

        if (!m_pGenlockManager->saveConfiguration(strPathName.c_str()))
        {
            MessageBox("Failed to load genlock configuration.", "Error", MB_OK | MB_ICONERROR);
        }
    }
}


void AMDGenlockMonitorDlg::OnInfo()
{
    char            cBuffer[128];
    unsigned int    uiModuleId  = 0;
    unsigned int    uiFWVersion = 0;
    string          strMessage;

    for (unsigned int i = 0; i < m_pGenlockManager->getNumGpus(); ++i)
    {
        if (m_pGenlockManager->getS400Info(i, uiModuleId, uiFWVersion))
        {
            sprintf_s(cBuffer,"\nGPU %d: connected to S400 id: %d Firmware: %d", i, uiModuleId, uiFWVersion);
        }
        else
        {
            sprintf_s(cBuffer,"\nGPU %d: not connected to S400", i);
        }
    }

    MessageBox(cBuffer, "S400 Module Info", MB_OK | MB_ICONINFORMATION);
}


void AMDGenlockMonitorDlg::OnExit()
{
    if (m_pGenlockManager)
    {
        delete m_pGenlockManager;
    }

    OnOK();
}


void AMDGenlockMonitorDlg::OnStart()
{
    // read cuurent display configuration
    if (m_pGenlockManager->enumDisplays())
    {
        m_bDoConfigCheck = (m_pGenlockManager->hasTimingServer() || m_pGenlockManager->getNumTimingClients() > 0);
    }
}


void AMDGenlockMonitorDlg::OnStop()
{
    m_bDoConfigCheck = false;
}
