#pragma once


#include "adl_sdk.h"
#include <vector>

class LogOutput;

class AMDGenlockConfigManager
{
public:

    AMDGenlockConfigManager(void);
    virtual ~AMDGenlockConfigManager(void);

    // Initialization and loading of ADL
    // [in] bLogEvents: enables logging of events to a file
    bool                init(bool bLogEvents = true);
    // Enumerates displays and creates a list of GPUs and mapped Displays. The function reads for each
    // GPU the Genlock config, for each Display the GenlockMode and for each S400 the Port State.
    unsigned int        enumDisplays();

    // Returns true is a valid input signal is present on the specific port.
    // [in] uiPort: 0: BNC, 1: RJ45 Port1, 2: RJ45 Port2 
    bool                signalOnPortPresent(unsigned int uiPort);
    // Returns true if a TimingServer is running. 
    bool                localTSPresent();
    // Returns true if the display uiTCIdx is running as TimingClient
    // [in] uiTCIdx: Display Index
    bool                localTCPresent(unsigned int uiTCIdx);
    // Returns true if all required external references are present. The function checks per GPU
    // the signal source and verifies if it is presenr.
    bool                referencePresent();
    // Returns true if all Displays still have the same Genlock state as at enumeration time. 
    bool                configurationPresent();

    // Applys the genlock settings captured at enumeration or load time to all displays. 
    // Returns tue if all displays could be reconfigured to their initial state.
    bool                restoreConfiguration();

    // Save the configuration of  the GPUs, all mapped displays and the Ports to a file.
    // Returns true if the file was written successfully
    // [in] pFileName: Path to the file.
    bool                saveConfiguration(const char* pFileName);
    // Loads the configuration from a file and tries to apply it to the current displays. It is required
    // that the current display configuration matches the configuration that was loaded from the file.
    // This means the number of GPUs and the number of displays need to match and the Adapter and display
    // Indexes read needs to match those that are present.
    // Returns true if the configuartion from file could be applied successfully.
    // [in] pFileName: Path to file
    bool                loadConfiguration(const char* pFileName);

    // Returns true if the local configuration contains a TimingServer
    bool                hasTimingServer()           { return  m_bTimingServer;       };
    // Returns the number of Timingclients that were enumerated.
    unsigned int        getNumTimingClients()       { return  m_TCList.size();       };
    
    // returns the number of GPUs in the system
    unsigned int        getNumGpus()                { return m_GpuList.size();          };
    // returns the signal source of a gpu
    // 0: BNC,  1: RJ45 port 1,  2: RJ45 Port 2  3: No Signal source
    // [in] uiGpuId: Id of the GPu to query
    unsigned int        getSignalSource(unsigned int uiGpuid);
    // returns true if the Gpu with id is connected to a S400
    // [in] uiGpuId: Id of the GPU to query
    bool                connectedToS400(unsigned int uiGpuId);

    bool                getS400Info(unsigned int uiGpuId, unsigned int &uiModuleId, unsigned int &uiFWVer);

private:

    // Data Structure to store per Display information
    typedef struct
    {
        unsigned int        uiDisplayId;
        unsigned int        uiAdapterId;
        unsigned int        uiSerialId;
        ADLGlSyncMode       adlGLSyncMode;
    }
    DisplayData;

    // Data Structure to store per GPU information
    typedef struct
    {
        unsigned int            uiBusId;
        unsigned int            uiFirstAdapterId;
        ADLGLSyncModuleID       adlGlSyncModule;
        ADLGLSyncGenlockConfig  adlGlSyncGenlockConfig;
    } GpuData;

    
    // Loads ADL functions
    bool            setupADL();
    // Reads the serial ID of a connected monitor
    unsigned int    readSerialId(unsigned int uiAdpaterId, unsigned int uiDisplayId);

    // Writes log message to a log file
    void            outputLog(unsigned int uiMsgType, const char* );

    bool                            m_bTimingServer;
    unsigned int                    m_uiTSIdx;
    

    std::vector<GpuData*>           m_GpuList;
    std::vector<DisplayData*>       m_DisplayList;
    std::vector<unsigned int>       m_TCList;

    ADLGlSyncPortInfo               m_adlGLSyncPortInfo[3];

    LogOutput*                      m_pLog;
};

