

#if defined(WIN32)
#include <Windows.h>

#define GETPROC GetProcAddress

#elif defined (LINUX)

#include <stdlib.h>
#include <string.h>
#include <dlfcn.h>

#define HMODULE void*
#define GETPROC dlsym
#define sprintf_s sprintf

#endif

#include "LogOutput.h"
#include "adl_prototypes.h"
#include "AMDGenlockConfigManager.h"


static ADL_PROCS  g_AdlCalls = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };


// Memory Allocation function needed for ADL
void* __stdcall ADL_Alloc ( int iSize )
{
    void* lpBuffer = malloc ( iSize );
    return lpBuffer;
}

// Memory Free function needed for ADL
void __stdcall ADL_Free ( void* lpBuffer )
{
    if ( NULL != lpBuffer )
    {
        free ( lpBuffer );
        lpBuffer = NULL;
    }
}

using namespace std;


AMDGenlockConfigManager::AMDGenlockConfigManager(void)
{
    m_bTimingServer      = false;
    m_uiTSIdx            = 0;

    m_GpuList.clear();
    m_DisplayList.clear();
    m_TCList.clear();

    memset(m_adlGLSyncPortInfo,   0, sizeof(ADLGlSyncPortInfo) * 3);

    m_pLog = NULL;
}


AMDGenlockConfigManager::~AMDGenlockConfigManager(void)
{
    vector<DisplayData*>::iterator itr;

    for (itr = m_DisplayList.begin(); itr != m_DisplayList.end(); ++itr)
    {
        delete (*itr);
    }

    vector<GpuData*>::iterator  itrGpu;

    for (itrGpu = m_GpuList.begin(); itrGpu != m_GpuList.end(); ++itrGpu)
    {
        delete (*itrGpu);
    }

    m_GpuList.clear();
    m_DisplayList.clear();
    m_TCList.clear();

    if (m_pLog)
    {
        m_pLog->closeLogFile();
        delete m_pLog;
    }
}


bool AMDGenlockConfigManager::init(bool bLogEvents)
{
    // Load ADL if not yet done
    if (g_AdlCalls.hModule == NULL)
    {
        if (!setupADL())
        {
            return false;
        }
    }

    // create log file
    if (bLogEvents)
    {
        m_pLog = new LogOutput;
        m_pLog->openLogFile("AMDGenlockMonitor.log");
    }

    return true;
}


unsigned int AMDGenlockConfigManager::enumDisplays()
{
    int             nNumDisplays;
    int             nNumAdapters;
    unsigned int    uiGpuId         = 0;
    unsigned int    uiDspIdx        = 0;
    LPAdapterInfo   pAdapterInfo    = NULL;

    if (g_AdlCalls.hModule == NULL)
        return 0;

    if (g_AdlCalls.ADL_Adapter_NumberOfAdapters_Get(&nNumAdapters) != ADL_OK  ||  nNumAdapters == 0)
    {
        return 0;
    }

    // delete any previously stored configuration
    vector<DisplayData*>::iterator itr;

    for (itr = m_DisplayList.begin(); itr != m_DisplayList.end(); ++itr)
    {
        delete (*itr);
    }

    vector<GpuData*>::iterator  itrGpu;

    for (itrGpu = m_GpuList.begin(); itrGpu != m_GpuList.end(); ++itrGpu)
    {
        delete (*itrGpu);
    }

    m_GpuList.clear();
    m_DisplayList.clear();
    m_TCList.clear();

    pAdapterInfo = new AdapterInfo[nNumAdapters];

    if (g_AdlCalls.ADL_Adapter_AdapterInfo_Get(pAdapterInfo, nNumAdapters * sizeof(AdapterInfo)) != ADL_OK)
    {
        delete [] pAdapterInfo;
        return 0;
    }

    // Loop through all adapters to find mapped displays
    for (int i = 0; i < nNumAdapters; ++i)
    {
        int     nAdapterStatus = 0;
        int     nAdapterIndex  = pAdapterInfo[i].iAdapterIndex;

        g_AdlCalls.ADL_Adapter_Active_Get(nAdapterIndex, & nAdapterStatus);

        if (nAdapterStatus  == ADL_TRUE)
        {
            bool    bNewGpu = true;

            // check if we found a new gpu
            for (unsigned int g = 0; g < m_GpuList.size(); ++g)
            {
                if (m_GpuList[g]->uiBusId == pAdapterInfo[i].iBusNumber)
                {
                    bNewGpu = false;
                    uiGpuId = g;

                    break;
                }
            }

            if (bNewGpu)
            {
                // Add a new GPU
                GpuData* pGpu = new GpuData;

                pGpu->uiBusId           = pAdapterInfo[i].iBusNumber;
                pGpu->uiFirstAdapterId  = nAdapterIndex;

                // Read GLSync Module information
                if (g_AdlCalls.ADL_Workstation_GLSyncModuleDetect_Get(pGpu->uiFirstAdapterId, 0, &(pGpu->adlGlSyncModule)) != ADL_OK)
                {
                    memset(&(pGpu->adlGlSyncModule), 0, sizeof(ADLGLSyncModuleID));
                }
                // Read Genlock configuration for each GPU
                if (g_AdlCalls.ADL_Workstation_GLSyncGenlockConfiguration_Get(pGpu->uiFirstAdapterId, 0, ADL_GLSYNC_CONFIGMASK_ALL, &(pGpu->adlGlSyncGenlockConfig)) != ADL_OK)
                {
                    // Failed to read a valid config, reset data structure
                    memset(&(pGpu->adlGlSyncGenlockConfig), 0, sizeof(ADLGLSyncGenlockConfig));
                }
                // Store Genlock configuration for this GPU.
                m_GpuList.push_back(pGpu);

                uiGpuId = m_GpuList.size() - 1;
            }

            // Found active adapter, now get disply information
            LPADLDisplayInfo    pDisplayInfo = NULL;

            g_AdlCalls.ADL_Display_DisplayInfo_Get(nAdapterIndex, &nNumDisplays, &pDisplayInfo, 0);

            for (int j = 0; j < nNumDisplays; ++j)
            {
                // check if display is connected and mapped
				if (pDisplayInfo[j].iDisplayInfoValue & ADL_DISPLAY_DISPLAYINFO_DISPLAYCONNECTED)
				{
					// check if display is mapped on adapter
					if (pDisplayInfo[j].iDisplayInfoValue & ADL_DISPLAY_DISPLAYINFO_DISPLAYMAPPED && pDisplayInfo[j].displayID.iDisplayLogicalAdapterIndex == nAdapterIndex)
					{
                        // Found a valid display
                        DisplayData* pDsp = new DisplayData;
                         
                        pDsp->uiAdapterId  = nAdapterIndex;
                        pDsp->uiDisplayId  = pDisplayInfo[j].displayID.iDisplayLogicalIndex;
                        pDsp->uiSerialId   = readSerialId(pDsp->uiAdapterId, pDsp->uiDisplayId);

                        if (m_GpuList[uiGpuId]->adlGlSyncModule.iModuleID != 0 && g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Get(nAdapterIndex, pDsp->uiDisplayId, &(pDsp->adlGLSyncMode)) == ADL_OK)
                        {
                            // The status vector indicates if a display is genlocked or not. The iStatusVector will have the
                            // ADL_GLSYNC_MODECNTL_STATUS_GENLOCK bit set for the TS as well. The ControlVector will only set the 
                            // ADL_GLSYNC_MODECNTL_GENLOCK bit if the TS is synchronized to an external reference.
                            if ((pDsp->adlGLSyncMode.iStatusVector & ADL_GLSYNC_MODECNTL_STATUS_GENLOCK))
                            {
                                // A display is genlocked if it is TimingClient or TimingServer 
                                if (pDsp->adlGLSyncMode.iControlVector & ADL_GLSYNC_MODECNTL_TIMINGSERVER)
                                {
                                    // Indicate that a TimingServer is present on this system
                                    m_bTimingServer = true;
                                    m_uiTSIdx       = uiDspIdx;
                                }
                                else 
                                {
                                    // Since the genlocked display is no TS it is a TC. Add index to TC list
                                    m_TCList.push_back(uiDspIdx);
                                }
                            }
                        }
                        else
                        {
                            memset(&(pDsp->adlGLSyncMode), 0, sizeof(ADLGlSyncMode));
                        }

                        m_DisplayList.push_back(pDsp);
                        ++uiDspIdx;
                    }
                }  // Display is connected
            } // Display loop

            ADL_Free(pDisplayInfo);
        }
    } // Adapter look

    delete [] pAdapterInfo;

    // Use first active adapter to read the S400 Port configuration
    if (m_DisplayList.size() > 0)
    {
        int* pLEDs = NULL;

        // read configuratiomn of the BNC
        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Get(m_DisplayList[0]->uiAdapterId, 0, ADL_GLSYNC_PORT_BNC, 0, &(m_adlGLSyncPortInfo[0]), &pLEDs) != ADL_OK)
        {
            memset(&m_adlGLSyncPortInfo[0],   0, sizeof(ADLGlSyncPortInfo));
        }

        ADL_Free(pLEDs);

        // read configuration of the RJ45 Port 1
        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Get(m_DisplayList[0]->uiAdapterId, 0, ADL_GLSYNC_PORT_RJ45PORT1, 0, &(m_adlGLSyncPortInfo[1]), &pLEDs) != ADL_OK)
        {
            memset(&(m_adlGLSyncPortInfo[1]),   0, sizeof(ADLGlSyncPortInfo));
        }

        ADL_Free(pLEDs);

        // read configuration of the RJ45 Port 2
        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Get(m_DisplayList[0]->uiAdapterId, 0, ADL_GLSYNC_PORT_RJ45PORT2, 0, &(m_adlGLSyncPortInfo[2]), &pLEDs) != ADL_OK)
        {
            memset(&(m_adlGLSyncPortInfo[2]),   0, sizeof(ADLGlSyncPortInfo));
        }

        ADL_Free(pLEDs);
    }


    return m_DisplayList.size();
}



bool AMDGenlockConfigManager::signalOnPortPresent(unsigned int uiPort)
{
    int*                pLEDs = NULL;
    ADLGlSyncPortInfo   adlPortInfo;

    if (uiPort == 0)
    {
        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Get(m_DisplayList[0]->uiAdapterId, 0, ADL_GLSYNC_PORT_BNC, 0, &adlPortInfo, &pLEDs) != ADL_OK)
        {
            return false;
        }
    }
    else if (uiPort == 1)
    {
        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Get(m_DisplayList[0]->uiAdapterId, 0, ADL_GLSYNC_PORT_RJ45PORT1, 0, &adlPortInfo, &pLEDs) != ADL_OK)
        {
            return false;
        }
    }
    else if (uiPort == 2)
    {
        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Get(m_DisplayList[0]->uiAdapterId, 0, ADL_GLSYNC_PORT_RJ45PORT2, 0, &adlPortInfo, &pLEDs) != ADL_OK)
        {
            return false;
        }
    }
    else
    {
        return false;
    }

    // Checck if input signal is present and if the frequency is valid
    if (adlPortInfo.iPortState != ADL_GLSYNC_PORTSTATE_INPUT)
    {
        return false;
    }
    else if (abs(adlPortInfo.iFrequency - m_adlGLSyncPortInfo[uiPort].iFrequency) >10000)
    {
        // the current frequenc differs more than 10 Hz from original one
       return false;
    }

    return true;
}



bool AMDGenlockConfigManager::localTSPresent()
{
    ADLGlSyncMode adlSyncMode;

    if (!m_bTimingServer)
    {
        outputLog(LogOutput::MSG_WARNING, "Requesting TS status, but no TS was defined on this system.");
        return false;
    }

    // Get status of the TS
    if (g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Get(m_DisplayList[m_uiTSIdx]->uiAdapterId, m_DisplayList[m_uiTSIdx]->uiDisplayId, &adlSyncMode) != ADL_OK)
    {
        outputLog(LogOutput::MSG_ERROR, "Failed to read status for TS.");
        return false;
    }

    // check status
    if (!(adlSyncMode.iControlVector & ADL_GLSYNC_MODECNTL_TIMINGSERVER))
    {
        outputLog(LogOutput::MSG_INFO, "TS no longer running");
        return false;
    }

    return true;
}


bool AMDGenlockConfigManager::localTCPresent(unsigned int uiTCIdx)
{
    ADLGlSyncMode adlSyncMode;

    if (uiTCIdx >= m_TCList.size())
    {
        // wrong ID
        return false;
    }

    // get the display index of the TC with id uiTCIdx
    unsigned int uiDisplayIdx = m_TCList[uiTCIdx];

    if (g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Get(m_DisplayList[uiDisplayIdx]->uiAdapterId, m_DisplayList[uiDisplayIdx]->uiDisplayId, &adlSyncMode) != ADL_OK)
    {
        char cBuffer[128];

        sprintf_s(cBuffer,"TC %d failed to read status. Adapter Idx: %d, Display Idx: %d", uiTCIdx, m_DisplayList[uiDisplayIdx]->uiAdapterId, m_DisplayList[uiDisplayIdx]->uiDisplayId);
        outputLog(LogOutput::MSG_ERROR, cBuffer);

        return false;
    }

    if (!(adlSyncMode.iStatusVector & ADL_GLSYNC_MODECNTL_STATUS_GENLOCK))
    {
        char cBuffer[128];

        sprintf_s(cBuffer,"TC %d no longer running. Adapter Idx: %d, Display Idx: %d", uiTCIdx, m_DisplayList[uiDisplayIdx]->uiAdapterId, m_DisplayList[uiDisplayIdx]->uiDisplayId);
        outputLog(LogOutput::MSG_INFO, cBuffer);

        return false;
    }

    return true;
}


bool AMDGenlockConfigManager::referencePresent()
{
    
    for (unsigned int i = 0; i < m_GpuList.size(); ++i)
    {
        int uiPort = 3;

        if (m_GpuList[i]->adlGlSyncGenlockConfig.iSignalSource == ADL_GLSYNC_SIGNALSOURCE_BNCPORT)
        {
            uiPort = 0;
        }
        else if (m_GpuList[i]->adlGlSyncGenlockConfig.iSignalSource == ADL_GLSYNC_SIGNALSOURCE_RJ45PORT1)
        {
            uiPort = 1;
        }
        else if (m_GpuList[i]->adlGlSyncGenlockConfig.iSignalSource == ADL_GLSYNC_SIGNALSOURCE_RJ45PORT2)
        {
            uiPort = 2;
        }

        if (uiPort < 3 && !signalOnPortPresent(uiPort))
        {
            return false;
        }
    }

    return true;
}



bool AMDGenlockConfigManager::configurationPresent()
{
    bool bStatus = true;

    if (m_bTimingServer && !localTSPresent())
    {
        // The system has a TSwhich is currently not running
        return false;
    }

    // check TCs
    for (unsigned int i = 0; i < m_TCList.size(); ++i)
    {
        if (!localTCPresent(i))
        {
            // TC was configured but is currently not running
            return false;
        }
    }

    return true;
}


bool AMDGenlockConfigManager::restoreConfiguration()
{
    char            cBuffer[256];
    bool            bGpuIsConnected = false;
    unsigned int    uiAdapterToS400 = 0;
    unsigned int    uiSerialId;

    if (m_DisplayList.size() == 0)
    {
        outputLog(LogOutput::MSG_INFO, "No displays present. Cannot restore settings.");
        return false;
    }

    // Restore Genlock configuration on each GPU
    for (unsigned int i = 0; i < m_GpuList.size(); ++i)
    {
        // Check if GPU is attached to a S400
        if (m_GpuList[i]->adlGlSyncModule.iModuleID != 0)
        {
            if (g_AdlCalls.ADL_Workstation_GLSyncGenlockConfiguration_Set(m_GpuList[i]->uiFirstAdapterId, 0, m_GpuList[i]->adlGlSyncGenlockConfig) != ADL_OK)
            {
                outputLog(LogOutput::MSG_ERROR, "Failed to restore Genlock configuration");
                return false;
            }
            else
            {
                sprintf_s(cBuffer,"Restored Genlock config on GPU %d", i);
                outputLog(LogOutput::MSG_INFO, cBuffer);
            }

            // Only one S400 per system is supported, so we pick any GPU that is connected
            // to the S400 and use the AdapterIndex to communicate with the S400. 
            bGpuIsConnected = true;
            uiAdapterToS400 = m_GpuList[i]->uiFirstAdapterId;
        }
    }

    // Restore TS
    if (m_bTimingServer)
    {
        uiSerialId = readSerialId(m_DisplayList[m_uiTSIdx]->uiAdapterId, m_DisplayList[m_uiTSIdx]->uiDisplayId);

        if (uiSerialId != m_DisplayList[m_uiTSIdx]->uiSerialId)
        {
            // The monitor that was connected when reading the genlock configuration does not match the monitor
            // that is currently connected. The configuration should not be restored. 
            outputLog(LogOutput::MSG_ERROR, "Failed to restore TS. The monitor configuration changed.");
            return false;
        }

        if (g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Set(m_DisplayList[m_uiTSIdx]->uiAdapterId, m_DisplayList[m_uiTSIdx]->uiDisplayId, m_DisplayList[m_uiTSIdx]->adlGLSyncMode) != ADL_OK)
        {
            outputLog(LogOutput::MSG_ERROR, "Failed to restore TS");
            return false;
        }
        else
        {
            sprintf_s(cBuffer,"Restored Genlock mode on TS Adapter %d, Display %d", m_DisplayList[m_uiTSIdx]->uiAdapterId, m_DisplayList[m_uiTSIdx]->uiDisplayId);
            outputLog(LogOutput::MSG_INFO, cBuffer);
        }
    }

    // Restore RJ45 Port 1. This is only needed if RJ45 was configured as output
    if (bGpuIsConnected && m_adlGLSyncPortInfo[1].iPortState & ADL_GLSYNC_PORTSTATE_OUTPUT)
    {
        ADLGlSyncPortControl adlPortCtl;

        adlPortCtl.iControlVector = ADL_GLSYNC_PORTCNTL_OUTPUT;
        adlPortCtl.iPortType      = ADL_GLSYNC_PORT_RJ45PORT1;
        adlPortCtl.iSignalSource  = m_adlGLSyncPortInfo[1].iSignalSource;

        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Set(uiAdapterToS400, 0, adlPortCtl) != ADL_OK)
        {
            outputLog(LogOutput::MSG_ERROR, "Failed to restore RJ 45 Port 1");
            return false;
        }
        else
        {
            outputLog(LogOutput::MSG_INFO, "Restored Port State on RJ45 Port 1");
        }
    }

    // Restore RJ45 Port 2. This is only needed if RJ45 was configured as output
    if (bGpuIsConnected && m_adlGLSyncPortInfo[2].iPortState & ADL_GLSYNC_PORTSTATE_OUTPUT)
    {
        ADLGlSyncPortControl adlPortCtl;

        adlPortCtl.iControlVector = ADL_GLSYNC_PORTCNTL_OUTPUT;
        adlPortCtl.iPortType      = ADL_GLSYNC_PORT_RJ45PORT2;
        adlPortCtl.iSignalSource  = m_adlGLSyncPortInfo[2].iSignalSource;

        if (g_AdlCalls.ADL_Workstation_GLSyncPortState_Set(uiAdapterToS400, 0, adlPortCtl) != ADL_OK)
        {
            outputLog(LogOutput::MSG_ERROR, "Failed to restore RJ 45 Port 2");
            return false;
        }
        else
        {
            outputLog(LogOutput::MSG_INFO, "Restored Port State on RJ45 Port 2");
        }
    }

    // Restore TimingClients
    for (unsigned int i = 0; i < m_TCList.size(); ++i)
    {
        char         cBuffer[128];
        unsigned int uiDspIdx = m_TCList[i];

        uiSerialId = readSerialId(m_DisplayList[uiDspIdx]->uiAdapterId, m_DisplayList[uiDspIdx]->uiDisplayId);

        if (uiSerialId != m_DisplayList[uiDspIdx]->uiSerialId)
        {
            // The monitor that was connected when reading the genlock configuration does not match the monitor
            // that is currently connected. The configuration should not be restored. 
            sprintf_s(cBuffer,"Failed to restore TC %d. Adapter Idx: %d, Display Idx: %d. The monitor configuration changed.", i, m_DisplayList[uiDspIdx]->uiAdapterId, m_DisplayList[uiDspIdx]->uiDisplayId);
            
            outputLog(LogOutput::MSG_ERROR, cBuffer);
            return false;
        }

        if (g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Set(m_DisplayList[uiDspIdx]->uiAdapterId, m_DisplayList[uiDspIdx]->uiDisplayId, m_DisplayList[uiDspIdx]->adlGLSyncMode) != ADL_OK)
        {
            sprintf_s(cBuffer,"Failed to restore TC %d. Adapter Idx: %d, Display Idx: %d", i, m_DisplayList[uiDspIdx]->uiAdapterId, m_DisplayList[uiDspIdx]->uiDisplayId);
            
            outputLog(LogOutput::MSG_ERROR, cBuffer);
            return false;
        }
        else
        {
            sprintf_s(cBuffer,"Restored Genlock mode on TC %d, Adapter %d, Display %d", i, m_DisplayList[uiDspIdx]->uiAdapterId, m_DisplayList[uiDspIdx]->uiDisplayId);
            outputLog(LogOutput::MSG_INFO, cBuffer);
        }
    }


    return true;
}


bool AMDGenlockConfigManager::saveConfiguration(const char* pFileName)
{
    fstream outFile;

    if (m_DisplayList.size() == 0)
    {
        return false;
    }

    outFile.open(pFileName, fstream::binary | fstream::out | fstream::trunc);

    if (!outFile.is_open())
    {
        return false;
    }

    // Write global information.
    // Number of GPUs (unsigned int) Number of Displays (unsigned int) Number of TCs (unsigned int)
    unsigned int uiGlobals[3];

    uiGlobals[0] = m_GpuList.size();
    uiGlobals[1] = m_DisplayList.size();
    uiGlobals[2] = m_TCList.size();

    outFile.write((const char*)uiGlobals, 3 * sizeof(int));

    // write GPU Data including the GenlockConfiguration
    for (unsigned int i = 0; i < m_GpuList.size(); ++i)
    {
        outFile.write((const char*)m_GpuList[i], sizeof(GpuData));
    }

    // write 3 x Port info (BNC, Port 1, Port 2)
    outFile.write((const char*)m_adlGLSyncPortInfo, 3 * sizeof(ADLGlSyncPortInfo));

    // write display information
    for (unsigned int i = 0; i < m_DisplayList.size(); ++i)
    {
        outFile.write((const char*)m_DisplayList[i], sizeof(DisplayData));
    }

    outFile.close();

    return true;
}



bool AMDGenlockConfigManager::loadConfiguration(const char* pFileName)
{
    char                    cBuffer[256];
    unsigned int            uiNumGpus;
    unsigned int            uiNumDisplays;
    unsigned int            uiNumTC;
    unsigned int            uiGlobals[3];
    fstream                 inFile;

    // Make a vaild display configuration exists 
    if (m_DisplayList.size() == 0)
    {
        outputLog(LogOutput::MSG_ERROR, "Failed to resore config from file. No currrent displays present.");
        return false;
    }

    inFile.open(pFileName, fstream::in | fstream::binary);

    if (!inFile.is_open())
    {
        sprintf_s(cBuffer, "Failed to resore config from file. File %s not found", pFileName);
        outputLog(LogOutput::MSG_ERROR, cBuffer);

        return false;
    }

    // read globals

    // read Number of Displays (unsigned int) Idx of TS (unsigned int) Number of TCs (unsigned int)
    inFile.read((char*)uiGlobals, 3 * sizeof(int));

    uiNumGpus       = uiGlobals[0];
    uiNumDisplays   = uiGlobals[1];
    uiNumTC         = uiGlobals[2];

    if (uiNumGpus != m_GpuList.size())
    {
        outputLog(LogOutput::MSG_ERROR, "Failed to resore config from file. The number of  GPUs do not match");

        inFile.close();
        return false;
    }
    
    // It is expected that the loaded configuration matches the current display config
    if (uiNumDisplays != m_DisplayList.size())
    {
        outputLog(LogOutput::MSG_ERROR, "Failed to resore config from file. The number of  displays do not match");

        inFile.close();
        return false;
    }
    
    // Read genlock configuratuion for each GPU
    for (unsigned int i = 0; i < uiNumGpus; ++i)
    {   
        unsigned int    j;
        GpuData         gpuData;

        inFile.read((char *)&gpuData, sizeof(GpuData));

        for (j = 0; j <  m_GpuList.size(); ++j)
        {
            if ((m_GpuList[j]->uiBusId == gpuData.uiBusId) && (m_GpuList[j]->uiFirstAdapterId == gpuData.uiFirstAdapterId))
            {
                memcpy(&(m_GpuList[j]->adlGlSyncGenlockConfig), &(gpuData.adlGlSyncGenlockConfig), sizeof(ADLGLSyncGenlockConfig));
                break;
            }
        }

        if (j == m_GpuList.size())
        {
            // Error, no matching gpu found
            sprintf_s(cBuffer,"Failed to find a mathcing GPU for Bus ID: %d, Adapter Idx: %d.", gpuData.uiBusId, gpuData.uiFirstAdapterId);
            outputLog(LogOutput::MSG_ERROR, cBuffer);

            inFile.close();
            return false;
        }
    }

    // Read port info of BNC, RJ45 1 and RJ45 2
    inFile.read((char *)m_adlGLSyncPortInfo, 3 * sizeof(ADLGlSyncPortInfo));

    // Clear list of TC ids, since this list will be generated when the display inf ois read
    m_TCList.clear();

    // Read per display information
    DisplayData DspData;

    for (unsigned int i = 0; i < uiNumDisplays; ++i)
    {
        unsigned int j = 0;

        inFile.read((char *)&DspData, sizeof(DisplayData));

        // find corresponding display in list of current displays
        for (j = 0; j < m_DisplayList.size(); ++j)
        {
            if (m_DisplayList[j]->uiAdapterId == DspData.uiAdapterId && m_DisplayList[j]->uiDisplayId == DspData.uiDisplayId)
            {
                // update Genlock data on current display based on teh read information
                memcpy(&(m_DisplayList[j]->adlGLSyncMode), &DspData.adlGLSyncMode, sizeof(ADLGlSyncMode));

                if (m_DisplayList[j]->adlGLSyncMode.iStatusVector & ADL_GLSYNC_MODECNTL_STATUS_GENLOCK)
                {
                    if (m_DisplayList[j]->adlGLSyncMode.iControlVector & ADL_GLSYNC_MODECNTL_TIMINGSERVER)
                    {
                        m_uiTSIdx       = j;
                        m_bTimingServer = true;
                    }
                    else
                    {
                        m_TCList.push_back(j);
                    }
                }
                // we found the matching adapter, quit loop
                break;
            }
        } // for (j =

        if (j == m_DisplayList.size())
        {
            // The displayed data read  from file did not match any of the current displays. A no display with the same
            // Adapter Index and Display Index was found. This might happen if teh display configuration was changed by 
            // adding and/or removing displays.
            sprintf_s(cBuffer,"Failed to find a mathcing display for Adapter Idx: %d, Display Idx: %d. The monitor configuration changed.", DspData.uiAdapterId, DspData.uiDisplayId);
            outputLog(LogOutput::MSG_ERROR, cBuffer);

            inFile.close();
            return false;
        }
    } // for (i = 

    if (inFile.fail())
    {
        outputLog(LogOutput::MSG_ERROR, "Failed to resore config from file. Read error");

        inFile.close();
        return false;
    }

    inFile.close();

    return true;
}


unsigned int AMDGenlockConfigManager::getSignalSource(unsigned int uiGpuId)
{
    if (uiGpuId >= m_GpuList.size())
    {
        return 3;
    }

    if (m_GpuList[uiGpuId]->adlGlSyncGenlockConfig.iSignalSource == ADL_GLSYNC_SIGNALSOURCE_BNCPORT)
    {
        return 0;
    }
    else if (m_GpuList[uiGpuId]->adlGlSyncGenlockConfig.iSignalSource == ADL_GLSYNC_SIGNALSOURCE_RJ45PORT1)
    {
        return 1;
    }
    else if (m_GpuList[uiGpuId]->adlGlSyncGenlockConfig.iSignalSource == ADL_GLSYNC_SIGNALSOURCE_RJ45PORT2)
    {
        return 2;
    }

    return 3;
}


bool AMDGenlockConfigManager::connectedToS400(unsigned int uiGpuId)
{
    if (uiGpuId >= m_GpuList.size())
    {
        return false;
    }

    return (m_GpuList[uiGpuId]->adlGlSyncModule.iModuleID != 0);
}


bool AMDGenlockConfigManager::getS400Info(unsigned int uiGpuId, unsigned int &uiModuleid, unsigned int &uiFWVer)
{
    if (uiGpuId >= m_GpuList.size())
    {
        return false;
    }

    uiModuleid = m_GpuList[uiGpuId]->adlGlSyncModule.iModuleID;
    uiFWVer    = m_GpuList[uiGpuId]->adlGlSyncModule.iFWUserSectorVersion;

    // If a S400 is connected the module id is > 0
    return (uiModuleid > 0);
}


unsigned int AMDGenlockConfigManager::readSerialId(unsigned int uiAdpaterId, unsigned int uiDisplayId)
{
    unsigned int        uiSerialId;
    ADLDisplayEDIDData  adlEdidData;

    memset(&adlEdidData, 0,   sizeof(ADLDisplayEDIDData));
    adlEdidData.iSize       = sizeof(ADLDisplayEDIDData);
    adlEdidData.iBlockIndex = 0; //First block

    if (g_AdlCalls.ADL_Display_EdidData_Get(uiAdpaterId, uiDisplayId, &adlEdidData) == ADL_OK)
    {
        // extract serial number byte 12-15 (little endian). 
        uiSerialId  =  (unsigned int)adlEdidData.cEDIDData[12];
        uiSerialId += ((unsigned int)adlEdidData.cEDIDData[13] <<  8);
        uiSerialId += ((unsigned int)adlEdidData.cEDIDData[14] << 16);
        uiSerialId += ((unsigned int)adlEdidData.cEDIDData[15] << 24);

        return uiSerialId;
    }

    return 0;
}


bool AMDGenlockConfigManager::setupADL()
{
    // check if ADL was already loaded
    if (g_AdlCalls.hModule)
    {
        return true;
    }

#if defined(WIN32)

    g_AdlCalls.hModule = (void*)LoadLibrary("atiadlxx.dll");

	if (g_AdlCalls.hModule == NULL)
       g_AdlCalls.hModule = (void*)LoadLibrary("atiadlxy.dll");

#elif defined(LINUX)

    g_AdlCalls.hModule = dlopen("libatiadlxx.so", RTLD_LAZY|RTLD_GLOBAL);

    if (g_AdlCalls.hModule == NULL)
        g_AdlCalls.hModule = dlopen("libatiadlxy.so", RTLD_LAZY|RTLD_GLOBAL);

#endif


	if (!g_AdlCalls.hModule)
		return false;

	// Get proc address of needed ADL functions
	g_AdlCalls.ADL_Main_Control_Create = (ADL_MAIN_CONTROL_CREATE)GETPROC((HMODULE)g_AdlCalls.hModule,"ADL_Main_Control_Create");
	if (!g_AdlCalls.ADL_Main_Control_Create)
		return false;

	g_AdlCalls.ADL_Main_Control_Destroy = (ADL_MAIN_CONTROL_DESTROY)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Main_Control_Destroy");
	if (!g_AdlCalls.ADL_Main_Control_Destroy)
		return false;

    g_AdlCalls.ADL_Adapter_NumberOfAdapters_Get = (ADL_ADAPTER_NUMBEROFADAPTERS_GET)GETPROC((HMODULE)g_AdlCalls.hModule,"ADL_Adapter_NumberOfAdapters_Get");
	if (!g_AdlCalls.ADL_Adapter_NumberOfAdapters_Get)
		return false;

	g_AdlCalls.ADL_Adapter_AdapterInfo_Get = (ADL_ADAPTER_ADAPTERINFO_GET)GETPROC((HMODULE)g_AdlCalls.hModule,"ADL_Adapter_AdapterInfo_Get");
	if (!g_AdlCalls.ADL_Adapter_AdapterInfo_Get)
		return false;

	g_AdlCalls.ADL_Display_DisplayInfo_Get = (ADL_DISPLAY_DISPLAYINFO_GET)GETPROC((HMODULE)g_AdlCalls.hModule,"ADL_Display_DisplayInfo_Get");
	if (!g_AdlCalls.ADL_Display_DisplayInfo_Get)
		return false;

	g_AdlCalls.ADL_Adapter_Active_Get = (ADL_ADAPTER_ACTIVE_GET)GETPROC((HMODULE)g_AdlCalls.hModule,"ADL_Adapter_Active_Get");
	if (!g_AdlCalls.ADL_Adapter_Active_Get)
		return false;

	g_AdlCalls.ADL_Display_Position_Get = (ADL_DISPLAY_POSITION_GET)GETPROC((HMODULE)g_AdlCalls.hModule,"ADL_Display_Position_Get");
	if (!g_AdlCalls.ADL_Display_Position_Get)
		return false;	

    g_AdlCalls.ADL_Display_EdidData_Get = (ADL_DISPLAY_EDIDDATA_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Display_EdidData_Get");
    if (!g_AdlCalls.ADL_Display_EdidData_Get)
		return false;	

    g_AdlCalls.ADL_Workstation_DisplayGenlockCapable_Get = (ADL_WORKSTATION_DISPLAYGENLOCKCAPABLE_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_DisplayGenlockCapable_Get");
	if (!g_AdlCalls.ADL_Workstation_DisplayGenlockCapable_Get)
		return false;

    g_AdlCalls.ADL_Workstation_GLSyncModuleDetect_Get = (ADL_WORKSTATION_GLSYNCMODULEDETECT_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_GLSyncModuleDetect_Get");
	if (!g_AdlCalls.ADL_Workstation_GLSyncModuleDetect_Get)
		return false;
   
    g_AdlCalls.ADL_Workstation_GLSyncModuleInfo_Get = (ADL_WORKSTATION_GLSYNCMODULEINFO_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_GLSyncModuleInfo_Get");
	if (!g_AdlCalls.ADL_Workstation_GLSyncModuleInfo_Get)
		return false;

    g_AdlCalls.ADL_Workstation_GLSyncGenlockConfiguration_Get = (ADL_WORKSTATION_GLSYNCGENLOCKCONFIGURATION_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_GLSyncGenlockConfiguration_Get");
	if (!g_AdlCalls.ADL_Workstation_GLSyncGenlockConfiguration_Get)
		return false;

    g_AdlCalls.ADL_Workstation_GLSyncGenlockConfiguration_Set = (ADL_WORKSTATION_GLSYNCGENLOCKCONFIGURATION_SET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_GLSyncGenlockConfiguration_Set");
	if (!g_AdlCalls.ADL_Workstation_GLSyncGenlockConfiguration_Set)
		return false;
               
    g_AdlCalls.ADL_Workstation_GLSyncPortState_Get = (ADL_WORKSTATION_GLSYNCPORTSTATE_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_GLSyncPortState_Get");
	if (!g_AdlCalls.ADL_Workstation_GLSyncPortState_Get)
		return false;
     
    g_AdlCalls.ADL_Workstation_GLSyncPortState_Set = (ADL_WORKSTATION_GLSYNCPORTSTATE_SET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_GLSyncPortState_Set");
	if (!g_AdlCalls.ADL_Workstation_GLSyncPortState_Set)
		return false;

    g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Get = (ADL_WORKSTATION_DISPLAYGLSYNCMODE_GET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_DisplayGLSyncMode_Get");
	if (!g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Get)
		return false;

    g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Set = (ADL_WORKSTATION_DISPLAYGLSYNCMODE_SET)GETPROC((HMODULE)g_AdlCalls.hModule, "ADL_Workstation_DisplayGLSyncMode_Set");
	if (!g_AdlCalls.ADL_Workstation_DisplayGLSyncMode_Set)
		return false;


	// Init ADL
	if (g_AdlCalls.ADL_Main_Control_Create(ADL_Alloc, 1) != ADL_OK)
		return false;

	return true;
}

void AMDGenlockConfigManager::outputLog(unsigned int uiMT, const char* pMessage)
{
    if (m_pLog)
    {
         m_pLog->print((LogOutput::MessageType)uiMT, pMessage);
    }
}
