//=====================================================================
// Copyright (c) 2020-2024    Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
//=====================================================================
#ifndef BC7_ENCODE_KERNEL_H
#define BC7_ENCODE_KERNEL_H

#pragma warning(disable : 4505)  // disable warnings on unreferenced local function has been removed

#include "common_def.h"

#define USE_VARYING

#ifndef ASPM_GPU

struct cmp_bc7_state
{
    CGV_FLOAT  block[16][4];
    CGV_UINT32 best_data[4];
};

typedef enum
{
    CGU_FORMAT_Unknown,  // Undefined texture format.

    // Channel Component formats-------------------------------------------------------------------------------
    CGU_FORMAT_RGBA_8888,  // RGBA format with 8-bit fixed channels.

    // Formats supported by GPU
    CGU_FORMAT_BC1,   // A four component opaque (or 1-bit alpha) compressed texture format for Microsoft DirectX10. Identical to DXT1.  Four bits per pixel.
    CGU_FORMAT_BC6H,  // BC6H compressed texture format
    CGU_FORMAT_BC7,   // BC7  compressed texture format

    // Formats supported by CPU
    CGU_FORMAT_GTC,  // GTC Gradient Texture Compressor
    CGU_FORMAT_APC,  // APC Texture Compressor
    CGU_FORMAT_MAX
} CGU_FORMAT;

//------------------------------------
// The structure describing a texture
//------------------------------------
struct CGU_Texture_Type
{
    // Optional Settings
    CGU_FLOAT m_fquality;      // Minimum resulting quality to maintain while processing the texture, default is 0.05
    CGU_INT8  m_nBlockHeight;  // Size of the texture tiles (blocks) to use.during processing
    CGU_INT8  m_nBlockWidth;   // default = 4
    CGU_INT8  m_nBlockDepth;   // default = 1

    // Required settings
    CGU_FORMAT m_format;      // Texture format
    CGU_UINT32 m_src_width;   // Width of the texture.
    CGU_UINT32 m_src_height;  // Height of the texture.
    CGU_UINT32 m_stride;      // Number of bytes to start of next line
    CGU_UINT32 m_dwDataSize;  // Size of the allocated texture data.
    CGU_UINT8* m_pData;       // Pointer to the texture data
};

#endif  // End of ASPM_CPU

#define SOURCE_BLOCK_SIZE 16      // Size of a source block in pixels (each pixel has RGBA:8888 channels)
#define COMPRESSED_BLOCK_SIZE 16  // Size of a compressed block in bytes
#define MAX_CHANNELS 4
#define MAX_SUBSETS 3       // Maximum number of possible subsets
#define MAX_SUBSET_SIZE 16  // Largest possible size for an individual subset
#define BC7_qFAST_THRESHOLD 0.50f
#define MAX_INDEX_BITS 4  // Maximum number of index bits

typedef struct
{
    CGV_FLOAT image;
    CGV_UINT8 index;
} CMP_di;

typedef struct
{
    CGV_FLOAT image;
    CGV_UINT8 index;
} CMP_du;

#define MAX_PARTITION_ENTRIES 64

#define MAX_PARTITIONS_TABLE 193

#define MAX_PARTITIONS 64  // Maximum number of partition types

#define EPSILON 0.00390625f
#define DIMENSION 4
#define BlockX 4
#define BlockY 4
#define QUANT_RT 250.0f  // quality = 0.05f
//==========================================================================================================
#define LOG_CL_RANGE 5
#define LOG_CL_BASE 2
#define BIT_BASE 5
#define BIT_RANGE 9
#define MAX_CLUSTERS_BIG 16
#define MAX_CLUSTERS 8
#define BTT(bits) (bits - BIT_BASE)
#define CLT(cl) (cl - LOG_CL_BASE)

#define MAX_TRY_QUANT_TRACE 2  // used in optQuantTrace_d : increasing this has no gain in quality!, keep it set at 2
#define NUM_BLOCK_TYPES 8      // Number of block types in the format

#define BC7_MAX_TRACE 25000

// If this is defined, ramp calculation is done via math floor and division.
// Otherwise, ramp calculation is done by bit shifting
#define USE_HIGH_PRECISION_INTERPOLATION_BC7

typedef struct
{
    CGU_INT32 k;
    CGV_FLOAT d;
} TRACE;

typedef struct
#ifdef ASPM
    BC7_EncodeState
#endif
{
    CGV_FLOAT image_src[64];
    CGV_UINT8 cmp_out[COMPRESSED_BLOCK_SIZE];

    // Common
    CGV_FLOAT opaque_err;  // error for coding alpha=255
    CGV_FLOAT best_err;

    // set per mode
    CGU_UINT8 channels3or4;
    CGU_UINT8 bits;
    CGU_INT   clusters;
    CGU_UINT8 componentBits;
    CGU_UINT8 numPartitionModes;
    CGU_INT   maxSubSets;
    CGU_UINT8 numClusters0[2];
    CGU_UINT8 numClusters1[2];
    CGU_UINT8 max_idxMode;
    CGU_INT   modeBits[2];
    CGU_BOOL  optimizedQ;

    CGU_UINT32 validModeMask;
    CGU_INT    part_count;
    CGU_UINT8  channels;

    // use_icmp
    CGV_UINT32 best_cmp_out[5];
    CGV_BOOL   cmp_isout16Bytes;
    CGU_INT    refineIterations;
    CGU_INT    fastSkipTreshold;
}
#ifndef ASPM
BC7_EncodeState
#endif
    ;

typedef struct
#ifdef ASPM
    cmp_mode_parameters
#endif
{
    CGV_INT    color_qendpoint[8];
    CGV_INT    alpha_qendpoint[8];
    CGV_UINT32 best_color_index[2];
    CGV_UINT32 best_alpha_index[2];
    CGV_UINT8  color_index[SOURCE_BLOCK_SIZE];
    CGV_UINT8  alpha_index[SOURCE_BLOCK_SIZE];
    CGV_UINT32 idxMode;
    CGV_UINT32 rotated_channel;
}
#ifndef ASPM
cmp_mode_parameters
#endif
    ;

typedef struct
#ifdef ASPM
    BC7_Encode
#endif
{
    // Global data setup at initialization time
    CGU_FLOAT  quality;          // range is 0 to 1
    CGU_FLOAT  errorThreshold;   // use 5 to 75
    CGU_UINT32 validModeMask;    // bit for mode masks def to 0xFF
    CGU_BOOL   imageNeedsAlpha;  // default: false
    CGU_BOOL   colourRestrict;   // default: false
    CGU_BOOL   alphaRestrict;    // default: false

    // Used to track errors in internal state code
    CGV_FLOAT opaque_err;
    CGV_FLOAT best_err;

    CGU_FLOAT minThreshold;
    CGU_FLOAT maxThreshold;
    ;

    // icmp code settings
    CGU_INT refineIterations;
    CGU_INT part_count;
    CGU_INT channels;
}
#ifndef ASPM
BC7_Encode
#endif
    ;

CMP_CONSTANT CGU_FLOAT rampWeights[5][SOURCE_BLOCK_SIZE] = {
    {0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f},  // 0 bit index
    {0.000000f,
     1.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f},  // 1 bit index
    {0.000000f,
     0.328125f,
     0.671875f,
     1.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f},  // 2 bit index
    {0.000000f,
     0.140625f,
     0.281250f,
     0.421875f,
     0.578125f,
     0.718750f,
     0.859375f,
     1.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f,
     0.000000f},  // 3 bit index
    {0.000000f,
     0.062500f,
     0.140625f,
     0.203125f,
     0.265625f,
     0.328125f,
     0.406250f,
     0.468750f,
     0.531250f,
     0.593750f,
     0.671875f,
     0.734375f,
     0.796875f,
     0.859375f,
     0.937500f,
     1.000000f}  // 4 bit index
};

#ifndef ASPM_GPU
typedef struct
#ifdef ASPM
    BC7_EncodeRamps
#endif
{
    CGU_INT ep_d[4][256];
#ifdef USE_BC7_SP_ERR_IDX
    CGU_UINT8 sp_err[3 * 4 * 256 * 2 * 2 * 16];
    CGU_INT   sp_idx[3 * 4 * 256 * 2 * 2 * 16 * 2];
#endif
#ifdef USE_BC7_RAMP
    CGU_FLOAT ramp[3 * 4 * 256 * 256 * 16];
#endif
    CGU_BOOL ramp_init;
}
#ifndef ASPM
BC7_EncodeRamps
#endif
    ;
#endif

CMP_CONSTANT CGU_UINT8 npv_nd[2][8] = {
    {1, 2, 4, 8, 16, 32, 0, 0},  // 3
    {1, 2, 4, 0, 0, 0, 0, 0}     // 4
};

typedef enum
{
    NO_ALPHA,
    COMBINED_ALPHA,
    SEPARATE_ALPHA
} CMP_BCE;

// Endpoint encoding type
typedef enum
{
    NO_PBIT,
    ONE_PBIT,
    TWO_PBIT,
    THREE_PBIT,
    FOUR_PBIT,
    FIVE_PBIT
} CMP_PBIT;

typedef struct
#ifdef ASPM
    BC7_Encode_local
#endif
{
    // Data for compressing a particular block mode
    CGV_INT   clusters[2];
    CGV_UINT8 parityBits;
    CGV_UINT8 componentBits[MAX_CHANNELS];

    CMP_BCE   encodingType;   // Type of block
    CGU_UINT8 partitionBits;  // Number of bits for partition data
    CGU_UINT8 rotationBits;   // Number of bits for component rotation
    CGU_UINT8 indexModeBits;  // Number of bits for index selection
    CMP_PBIT  pBitType;       // Type of P-bit encoding
    CGU_UINT8 subsetCount;    // Number of subsets
    CGU_UINT8 indexBits[2];   // Number of bits per index in each index set

    // Bulky temporary data used during compression of a block
    CGV_UINT8 storedindex[MAX_PARTITIONS][MAX_SUBSETS][MAX_SUBSET_SIZE];
    CGV_FLOAT storedError[MAX_PARTITIONS];
    CGV_UINT8 sortedModes[MAX_PARTITIONS];

    // This stores the min and max for the components of the block, and the ranges
    CGV_FLOAT blockMin[MAX_CHANNELS];
    CGV_FLOAT blockMax[MAX_CHANNELS];
    CGV_FLOAT blockRange[MAX_CHANNELS];
    CGV_FLOAT blockMaxRange;
}
#ifndef ASPM
BC7_Encode_local
#endif
    ;

typedef enum
{
    CART,
    SAME_PAR,
    BCC,
    SAME_FCC,
    FCC,
    FCC_SAME_BCC,
} CMP_qt;

// Block component encoding

// Descriptor structure for block encodings
typedef struct
{
    CMP_UNIFORM CMP_BCE  encodingType;   // Type of block
    CGU_UINT8            partitionBits;  // Number of bits for partition data
    CGU_UINT8            rotationBits;   // Number of bits for component rotation
    CGU_UINT8            indexModeBits;  // Number of bits for index selection
    CGU_UINT8            scalarBits;     // Number of bits for one scalar endpoint
    CGU_UINT8            vectorBits;     // Number of bits for one vector endpoint(excluding P bits)
    CMP_UNIFORM CMP_PBIT pBitType;       // Type of P-bit encoding
    CGU_UINT8            subsetCount;    // Number of subsets
    CGU_UINT8            indexBits[2];   // Number of bits per index in each index set
} CMP_BTI;

typedef enum
{
    COMP_RED   = 0,
    COMP_GREEN = 1,
    COMP_BLUE  = 2,
    COMP_ALPHA = 3
} COMPONENT;

CMP_CONSTANT CGU_UINT8 componentRotations[4][4] = {{COMP_ALPHA, COMP_RED, COMP_GREEN, COMP_BLUE},
                                                   {COMP_RED, COMP_ALPHA, COMP_GREEN, COMP_BLUE},
                                                   {COMP_GREEN, COMP_RED, COMP_ALPHA, COMP_BLUE},
                                                   {COMP_BLUE, COMP_RED, COMP_GREEN, COMP_ALPHA}};

CMP_CONSTANT CMP_BTI bti[NUM_BLOCK_TYPES] = {
    //encodingType,partitionBits,rotationBits,indexModeBits,scalarBits,vectorBits,pBitType,     subsetCount,indexBits[0]&[1]
    {NO_ALPHA, 4, 0, 0, 0, 12, TWO_PBIT, 3, {3, 0}},        // Format Mode 0
    {NO_ALPHA, 6, 0, 0, 0, 18, ONE_PBIT, 2, {3, 0}},        // Format Mode 1
    {NO_ALPHA, 6, 0, 0, 0, 15, NO_PBIT, 3, {2, 0}},         // Format Mode 2
    {NO_ALPHA, 6, 0, 0, 0, 21, TWO_PBIT, 2, {2, 0}},        // Format Mode 3
    {SEPARATE_ALPHA, 0, 2, 1, 6, 15, NO_PBIT, 1, {2, 3}},   // Format Mode 4
    {SEPARATE_ALPHA, 0, 2, 0, 8, 21, NO_PBIT, 1, {2, 2}},   // Format Mode 5
    {COMBINED_ALPHA, 0, 0, 0, 0, 28, TWO_PBIT, 1, {4, 0}},  // Format Mode 6
    {COMBINED_ALPHA, 6, 0, 0, 0, 20, TWO_PBIT, 2, {2, 0}}   // Format Mode 7
};

CMP_CONSTANT CGU_UINT8 par_vectors_nd[2][8][64][2][4] = {
    {
        // 3D
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{1, 1, 1, 0}, {1, 1, 1, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {1, 1, 1, 0}}, {{1, 1, 1, 0}, {0, 0, 0, 0}}, {{1, 1, 1, 0}, {1, 1, 1, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{1, 1, 0, 0}, {1, 1, 0, 0}}, {{1, 0, 1, 0}, {1, 0, 1, 0}}, {{0, 1, 1, 0}, {0, 1, 1, 0}}, {{0, 0, 0, 0}, {1, 1, 1, 0}},
         {{1, 1, 1, 0}, {0, 0, 0, 0}}, {{0, 1, 0, 0}, {0, 1, 0, 0}}, {{1, 1, 1, 0}, {1, 1, 1, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{1, 1, 0, 0}, {0, 0, 0, 0}}, {{1, 0, 1, 0}, {0, 0, 0, 0}}, {{0, 1, 1, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {1, 1, 0, 0}},
         {{1, 1, 0, 0}, {1, 1, 0, 0}}, {{1, 0, 1, 0}, {1, 1, 0, 0}}, {{0, 1, 1, 0}, {1, 1, 0, 0}}, {{0, 0, 0, 0}, {1, 0, 1, 0}}, {{1, 1, 0, 0}, {1, 0, 1, 0}},
         {{1, 0, 1, 0}, {1, 0, 1, 0}}, {{0, 1, 1, 0}, {1, 0, 1, 0}}, {{0, 0, 0, 0}, {0, 1, 1, 0}}, {{1, 1, 0, 0}, {0, 1, 1, 0}}, {{1, 0, 1, 0}, {0, 1, 1, 0}},
         {{0, 1, 1, 0}, {0, 1, 1, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{1, 1, 0, 0}, {0, 0, 0, 0}}, {{1, 0, 1, 0}, {0, 0, 0, 0}}, {{0, 1, 1, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {1, 1, 0, 0}},
         {{1, 1, 0, 0}, {1, 1, 0, 0}}, {{1, 0, 1, 0}, {1, 1, 0, 0}}, {{0, 1, 1, 0}, {1, 1, 0, 0}}, {{0, 0, 0, 0}, {1, 0, 1, 0}}, {{1, 1, 0, 0}, {1, 0, 1, 0}},
         {{1, 0, 1, 0}, {1, 0, 1, 0}}, {{0, 1, 1, 0}, {1, 0, 1, 0}}, {{0, 0, 0, 0}, {0, 1, 1, 0}}, {{1, 1, 0, 0}, {0, 1, 1, 0}}, {{1, 0, 1, 0}, {0, 1, 1, 0}},
         {{0, 1, 1, 0}, {0, 1, 1, 0}}, {{1, 0, 0, 0}, {1, 1, 1, 0}}, {{0, 1, 0, 0}, {1, 1, 1, 0}}, {{0, 0, 1, 0}, {1, 1, 1, 0}}, {{1, 1, 1, 0}, {1, 1, 1, 0}},
         {{1, 0, 0, 0}, {0, 0, 1, 0}}, {{0, 1, 0, 0}, {0, 0, 1, 0}}, {{0, 0, 1, 0}, {0, 0, 1, 0}}, {{1, 1, 1, 0}, {0, 0, 1, 0}}, {{1, 0, 0, 0}, {1, 0, 0, 0}},
         {{0, 1, 0, 0}, {1, 0, 0, 0}}, {{0, 0, 1, 0}, {1, 0, 0, 0}}, {{1, 1, 1, 0}, {1, 0, 0, 0}}, {{1, 0, 0, 0}, {0, 1, 0, 0}}, {{0, 1, 0, 0}, {0, 1, 0, 0}},
         {{0, 0, 1, 0}, {0, 1, 0, 0}}, {{1, 1, 1, 0}, {0, 1, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
    },
    {
        // 4D
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{1, 1, 1, 1}, {1, 1, 1, 1}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {1, 1, 1, 1}}, {{1, 1, 1, 1}, {0, 0, 0, 0}}, {{1, 1, 1, 1}, {1, 1, 1, 1}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 1, 1, 1}}, {{0, 1, 1, 1}, {0, 0, 0, 0}}, {{0, 1, 1, 1}, {0, 1, 1, 1}}, {{1, 0, 0, 0}, {1, 0, 0, 0}},
         {{1, 0, 0, 0}, {1, 1, 1, 1}}, {{1, 1, 1, 1}, {1, 0, 0, 0}}, {{1, 1, 1, 1}, {1, 1, 1, 1}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 1, 1, 1}}, {{0, 1, 1, 1}, {0, 0, 0, 0}}, {{0, 1, 1, 1}, {0, 1, 1, 1}}, {{1, 0, 0, 0}, {1, 0, 0, 0}},
         {{1, 0, 0, 0}, {1, 1, 1, 1}}, {{1, 1, 1, 1}, {1, 0, 0, 0}}, {{1, 1, 1, 1}, {1, 1, 1, 1}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 1, 1}},
         {{0, 0, 1, 1}, {0, 0, 0, 0}}, {{0, 1, 0, 1}, {0, 1, 0, 1}}, {{1, 0, 0, 0}, {1, 0, 0, 0}}, {{1, 0, 0, 0}, {1, 0, 1, 1}}, {{1, 0, 1, 1}, {1, 0, 0, 0}},
         {{1, 1, 0, 1}, {1, 1, 0, 1}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
        {{{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}},
         {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}, {{0, 0, 0, 0}, {0, 0, 0, 0}}},
    },
};

CMP_CONSTANT CGU_UINT32 subset_mask_table[] = {
    // 2 subset region patterns
    0x0000CCCCu,  // 0   1100 1100 1100 1100  (MSB..LSB)
    0x00008888u,  // 1   1000 1000 1000 1000
    0x0000EEEEu,  // 2   1110 1110 1110 1110
    0x0000ECC8u,  // 3   1110 1100 1100 1000
    0x0000C880u,  // 4   1100 1000 1000 0000
    0x0000FEECu,  // 5   1111 1110 1110 1100
    0x0000FEC8u,  // 6   1111 1110 1100 1000
    0x0000EC80u,  // 7   1110 1100 1000 0000
    0x0000C800u,  // 8   1100 1000 0000 0000
    0x0000FFECu,  // 9   1111 1111 1110 1100
    0x0000FE80u,  // 10  1111 1110 1000 0000
    0x0000E800u,  // 11  1110 1000 0000 0000
    0x0000FFE8u,  // 12  1111 1111 1110 1000
    0x0000FF00u,  // 13  1111 1111 0000 0000
    0x0000FFF0u,  // 14  1111 1111 1111 0000
    0x0000F000u,  // 15  1111 0000 0000 0000
    0x0000F710u,  // 16  1111 0111 0001 0000
    0x0000008Eu,  // 17  0000 0000 1000 1110
    0x00007100u,  // 18  0111 0001 0000 0000
    0x000008CEu,  // 19  0000 1000 1100 1110
    0x0000008Cu,  // 20  0000 0000 1000 1100
    0x00007310u,  // 21  0111 0011 0001 0000
    0x00003100u,  // 22  0011 0001 0000 0000
    0x00008CCEu,  // 23  1000 1100 1100 1110
    0x0000088Cu,  // 24  0000 1000 1000 1100
    0x00003110u,  // 25  0011 0001 0001 0000
    0x00006666u,  // 26  0110 0110 0110 0110
    0x0000366Cu,  // 27  0011 0110 0110 1100
    0x000017E8u,  // 28  0001 0111 1110 1000
    0x00000FF0u,  // 29  0000 1111 1111 0000
    0x0000718Eu,  // 30  0111 0001 1000 1110
    0x0000399Cu,  // 31  0011 1001 1001 1100
    0x0000AAAAu,  // 32  1010 1010 1010 1010
    0x0000F0F0u,  // 33  1111 0000 1111 0000
    0x00005A5Au,  // 34  0101 1010 0101 1010
    0x000033CCu,  // 35  0011 0011 1100 1100
    0x00003C3Cu,  // 36  0011 1100 0011 1100
    0x000055AAu,  // 37  0101 0101 1010 1010
    0x00009696u,  // 38  1001 0110 1001 0110
    0x0000A55Au,  // 39  1010 0101 0101 1010
    0x000073CEu,  // 40  0111 0011 1100 1110
    0x000013C8u,  // 41  0001 0011 1100 1000
    0x0000324Cu,  // 42  0011 0010 0100 1100
    0x00003BDCu,  // 43  0011 1011 1101 1100
    0x00006996u,  // 44  0110 1001 1001 0110
    0x0000C33Cu,  // 45  1100 0011 0011 1100
    0x00009966u,  // 46  1001 1001 0110 0110
    0x00000660u,  // 47  0000 0110 0110 0000
    0x00000272u,  // 48  0000 0010 0111 0010
    0x000004E4u,  // 49  0000 0100 1110 0100
    0x00004E40u,  // 50  0100 1110 0100 0000
    0x00002720u,  // 51  0010 0111 0010 0000
    0x0000C936u,  // 52  1100 1001 0011 0110
    0x0000936Cu,  // 53  1001 0011 0110 1100
    0x000039C6u,  // 54  0011 1001 1100 0110
    0x0000639Cu,  // 55  0110 0011 1001 1100
    0x00009336u,  // 56  1001 0011 0011 0110
    0x00009CC6u,  // 57  1001 1100 1100 0110
    0x0000817Eu,  // 58  1000 0001 0111 1110
    0x0000E718u,  // 59  1110 0111 0001 1000
    0x0000CCF0u,  // 60  1100 1100 1111 0000
    0x00000FCCu,  // 61  0000 1111 1100 1100
    0x00007744u,  // 62  0111 0111 0100 0100
    0x0000EE22u,  // 63  1110 1110 0010 0010
    // 3 Subset region patterns
    0xF60008CCu,  // 0    1111 0110 0000 0000 : 0000 1000 1100 1100 = 2222122011001100 (MSB...LSB)
    0x73008CC8u,  // 1    0111 0011 0000 0000 : 1000 1100 1100 1000 = 1222112211001000
    0x3310CC80u,  // 2    0011 0011 0001 0000 : 1100 1100 1000 0000 = 1122112210020000
    0x00CEEC00u,  // 3    0000 0000 1100 1110 : 1110 1100 0000 0000 = 1110110022002220
    0xCC003300u,  // 4    1100 1100 0000 0000 : 0011 0011 0000 0000 = 2211221100000000
    0xCC0000CCu,  // 5    1100 1100 0000 0000 : 0000 0000 1100 1100 = 2200220011001100
    0x00CCFF00u,  // 6    0000 0000 1100 1100 : 1111 1111 0000 0000 = 1111111122002200
    0x3300CCCCu,  // 7    0011 0011 0000 0000 : 1100 1100 1100 1100 = 1122112211001100
    0xF0000F00u,  // 8    1111 0000 0000 0000 : 0000 1111 0000 0000 = 2222111100000000
    0xF0000FF0u,  // 9    1111 0000 0000 0000 : 0000 1111 1111 0000 = 2222111111110000
    0xFF0000F0u,  // 10   1111 1111 0000 0000 : 0000 0000 1111 0000 = 2222222211110000
    0x88884444u,  // 11   1000 1000 1000 1000 : 0100 0100 0100 0100 = 2100210021002100
    0x88886666u,  // 12   1000 1000 1000 1000 : 0110 0110 0110 0110 = 2110211021102110
    0xCCCC2222u,  // 13   1100 1100 1100 1100 : 0010 0010 0010 0010 = 2210221022102210
    0xEC80136Cu,  // 14   1110 1100 1000 0000 : 0001 0011 0110 1100 = 2221221121101100
    0x7310008Cu,  // 15   0111 0011 0001 0000 : 0000 0000 1000 1100 = 0222002210021100
    0xC80036C8u,  // 16   1100 1000 0000 0000 : 0011 0110 1100 1000 = 2211211011001000
    0x310008CEu,  // 17   0011 0001 0000 0000 : 0000 1000 1100 1110 = 0022100211001110
    0xCCC03330u,  // 18   1100 1100 1100 0000 : 0011 0011 0011 0000 = 2211221122110000
    0x0CCCF000u,  // 19   0000 1100 1100 1100 : 1111 0000 0000 0000 = 1111220022002200
    0xEE0000EEu,  // 20   1110 1110 0000 0000 : 0000 0000 1110 1110 = 2220222011101110
    0x77008888u,  // 21   0111 0111 0000 0000 : 1000 1000 1000 1000 = 1222122210001000
    0xCC0022C0u,  // 22   1100 1100 0000 0000 : 0010 0010 1100 0000 = 2210221011000000
    0x33004430u,  // 23   0011 0011 0000 0000 : 0100 0100 0011 0000 = 0122012200110000
    0x00CC0C22u,  // 24   0000 0000 1100 1100 : 0000 1100 0010 0010 = 0000110022102210
    0xFC880344u,  // 25   1111 1100 1000 1000 : 0000 0011 0100 0100 = 2222221121002100
    0x06606996u,  // 26   0000 0110 0110 0000 : 0110 1001 1001 0110 = 0110122112210110
    0x66009960u,  // 27   0110 0110 0000 0000 : 1001 1001 0110 0000 = 1221122101100000
    0xC88C0330u,  // 28   1100 1000 1000 1100 : 0000 0011 0011 0000 = 2200201120112200
    0xF9000066u,  // 29   1111 1001 0000 0000 : 0000 0000 0110 0110 = 2222200201100110
    0x0CC0C22Cu,  // 30   0000 1100 1100 0000 : 1100 0010 0010 1100 = 1100221022101100
    0x73108C00u,  // 31   0111 0011 0001 0000 : 1000 1100 0000 0000 = 1222112200020000
    0xEC801300u,  // 32   1110 1100 1000 0000 : 0001 0011 0000 0000 = 2221221120000000
    0x08CEC400u,  // 33   0000 1000 1100 1110 : 1100 0100 0000 0000 = 1100210022002220
    0xEC80004Cu,  // 34   1110 1100 1000 0000 : 0000 0000 0100 1100 = 2220220021001100
    0x44442222u,  // 35   0100 0100 0100 0100 : 0010 0010 0010 0010 = 0210021002100210
    0x0F0000F0u,  // 36   0000 1111 0000 0000 : 0000 0000 1111 0000 = 0000222211110000
    0x49242492u,  // 37   0100 1001 0010 0100 : 0010 0100 1001 0010 = 0210210210210210
    0x42942942u,  // 38   0100 0010 1001 0100 : 0010 1001 0100 0010 = 0210102121020210
    0x0C30C30Cu,  // 39   0000 1100 0011 0000 : 1100 0011 0000 1100 = 1100221100221100
    0x03C0C03Cu,  // 40   0000 0011 1100 0000 : 1100 0000 0011 1100 = 1100002222111100
    0xFF0000AAu,  // 41   1111 1111 0000 0000 : 0000 0000 1010 1010 = 2222222210101010
    0x5500AA00u,  // 42   0101 0101 0000 0000 : 1010 1010 0000 0000 = 1212121200000000
    0xCCCC3030u,  // 43   1100 1100 1100 1100 : 0011 0000 0011 0000 = 2211220022112200
    0x0C0CC0C0u,  // 44   0000 1100 0000 1100 : 1100 0000 1100 0000 = 1100220011002200
    0x66669090u,  // 45   0110 0110 0110 0110 : 1001 0000 1001 0000 = 1221022012210220
    0x0FF0A00Au,  // 46   0000 1111 1111 0000 : 1010 0000 0000 1010 = 1010222222221010
    0x5550AAA0u,  // 47   0101 0101 0101 0000 : 1010 1010 1010 0000 = 1212121212120000
    0xF0000AAAu,  // 48   1111 0000 0000 0000 : 0000 1010 1010 1010 = 2222101010101010
    0x0E0EE0E0u,  // 49   0000 1110 0000 1110 : 1110 0000 1110 0000 = 1110222011102220
    0x88887070u,  // 50   1000 1000 1000 1000 : 0111 0000 0111 0000 = 2111200021112000
    0x99906660u,  // 51   1001 1001 1001 0000 : 0110 0110 0110 0000 = 2112211221120000
    0xE00E0EE0u,  // 52   1110 0000 0000 1110 : 0000 1110 1110 0000 = 2220111011102220
    0x88880770u,  // 53   1000 1000 1000 1000 : 0000 0111 0111 0000 = 2000211121112000
    0xF0000666u,  // 54   1111 0000 0000 0000 : 0000 0110 0110 0110 = 2222011001100110
    0x99006600u,  // 55   1001 1001 0000 0000 : 0110 0110 0000 0000 = 2112211200000000
    0xFF000066u,  // 56   1111 1111 0000 0000 : 0000 0000 0110 0110 = 2222222201100110
    0xC00C0CC0u,  // 57   1100 0000 0000 1100 : 0000 1100 1100 0000 = 2200110011002200
    0xCCCC0330u,  // 58   1100 1100 1100 1100 : 0000 0011 0011 0000 = 2200221122112200
    0x90006000u,  // 59   1001 0000 0000 0000 : 0110 0000 0000 0000 = 2112000000000000
    0x08088080u,  // 60   0000 1000 0000 1000 : 1000 0000 1000 0000 = 1000200010002000
    0xEEEE1010u,  // 61   1110 1110 1110 1110 : 0001 0000 0001 0000 = 2221222022212220
    0xFFF0000Au,  // 62   1111 1111 1111 0000 : 0000 0000 0000 1010 = 2222222222221010
    0x731008CEu,  // 63   0111 0011 0001 0000 : 0000 1000 1100 1110 = 0222102211021110
};

CMP_CONSTANT CGV_INT rampI[5 * SOURCE_BLOCK_SIZE] = {
    0, 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  // 0 bit index
    0, 64, 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  // 1 bit index
    0, 21, 43, 64, 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  // 2 bit index
    0, 9,  18, 27, 37, 46, 55, 64, 0,  0,  0,  0,  0,  0,  0,  0,  // 3 bit index
    0, 4,  9,  13, 17, 21, 26, 30, 34, 38, 43, 47, 51, 55, 60, 64  // 4 bit index
};

// same as  CMP SDK v3.1 BC7_FIXUPINDEX1 &  BC7_FIXUPINDEX2 for each partition range 0..63
// The data is saved as a packed INT = (BC7_FIXUPINDEX1 << 4 + BC7_FIXUPINDEX2)
CMP_CONSTANT CGV_UINT32 FIXUPINDEX[] = {
    // 2 subset partitions 0..63
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0x20u,
    0x80u,
    0x20u,
    0x20u,
    0x80u,
    0x80u,
    0xf0u,
    0x20u,
    0x80u,
    0x20u,
    0x20u,
    0x80u,
    0x80u,
    0x20u,
    0x20u,
    0xf0u,
    0xf0u,
    0x60u,
    0x80u,
    0x20u,
    0x80u,
    0xf0u,
    0xf0u,
    0x20u,
    0x80u,
    0x20u,
    0x20u,
    0x20u,
    0xf0u,
    0xf0u,
    0x60u,
    0x60u,
    0x20u,
    0x60u,
    0x80u,
    0xf0u,
    0xf0u,
    0x20u,
    0x20u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0xf0u,
    0x20u,
    0x20u,
    0xf0u,
    // 3 subset partitions 64..128
    0x3fu,
    0x38u,
    0xf8u,
    0xf3u,
    0x8fu,
    0x3fu,
    0xf3u,
    0xf8u,
    0x8fu,
    0x8fu,
    0x6fu,
    0x6fu,
    0x6fu,
    0x5fu,
    0x3fu,
    0x38u,
    0x3fu,
    0x38u,
    0x8fu,
    0xf3u,
    0x3fu,
    0x38u,
    0x6fu,
    0xa8u,
    0x53u,
    0x8fu,
    0x86u,
    0x6au,
    0x8fu,
    0x5fu,
    0xfau,
    0xf8u,
    0x8fu,
    0xf3u,
    0x3fu,
    0x5au,
    0x6au,
    0xa8u,
    0x89u,
    0xfau,
    0xf6u,
    0x3fu,
    0xf8u,
    0x5fu,
    0xf3u,
    0xf6u,
    0xf6u,
    0xf8u,
    0x3fu,
    0xf3u,
    0x5fu,
    0x5fu,
    0x5fu,
    0x8fu,
    0x5fu,
    0xafu,
    0x5fu,
    0xafu,
    0x8fu,
    0xdfu,
    0xf3u,
    0xcfu,
    0x3fu,
    0x38u};

#ifndef ASPM_GPU
// =============================== USED BY DECODER THIS CODE NEEDS TO BE UPDATED =========================================
CMP_CONSTANT CGU_UINT32 BC7_FIXUPINDICES_LOCAL[MAX_SUBSETS][MAX_PARTITIONS][3] = {
    // One subset
    {
        {0, 0, 0},
    },

    {
        {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15},
        {0, 15}, {0, 2},  {0, 8},  {0, 2},  {0, 2},  {0, 8},  {0, 8},  {0, 15}, {0, 2},  {0, 8},  {0, 2},  {0, 2},  {0, 8},  {0, 8},  {0, 2},  {0, 2},
        {0, 15}, {0, 15}, {0, 6},  {0, 8},  {0, 2},  {0, 8},  {0, 15}, {0, 15}, {0, 2},  {0, 8},  {0, 2},  {0, 2},  {0, 2},  {0, 15}, {0, 15}, {0, 6},
        {0, 6},  {0, 2},  {0, 6},  {0, 8},  {0, 15}, {0, 15}, {0, 2},  {0, 2},  {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 15}, {0, 2},  {0, 2},  {0, 15},
    },

    // Three subsets
    {{0, 3, 15},  {0, 3, 8},  {0, 15, 8},  {0, 15, 3}, {0, 8, 15},  {0, 3, 15}, {0, 15, 3},  {0, 15, 8}, {0, 8, 15},  {0, 8, 15}, {0, 6, 15},
     {0, 6, 15},  {0, 6, 15}, {0, 5, 15},  {0, 3, 15}, {0, 3, 8},   {0, 3, 15}, {0, 3, 8},   {0, 8, 15}, {0, 15, 3},  {0, 3, 15}, {0, 3, 8},
     {0, 6, 15},  {0, 10, 8}, {0, 5, 3},   {0, 8, 15}, {0, 8, 6},   {0, 6, 10}, {0, 8, 15},  {0, 5, 15}, {0, 15, 10}, {0, 15, 8}, {0, 8, 15},
     {0, 15, 3},  {0, 3, 15}, {0, 5, 10},  {0, 6, 10}, {0, 10, 8},  {0, 8, 9},  {0, 15, 10}, {0, 15, 6}, {0, 3, 15},  {0, 15, 8}, {0, 5, 15},
     {0, 15, 3},  {0, 15, 6}, {0, 15, 6},  {0, 15, 8}, {0, 3, 15},  {0, 15, 3}, {0, 5, 15},  {0, 5, 15}, {0, 5, 15},  {0, 8, 15}, {0, 5, 15},
     {0, 10, 15}, {0, 5, 15}, {0, 10, 15}, {0, 8, 15}, {0, 13, 15}, {0, 15, 3}, {0, 12, 15}, {0, 3, 15}, {0, 3, 8}

    },

};

CMP_STATIC void SetDefaultBC7Options(BC7_Encode* BC7Encode)
{
    if (BC7Encode)
    {
        // Set for max quality
        BC7Encode->quality        = 1.0f;
        BC7Encode->minThreshold   = 5.0f;
        BC7Encode->maxThreshold   = 80.0f;
        BC7Encode->errorThreshold = 5.0f;
        BC7Encode->validModeMask  = 0xFF;

        BC7Encode->imageNeedsAlpha = FALSE;
        BC7Encode->colourRestrict  = FALSE;
        BC7Encode->alphaRestrict   = FALSE;

        BC7Encode->channels   = 4;
        BC7Encode->part_count = 128;
    }
}

#ifndef ASPM
//=====================
// Used by Decoder
//=====================
__constant CGU_FLOAT rampLerpWeightsBC7[5][16] = {
    {0.0},                                                                                    // 0 bit index
    {0.0, 1.0},                                                                               // 1 bit index
    {0.0, 21.0 / 64.0, 43.0 / 64.0, 1.0},                                                     // 2 bit index
    {0.0, 9.0 / 64.0, 18.0 / 64.0, 27.0 / 64.0, 37.0 / 64.0, 46.0 / 64.0, 55.0 / 64.0, 1.0},  // 3 bit index
    {0.0,
     4.0 / 64.0,
     9.0 / 64.0,
     13.0 / 64.0,
     17.0 / 64.0,
     21.0 / 64.0,
     26.0 / 64.0,
     30.0 / 64.0,
     34.0 / 64.0,
     38.0 / 64.0,
     43.0 / 64.0,
     47.0 / 64.0,
     51.0 / 64.0,
     55.0 / 64.0,
     60.0 / 64.0,
     1.0}  // 4 bit index
};

__constant CGU_UINT8 BC7_PARTITIONS[MAX_SUBSETS][MAX_PARTITIONS][MAX_SUBSET_SIZE] = {
    // Single subset partitions for both BC6H abd BC7
    {
        {
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
        },
    },

    {
        {// 0
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1},

        {// 1
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1},

        {// 2
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1},

        {// 3
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         1},

        {// 4
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1},

        {// 5
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 6
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 7
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         1},

        {// 8
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1},

        {// 9
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 10
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 11
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         1,
         1,
         1},

        {// 12
         0,
         0,
         0,
         1,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 13
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 14
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1},

        {// 15
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1},

        {// 16
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         1},

        {// 17
         0,
         1,
         1,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0},

        {// 18
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         1,
         1,
         0},

        {// 19
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         0},

        {// 20
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0},

        {// 21
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         0},

        {// 22
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         1,
         0,
         0},

        {// 23
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         1},

        {// 24
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         0},

        {// 25
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         1,
         0,
         0},

        {// 26
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0},

        {// 27
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         0,
         0},

        {// 28
         0,
         0,
         0,
         1,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         0,
         1,
         0,
         0,
         0},

        {// 29
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         1,
         0,
         0,
         0,
         0},

        {// 30
         0,
         1,
         1,
         1,
         0,
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         1,
         1,
         0},

        {// 31
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0},
        // -----------  BC7 only shapes from here on -------------
        {// 32
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         1},

        {// 33
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1},

        {// 34
         0,
         1,
         0,
         1,
         1,
         0,
         1,
         0,
         0,
         1,
         0,
         1,
         1,
         0,
         1,
         0},

        {// 35
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0},

        {// 36
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0},

        {// 37
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         0},

        {// 38
         0,
         1,
         1,
         0,
         1,
         0,
         0,
         1,
         0,
         1,
         1,
         0,
         1,
         0,
         0,
         1},

        {// 39
         0,
         1,
         0,
         1,
         1,
         0,
         1,
         0,
         1,
         0,
         1,
         0,
         0,
         1,
         0,
         1},

        {// 40
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         0},

        {// 41
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         0},

        {// 42
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         0,
         0},

        {// 43
         0,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         0,
         0},

        {// 44
         0,
         1,
         1,
         0,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         0,
         1,
         1,
         0},

        {// 45
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         0,
         1,
         1},

        {// 46
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1},

        {// 47
         0,
         0,
         0,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         0,
         0},

        {// 48
         0,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         0,
         0,
         0,
         0},

        {// 49
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         0,
         0,
         0},

        {// 50
         0,
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         0},

        {// 51
         0,
         0,
         0,
         0,
         0,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         0,
         0},

        {// 52
         0,
         1,
         1,
         0,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         1,
         0,
         0,
         1,
         1},

        {// 53
         0,
         0,
         1,
         1,
         0,
         1,
         1,
         0,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         1},

        {// 54
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0},

        {// 55
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         0,
         1,
         1,
         0},

        {// 56
         0,
         1,
         1,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         0,
         0,
         1},

        {// 57
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1},

        {// 58
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         0,
         1,
         0,
         0,
         0,
         0,
         0,
         0,
         1},

        {// 59
         0,
         0,
         0,
         1,
         1,
         0,
         0,
         0,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         1},

        {// 60
         0,
         0,
         0,
         0,
         1,
         1,
         1,
         1,
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1},

        {// 61
         0,
         0,
         1,
         1,
         0,
         0,
         1,
         1,
         1,
         1,
         1,
         1,
         0,
         0,
         0,
         0},

        {// 62
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1,
         0},

        {// 63
         0,
         1,
         0,
         0,
         0,
         1,
         0,
         0,
         0,
         1,
         1,
         1,
         0,
         1,
         1,
         1},
    },

    // Table.P3 - only for BC7

    {

        {0, 0, 1, 1, 0, 0, 1, 1, 0, 2, 2, 1, 2, 2, 2, 2},

        {0, 0, 0, 1, 0, 0, 1, 1, 2, 2, 1, 1, 2, 2, 2, 1},

        {0, 0, 0, 0, 2, 0, 0, 1, 2, 2, 1, 1, 2, 2, 1, 1},

        {0, 2, 2, 2, 0, 0, 2, 2, 0, 0, 1, 1, 0, 1, 1, 1},

        {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, 1, 1, 2, 2},

        {0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 2, 2, 0, 0, 2, 2},

        {0, 0, 2, 2, 0, 0, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1},

        {0, 0, 1, 1, 0, 0, 1, 1, 2, 2, 1, 1, 2, 2, 1, 1},

        {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2},

        {0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2},

        {0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2},

        {0, 0, 1, 2, 0, 0, 1, 2, 0, 0, 1, 2, 0, 0, 1, 2},

        {0, 1, 1, 2, 0, 1, 1, 2, 0, 1, 1, 2, 0, 1, 1, 2},

        {0, 1, 2, 2, 0, 1, 2, 2, 0, 1, 2, 2, 0, 1, 2, 2},

        {0, 0, 1, 1, 0, 1, 1, 2, 1, 1, 2, 2, 1, 2, 2, 2},

        {0, 0, 1, 1, 2, 0, 0, 1, 2, 2, 0, 0, 2, 2, 2, 0},

        {0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 2, 1, 1, 2, 2},

        {0, 1, 1, 1, 0, 0, 1, 1, 2, 0, 0, 1, 2, 2, 0, 0},

        {0, 0, 0, 0, 1, 1, 2, 2, 1, 1, 2, 2, 1, 1, 2, 2},

        {0, 0, 2, 2, 0, 0, 2, 2, 0, 0, 2, 2, 1, 1, 1, 1},

        {0, 1, 1, 1, 0, 1, 1, 1, 0, 2, 2, 2, 0, 2, 2, 2},

        {0, 0, 0, 1, 0, 0, 0, 1, 2, 2, 2, 1, 2, 2, 2, 1},

        {0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 2, 2, 0, 1, 2, 2},

        {0, 0, 0, 0, 1, 1, 0, 0, 2, 2, 1, 0, 2, 2, 1, 0},

        {0, 1, 2, 2, 0, 1, 2, 2, 0, 0, 1, 1, 0, 0, 0, 0},

        {0, 0, 1, 2, 0, 0, 1, 2, 1, 1, 2, 2, 2, 2, 2, 2},

        {0, 1, 1, 0, 1, 2, 2, 1, 1, 2, 2, 1, 0, 1, 1, 0},

        {0, 0, 0, 0, 0, 1, 1, 0, 1, 2, 2, 1, 1, 2, 2, 1},

        {0, 0, 2, 2, 1, 1, 0, 2, 1, 1, 0, 2, 0, 0, 2, 2},

        {0, 1, 1, 0, 0, 1, 1, 0, 2, 0, 0, 2, 2, 2, 2, 2},

        {0, 0, 1, 1, 0, 1, 2, 2, 0, 1, 2, 2, 0, 0, 1, 1},

        {0, 0, 0, 0, 2, 0, 0, 0, 2, 2, 1, 1, 2, 2, 2, 1},

        {0, 0, 0, 0, 0, 0, 0, 2, 1, 1, 2, 2, 1, 2, 2, 2},

        {0, 2, 2, 2, 0, 0, 2, 2, 0, 0, 1, 2, 0, 0, 1, 1},

        {0, 0, 1, 1, 0, 0, 1, 2, 0, 0, 2, 2, 0, 2, 2, 2},

        {0, 1, 2, 0, 0, 1, 2, 0, 0, 1, 2, 0, 0, 1, 2, 0},

        {0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 0, 0, 0, 0},

        {0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0},

        {0, 1, 2, 0, 2, 0, 1, 2, 1, 2, 0, 1, 0, 1, 2, 0},

        {0, 0, 1, 1, 2, 2, 0, 0, 1, 1, 2, 2, 0, 0, 1, 1},

        {0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 0, 0, 0, 0, 1, 1},

        {0, 1, 0, 1, 0, 1, 0, 1, 2, 2, 2, 2, 2, 2, 2, 2},

        {0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 2, 1, 2, 1, 2, 1},

        {0, 0, 2, 2, 1, 1, 2, 2, 0, 0, 2, 2, 1, 1, 2, 2},

        {0, 0, 2, 2, 0, 0, 1, 1, 0, 0, 2, 2, 0, 0, 1, 1},

        {0, 2, 2, 0, 1, 2, 2, 1, 0, 2, 2, 0, 1, 2, 2, 1},

        {0, 1, 0, 1, 2, 2, 2, 2, 2, 2, 2, 2, 0, 1, 0, 1},

        {0, 0, 0, 0, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1, 2, 1},

        {0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 2, 2, 2, 2},

        {0, 2, 2, 2, 0, 1, 1, 1, 0, 2, 2, 2, 0, 1, 1, 1},

        {0, 0, 0, 2, 1, 1, 1, 2, 0, 0, 0, 2, 1, 1, 1, 2},

        {0, 0, 0, 0, 2, 1, 1, 2, 2, 1, 1, 2, 2, 1, 1, 2},

        {0, 2, 2, 2, 0, 1, 1, 1, 0, 1, 1, 1, 0, 2, 2, 2},

        {0, 0, 0, 2, 1, 1, 1, 2, 1, 1, 1, 2, 0, 0, 0, 2},

        {0, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 0, 2, 2, 2, 2},

        {0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 1, 2, 2, 1, 1, 2},

        {0, 1, 1, 0, 0, 1, 1, 0, 2, 2, 2, 2, 2, 2, 2, 2},

        {0, 0, 2, 2, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 2, 2},

        {0, 0, 2, 2, 1, 1, 2, 2, 1, 1, 2, 2, 0, 0, 2, 2},

        {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 1, 1, 2},

        {0, 0, 0, 2, 0, 0, 0, 1, 0, 0, 0, 2, 0, 0, 0, 1},

        {0, 2, 2, 2, 1, 2, 2, 2, 0, 2, 2, 2, 1, 2, 2, 2},

        {0, 1, 0, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2},

        {0, 1, 1, 1, 2, 0, 1, 1, 2, 2, 0, 1, 2, 2, 2, 0},
    },
};
#endif

#endif  // !ASPM_GPU

#endif
