//===============================================================================
// Copyright (c) 2007-2024  Advanced Micro Devices, Inc. All rights reserved.
// Copyright (c) 2004-2006 ATI Technologies Inc.
//===============================================================================
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files(the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and / or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions :
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
//
//  File Name:   Codec_ETC.cpp
//  Description: implementation of the CCodec_ETC class
//
//////////////////////////////////////////////////////////////////////////////
#pragma warning(disable : 4100)

#include "common.h"
#include "codec_etc_rgb.h"
#include "compressonator_tc.h"
#include "etcpack_lib.h"

//////////////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////////////

CCodec_ETC_RGB::CCodec_ETC_RGB()
    : CCodec_ETC(CT_ETC_RGB)
{
}

CCodec_ETC_RGB::~CCodec_ETC_RGB()
{
}

CCodecBuffer* CCodec_ETC_RGB::CreateBuffer(CMP_BYTE  nBlockWidth,
                                           CMP_BYTE  nBlockHeight,
                                           CMP_BYTE  nBlockDepth,
                                           CMP_DWORD dwWidth,
                                           CMP_DWORD dwHeight,
                                           CMP_DWORD dwPitch,
                                           CMP_BYTE* pData,
                                           CMP_DWORD dwDataSize) const
{
    return CreateCodecBuffer(CBT_4x4Block_4BPP, 4, 4, 1, dwWidth, dwHeight, dwPitch, pData, dwDataSize);
}

CodecError CCodec_ETC_RGB::Compress(CCodecBuffer&       bufferIn,
                                    CCodecBuffer&       bufferOut,
                                    Codec_Feedback_Proc pFeedbackProc,
                                    CMP_DWORD_PTR       pUser1,
                                    CMP_DWORD_PTR       pUser2)
{
    assert(bufferIn.GetWidth() == bufferOut.GetWidth());
    assert(bufferIn.GetHeight() == bufferOut.GetHeight());

    if (bufferIn.GetWidth() != bufferOut.GetWidth() || bufferIn.GetHeight() != bufferOut.GetHeight())
        return CE_Unknown;

    readCompressParams();

    const CMP_DWORD dwBlocksX = ((bufferIn.GetWidth() + 3) >> 2);
    const CMP_DWORD dwBlocksY = ((bufferIn.GetHeight() + 3) >> 2);

    CMP_BYTE  srcBlock[BLOCK_SIZE_4X4X4];
    CMP_DWORD compressedBlock[2];
    for (CMP_DWORD j = 0; j < dwBlocksY; j++)
    {
        for (CMP_DWORD i = 0; i < dwBlocksX; i++)
        {
            bufferIn.ReadBlockRGBA(i * 4, j * 4, 4, 4, srcBlock);
            CompressRGBBlock(srcBlock, compressedBlock);
            bufferOut.WriteBlock(i * 4, j * 4, compressedBlock, 2);
        }
        if (pFeedbackProc)
        {
            float fProgress = 100.f * (j * dwBlocksX) / (dwBlocksX * dwBlocksY);
            if (pFeedbackProc(fProgress, pUser1, pUser2))
                return CE_Aborted;
        }
    }

    return CE_OK;
}

CodecError CCodec_ETC_RGB::Decompress(CCodecBuffer&       bufferIn,
                                      CCodecBuffer&       bufferOut,
                                      Codec_Feedback_Proc pFeedbackProc,
                                      CMP_DWORD_PTR       pUser1,
                                      CMP_DWORD_PTR       pUser2)
{
    assert(bufferIn.GetWidth() == bufferOut.GetWidth());
    assert(bufferIn.GetHeight() == bufferOut.GetHeight());

    if (bufferIn.GetWidth() != bufferOut.GetWidth() || bufferIn.GetHeight() != bufferOut.GetHeight())
        return CE_Unknown;

    const CMP_DWORD dwBlocksX  = ((bufferIn.GetWidth() + 3) >> 2);
    const CMP_DWORD dwBlocksY  = ((bufferIn.GetHeight() + 3) >> 2);
    const CMP_DWORD dwBlocksXY = dwBlocksX * dwBlocksY;

    CMP_DWORD compressedBlock[2];
    CMP_BYTE  destBlock[BLOCK_SIZE_4X4X4];
    for (CMP_DWORD j = 0; j < dwBlocksY; j++)
    {
        for (CMP_DWORD i = 0; i < dwBlocksX; i++)
        {
            bufferIn.ReadBlock(i * 4, j * 4, compressedBlock, 2);
            DecompressRGBBlock(destBlock, compressedBlock);
            bufferOut.WriteBlockRGBA(i * 4, j * 4, 4, 4, destBlock);
        }

        if (pFeedbackProc)
        {
            float fProgress = 100.f * (j * dwBlocksX) / dwBlocksXY;
            if (pFeedbackProc(fProgress, pUser1, pUser2))
            {
                return CE_Aborted;
            }
        }
    }

    return CE_OK;
}
