/*****************************************************************************\
*
*  Module Name    simple_render.cpp
*  Project        Radeon ProRender rendering tutorial
*
*  Description    Radeon ProRender SDK tutorials 
*
*  Copyright 2011 - 2017 Advanced Micro Devices, Inc. (unpublished)
*
*  All rights reserved.  This notice is intended as a precaution against
*  inadvertent publication and does not imply publication or any waiver
*  of confidentiality.  The year included in the foregoing notice is the
*  year of creation of the work.
*
\*****************************************************************************/
#include "RadeonProRender.h"
#include "rprDeprecatedApi.h"
#include "Math/mathutils.h"
#include "../common/common.h"
#include <cassert>
#include <iostream>

int main()
{
	//	enable Radeon ProRender API trace
	//	set this before any rpr API calls
//	rprContextSetParameter1u(0,"tracing",1);

    std::cout << "Radeon ProRender SDK simple rendering tutorial.\n";
    // Indicates whether the last operation has suceeded or not
    rpr_int status = RPR_SUCCESS;
	// Create OpenCL context using a single GPU 
	rpr_context context = NULL;
	
	// Register Tahoe ray tracing plugin.
	rpr_int tahoePluginID = rprRegisterPlugin("Tahoe64.dll"); 
	assert(tahoePluginID != -1);
	rpr_int plugins[] = { tahoePluginID };
	size_t pluginCount = sizeof(plugins) / sizeof(plugins[0]);

	// Create context using a single GPU 
	CHECK( rprCreateContext(RPR_API_VERSION, plugins, pluginCount, RPR_CREATION_FLAGS_ENABLE_GPU0, NULL, NULL, &context) );

	// Set active plugin.
	CHECK(  rprContextSetActivePlugin(context, plugins[0]) );


	rpr_material_system matsys;
	CHECK( rprContextCreateMaterialSystem(context, 0, &matsys) );
	// Check if it is created successfully
	if (status != RPR_SUCCESS)
	{
		std::cout << "Context creation failed: check your OpenCL runtime and driver versions.\n";
		return -1;
	}

	std::cout << "Context successfully created.\n";

	// Create a scene
	rpr_scene scene;
	CHECK( rprContextCreateScene(context, &scene) );

    // Create cube mesh
	rpr_shape cube;
	{
		CHECK( rprContextCreateMesh(context,
			(rpr_float const*)&cube_data[0], 24, sizeof(vertex),
			(rpr_float const*)((char*)&cube_data[0] + sizeof(rpr_float)*3), 24, sizeof(vertex),
			(rpr_float const*)((char*)&cube_data[0] + sizeof(rpr_float)*6), 24, sizeof(vertex),
			(rpr_int const*)indices, sizeof(rpr_int),
			(rpr_int const*)indices, sizeof(rpr_int),
			(rpr_int const*)indices, sizeof(rpr_int),
			num_face_vertices, 12, &cube) );

		// Add cube into the scene
		CHECK( rprSceneAttachShape(scene, cube) );

		// Create a transform: -2 unit along X axis and 1 unit up Y axis
		RadeonProRender::matrix m = RadeonProRender::translation(RadeonProRender::float3(-2, 1, 0));

		// Set the transform 
		CHECK( rprShapeSetTransform(cube, RPR_TRUE, &m.m00) );
	}
    // Create plane mesh
	rpr_shape plane;
	{
		CHECK( rprContextCreateMesh(context,
			(rpr_float const*)&plane_data[0], 4, sizeof(vertex),
			(rpr_float const*)((char*)&plane_data[0] + sizeof(rpr_float)*3), 4, sizeof(vertex),
			(rpr_float const*)((char*)&plane_data[0] + sizeof(rpr_float)*6), 4, sizeof(vertex),
			(rpr_int const*)indices, sizeof(rpr_int),
			(rpr_int const*)indices, sizeof(rpr_int),
			(rpr_int const*)indices, sizeof(rpr_int),
			num_face_vertices, 2, &plane) );

		// Add plane into the scene
		CHECK( rprSceneAttachShape(scene, plane) );
	}
    // Create camera
	rpr_camera camera;
	{
		CHECK( rprContextCreateCamera(context, &camera) );

		// Position camera in world space: 
		// Camera position is (5,5,20)
		// Camera aimed at (0,0,0)
		// Camera up vector is (0,1,0)
		CHECK( rprCameraLookAt(camera, 0, 5, 20, 0, 1, 0, 0, 1, 0) );

		CHECK( rprCameraSetFocalLength(camera, 75.f) );

		// Set camera for the scene
		CHECK( rprSceneSetCamera(scene, camera) );
	}
    // Set scene to render for the context
    CHECK( rprContextSetScene(context, scene) );
 
    // Create framebuffer to store rendering result
    rpr_framebuffer_desc desc;
    desc.fb_width = 800;
    desc.fb_height = 600;

    // 4 component 32-bit float value each
    rpr_framebuffer_format fmt = {4, RPR_COMPONENT_TYPE_FLOAT32};
	rpr_framebuffer frame_buffer;
	CHECK( rprContextCreateFrameBuffer(context, fmt, &desc, &frame_buffer) );

    // Clear framebuffer to black color
    CHECK( rprFrameBufferClear(frame_buffer) );

    // Set framebuffer for the context
    CHECK( rprContextSetAOV(context, RPR_AOV_COLOR, frame_buffer) );

	CHECK( rprContextSetParameter1u(context, "rendermode", RPR_RENDER_MODE_NORMAL) );



	/////////  Intstancing Tutorial  //////////

	rpr_shape instance;
	{
		rprContextCreateInstance(context, cube, &instance);

		// Create a transform: -2 unit along X axis and 1 unit up Y axis
		RadeonProRender::matrix m = RadeonProRender::translation(RadeonProRender::float3(2, 1, 0));

		// Set the transform 
		CHECK(rprShapeSetTransform(instance, RPR_TRUE, &m.m00));

		CHECK(rprSceneAttachShape(scene, instance));
	}

    // Progressively render an image
    for (int i = 0; i < NUM_ITERATIONS; ++i)
    {
        CHECK( rprContextRender(context) );
    }

    std::cout << "Rendering finished.\n";

    // Save the result to file
    CHECK( rprFrameBufferSaveToFile(frame_buffer, "03.png") );

    // Release the stuff we created
	CHECK(rprObjectDelete(matsys));
    CHECK(rprObjectDelete(plane));
	CHECK(rprObjectDelete(cube));
	CHECK(rprObjectDelete(instance));
	CHECK(rprObjectDelete(scene));
	CHECK(rprObjectDelete(camera));
	CHECK(rprObjectDelete(frame_buffer));
	CHECK(rprObjectDelete(context));
    return 0;
}


// Things to try in this tutorial:
// 1) Try to create a metacube with multiple instance
// 2) Compare memory usage with and without instancing
   // rpr_render_statistics renderstat;
   // CHECK(rprContextGetInfo(context, RPR_CONTEXT_RENDER_STATISTICS, sizeof(renderstat), &renderstat, NULL));
