/********************************************************************************
**
** Copyright 2015 - 2016 Advanced Micro Devices, Inc.
**
** AMD is granting you permission to use this software and documentation(if any)
** (collectively, the "Materials") pursuant to the terms and conditions of the
** Software License Agreement included with the Materials.If you do not have a
** copy of the Software License Agreement, contact your AMD representative for a
** copy.
**
** You agree that you will not reverse engineer or decompile the Materials, in
** whole or in part, except as allowed by applicable law.
**
** WARRANTY DISCLAIMER : THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF
** ANY KIND.AMD DISCLAIMS ALL WARRANTIES, EXPRESS, IMPLIED, OR STATUTORY,
** INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE, TITLE, NON - INFRINGEMENT, THAT THE
** SOFTWARE WILL RUN UNINTERRUPTED OR ERROR - FREE OR WARRANTIES ARISING FROM
** CUSTOM OF TRADE OR COURSE OF USAGE.THE ENTIRE RISK ASSOCIATED WITH THE USE OF
** THE SOFTWARE IS ASSUMED BY YOU.Some jurisdictions do not allow the exclusion
** of implied warranties, so the above exclusion may not apply to You.
**
** LIMITATION OF LIABILITY AND INDEMNIFICATION : AMD AND ITS LICENSORS WILL NOT,
** UNDER ANY CIRCUMSTANCES BE LIABLE TO YOU FOR ANY PUNITIVE, DIRECT,
** INCIDENTAL, INDIRECT, SPECIAL OR CONSEQUENTIAL DAMAGES ARISING FROM USE OF
** THE SOFTWARE OR THIS AGREEMENT EVEN IF AMD AND ITS LICENSORS HAVE BEEN
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.In no event shall AMD's total
** liability to You for all damages, losses, and causes of action (whether in
** contract, tort (including negligence) or otherwise) exceed the amount of $100
** USD.  You agree to defend, indemnify and hold harmless AMD and its licensors,
** and any of their directors, officers, employees, affiliates or agents from
** and against any and all loss, damage, liability and other expenses (including
** reasonable attorneys' fees), resulting from Your use of the Software or
** violation of the terms and conditions of this Agreement.
**
** U.S.GOVERNMENT RESTRICTED RIGHTS : The Materials are provided with
** "RESTRICTED RIGHTS." Use, duplication, or disclosure by the Government is
** subject to the restrictions as set forth in FAR 52.227 - 14 and DFAR252.227 -
** 7013, et seq., or its successor.Use of the Materials by the Government
** constitutes acknowledgment of AMD's proprietary rights in them.
**
** EXPORT RESTRICTIONS: The Materials may be subject to export restrictions as
**                      stated in the Software License Agreement.
**
********************************************************************************/

#ifndef __LIQUID_FLASH__H__
#define __LIQUID_FLASH__H__

#ifdef __cplusplus
extern "C" {
#endif

#if defined(_WIN64) || defined(_WINDOWS) || defined(_WIN32)

#include <wchar.h>

#define LF_API

    typedef char                lf_char8;
    typedef unsigned char       lf_uint8;
    typedef char                lf_int8;
    typedef unsigned short      lf_uint16;
    typedef signed short        lf_int16;
    typedef signed __int32      lf_int32;
    typedef unsigned __int32    lf_uint32;
    typedef signed __int64      lf_int64;
    typedef unsigned __int64    lf_uint64;
    typedef wchar_t             lf_char;

    typedef void*               lf_pvoid;
    typedef void*               lf_pvoid64;

    typedef void*               lf_handle;

#elif defined(__linux__)

#include <stdbool.h>
#include <stdint.h>

#define LF_API

#ifndef NULL
#   define NULL 0
#endif

    typedef int8_t              lf_char8;
    typedef uint8_t             lf_uint8;
    typedef int8_t              lf_int8;
    typedef uint16_t            lf_uint16;
    typedef int16_t             lf_int16;
    typedef int32_t             lf_int32;
    typedef uint32_t            lf_uint32;
    typedef int64_t             lf_int64;
    typedef uint64_t            lf_uint64;
    typedef char                lf_char;

    typedef void*               lf_pvoid;
    typedef void*               lf_pvoid64;

    typedef void*               lf_handle;

#endif

typedef enum lf_status_
{
    lf_success                  =  0,
    lf_os_error                 =  1,
    lf_invalid_parameter        =  2,
    lf_unsupported_gpu          =  3,
    lf_unsupported_platform     =  4,
    lf_out_of_resources         =  5,
    lf_invalid_file_object      =  6,
    lf_invalid_event_object     =  7,
    lf_invalid_context          =  8,
    lf_invalid_input            =  9,
    lf_file_open_fail           =  10,
    lf_unsupported_storage      =  11,
    lf_range_exceed_file_size   =  12,
    lf_range_exceed_buffer_size =  13,
    lf_io_error                 =  14, /* This error may happen due to P2P chipset support issues. */
    lf_io_pending               =  15,
    lf_file_close_error         =  16,
    lf_out_of_host_memory       =  17,
    lf_not_implemented          =  18,
    lf_wait_timeout             =  19,
    lf_no_access                =  20,
    lf_wait_error               =  21,
    lf_invalid_access           =  22,
    lf_kmode_not_supported      =  23,

lf_unsupported_drive = 50,
lf_no_mapping = 51,

    lf_unspecified_error        =  99
} lf_status;


typedef struct lf_file_*        lf_file;
typedef struct lf_event_*       lf_event;

typedef lf_uint32   lf_file_flags;

#define LF_READ                 (0x1)
#define LF_WRITE                (0x2)
#define LF_ATTACHED             (0x4) /*! This flag is for the internal use of the library
                                          It signifies the file object was created with
                                          'lfAttachToFile'. */

typedef struct _lf_version_info
{
    lf_uint16   MajorVersion;   /*! [out] Major version of LF library */
    lf_uint16   MinorVersion;   /*! [out] Minor version of LF library */
    lf_uint16   Build;          /*! [out] Build number of library     */
} lf_version_info;

typedef struct _region_descriptor
{
    lf_uint64   fileOffset;     /*! [in] File offset in blocks   */
    lf_uint64   bufferOffset;   /*! [in] Buffer offset in blocks */
    lf_uint64   regionSize;     /*! [in] Region size in blocks   */
} lf_region_descriptor;



/**
*********************************************************************************
* @brief Initialize threading
*
* @return       None
*********************************************************************************
*/
bool LF_API lfInit();

/**
*********************************************************************************
* @brief Close threading
*
* @return       None
*********************************************************************************
*/
void LF_API lfTerminate();


/**
*********************************************************************************
* @brief Returns version information of library
*
* @param[out]   pVersionInfo        - pointer to version info structure
*
* @return       None
*********************************************************************************
*/
void LF_API lfGetVersion(
    lf_version_info*    pVersionInfo);

/**
* @brief Opens a file for P2P transfer.
*
* @details
*
* @param[in] fileName The path of the file to be transfered to GPU memory.
*                      On Windows the string is expected to be encoded with UTF16.
*                      On Linux the string is expected to be encoded with UTF8.
* @param[in] flags flags indicating if the file is to be opened for
*                  read or for write.
* @param[out] errCodeRet return code.
*               lf_file_open_fail - If the file does not exist or the user
*                                   does not have permission to open the file.
                lf_unsupported_drive - If the file is located on unsupported
                                       drive.
*
* @return       Handle to the file object on success, Null on failure.
*********************************************************************************
*/
lf_file LF_API lfOpenFile(
    const lf_char*  fileName,
    lf_file_flags   flags,
    lf_status*      errCodeRet);

/**
* @brief Attaches to file for P2P transfer.
*
* @details
*
* @param[in]  fileHandle Handle to the file
*                        On Windows file HANDLE.
*                        On Linux int.
* @param[out] errCodeRet return code.
*               lf_file_open_fail    - If the file does not exist or the user
*                                      does not have permission to open the file.
*               lf_unsupported_drive - If the file is located on unsupported
                                       drive.
*
* @return       Handle to the file object on success, Null on failure.
*********************************************************************************
*/
#ifdef ATTACH_TO_FILE
lf_file LF_API lfAttachToFile(
    void*           fileHandle,
    lf_status*      errCodeRet);
#endif
/**
*********************************************************************************
* @brief Returns the file block size in bytes.
*
* @detail File data is partitioned to blocks. P2P data transfers can only
*         be made on block boundary. The block size if common to all files
*         located on a specific drive.
*
* @param file       File object for query.
* @param blockSize  Size of block in bytes.
*
* @retval lf_status Status of the operation
*********************************************************************************
*/
lf_status LF_API lfGetFileBlockSize(
    lf_file     file,
    lf_uint32*  blockSize);

/**
*********************************************************************************
* @brief Returns file size in bytes.
*
* @detail returns file size in bytes
*
* @param file       File object for query.
* @param fileSize   File size in bytes.
*
* @retval lf_status Status of the operation
*********************************************************************************
*/
lf_status LF_API lfGetFileSize(
    lf_file     file,
    lf_uint64*  fileSize);

/**
*********************************************************************************
* @brief Transfers the content of a file, P2P style, to CPU mapped device memory.
*
* @details This function may operate in synchronous or asynchronous fashion.
*
* @param[in] destinationBuffer Pointer to destination buffer to copy the data to.
*
* @param[in] destinationBufferSize Length of destination buffer to copy the data to.
*
* @param[in] file Source file object for the data transfer.
*
* @param[in] regionCount Number or regions in the file to copy.
*
* @param[in] pDescriptors Pointer to an array of region descriptor detailing
*                         the offset and size of regions to be copied.
*
* @param[in] evt   An event object that identifies this particular read command
*                  and can be used to query or queue a wait
*                  for this particular command to complete. Event
*                  can be NULL in which case the call will be synchronous.
*
* @retval lf_invalide_file  - If the file object is invalid.
* @retval lf_io_error       - On bus error due to chipset support issue.
*********************************************************************************
*/
lf_status LF_API lfReadFile(
    void*                   destinationBuffer,
    lf_uint64               destinationBufferSize,
    lf_file                 file,
    lf_uint32               regionCount,
    lf_region_descriptor*   pDescriptors,
    lf_event                evt);

/**
*********************************************************************************
* @brief Transfers the content of a file, P2P style, from CPU mapped device memory.
*
* @details This function may operate in synchronous or asynchronous fashion.
*
* @param[in] sourceBuffer Pointer to source buffer to copy the data to.
*
* @param[in] sourceBufferSize Length of source buffer to copy the data to.
*
* @param[in] file Source file object for the data transfer.
*
* @param[in] regionCount Number or regions in the file to copy.
*
* @param[in] pDescriptors Pointer to an array of region descriptor detailing
*                         the offset and size of regions to be copied.
*
* @param[in] evt   An event object that identifies this particular write
*                  command and can be used to query or queue a wait
*                  for this particular command to complete.
*                  Event can be NULL in which case the call will be synchronous.
*
* @retval lf_invalide_file  - If the file object is invalid.
* @retval lf_io_error       - On bus error due to chipset support issue.
*********************************************************************************
*/
lf_status LF_API lfWriteFile(
    void*                   sourceBuffer,
    lf_uint64               sourceBufferSize,
    lf_file                 file,
    lf_uint32               regionCount,
    lf_region_descriptor*   pDescriptors,
    lf_event                evt);

/**
*********************************************************************************
* @brief release a file object.
*
* @details All pending read/write operations on the file will be
*          finished before the function returns.
*
* @param[in] file File object handle to close.
*
* @retval lf_success            - On success.
* @retval lf_file_close_error   - If OS file close API fails.
*********************************************************************************
*/
lf_status LF_API lfReleaseFile(
    lf_file file);

/**
*********************************************************************************
* @brief Create Event Object
*
* @details
*
* @param[in] event Event object to wait on.
*
* @retval lf_success            - On success.
* @retval lf_out_of_host_memory - If event cannot be created.
*********************************************************************************
*/
lf_event LF_API lfCreateEvent(
    lf_status* status);

/**
*********************************************************************************
* @brief waits on event object.
*
* @details
*
* @param[in] evt   Event object to wait on.
*
* @retval lf_success                - On success.
* @retval lf_invalid_event_object   - If invalid event is sent.
*********************************************************************************
*/
lf_status LF_API lfWaitEvent(
    lf_event evt);

/**
*********************************************************************************
* @brief delete event object.
*
* @details Deletes event object without waiting on it.
*
* @param[in] evt   Event object to wait on.
*
* @retval lf_success                - On success.
* @retval lf_invalid_event_object   - If invalid event is sent.
*********************************************************************************
*/
lf_status LF_API lfReleaseEvent(
    lf_event evt);


/**
*********************************************************************************
* @brief Checks whether LF extension should be supported for a specific GPU
*
* @param[in] deviceId   - GPU device ID.
* @param[in] revisionId - GPU revision ID.
*
* @retval lf_success                - If the GPU is supported with the extension.
* @retval lf_unsupported_gpu        - If the device ID is unrecognized or unsupported with the
                                      the extension.
*********************************************************************************
*/
lf_status LF_API lfCheckExtensionSupportForDevice(
    lf_uint32 deviceId,
    lf_uint32 revisionId);

#ifdef __cplusplus /* closure of 'extern "C"' scope*/
}
#endif

#endif /* __LIQUID_FLASH__H__ */
