# =====================================================================
# Copyright 2017 (c), Advanced Micro Devices, Inc. All rights reserved.
# =====================================================================

import sys
import os.path
from subprocess import Popen
import subprocess
import re

DEVICES_HDR_NAME = 'lc_targets.h'
LLC_OPTIONS = '-mtriple=amdgcn-amd-amdhsa -mcpu=help'
SUPPORTED_DEVICES = {'gfx900'}

HDR_TITLE = '//=====================================================================\n'  \
            '// Copyright 2017 (c), Advanced Micro Devices, Inc. All rights reserved.\n' \
            '//\n' \
            '// This file is auto-generated by the LC Device Generator utility.\n'       \
            '//=====================================================================\n\n'

def GetDeiveNames(llc_output_lines):
    deviceNames = []
    numLines = len(llc_output_lines)
    for i in range(0, numLines):
        if re.search(r'Available CPUs', llc_output_lines[i]):
            i += 1
            # skip empty lines before the list of devices
            while llc_output_lines[i] == "" and i < numLines:
                i += 1
            if i == numLines:
                return []
            # Process the list of devices
            while llc_output_lines[i] != "" and i < numLines:
                deviceInfo = re.search('(.+)\-', llc_output_lines[i])
                if deviceInfo:
                    deviceName = deviceInfo.group(1).strip()
                    if len(deviceName) > 0:
                        deviceNames.append(deviceName)
                i += 1
            # while ..
            return deviceNames
        # if re.search(r'Available CPUs')
    # for i in range(0, numLines)
    return []
# GetDeiveNames()

def GenDevicesHdr(hdrFileDir, deviceNames):
    hdrFilePath = hdrFileDir + '/' + DEVICES_HDR_NAME
    hdrFile = open(hdrFilePath, 'w')
    if not hdrFile:
        print('Failed to create a header file.')
        return False
    hdrFile.write(HDR_TITLE)
    hdrFile.write('#include <vector>\n\n')
    devicesDeclString = 'static const std::vector<std::string>  LC_TARGET_NAMES = { '
    first = True;
    for deviceName in deviceNames:
        if deviceName in SUPPORTED_DEVICES:
            if not first:
                devicesDeclString += ', '
            devicesDeclString += '"' + deviceName + '"'
            first = False
    devicesDeclString += ' };\n'
    hdrFile.write(devicesDeclString)
    hdrFile.flush()
    hdrFile.close()
    return True
# GenDevicesHdr()

if __name__ == '__main__':
    if len(sys.argv) == 3:
        llc_path = sys.argv[1]
        hdr_dir  = sys.argv[2]
        if os.path.exists(llc_path) and os.path.exists(hdr_dir):
            llc_proc = Popen(llc_path + ' ' + LLC_OPTIONS, stdin = subprocess.PIPE, stdout = subprocess.PIPE, stderr = subprocess.PIPE)
            result = llc_proc.communicate('', 10)
            if llc_proc.returncode == 1:
                print('Failed to execute LLC\n')
                sys.exit(1)
            llc_output = result[1] # stderr
            deviceNames = GetDeiveNames(llc_output.decode().splitlines())
            if (len(deviceNames) == 0):
                print('Failed to get the device list.')
                sys.exit(1)
            else:
                if GenDevicesHdr(hdr_dir, deviceNames) == False:
                    sys.exit(1)
    else:
        # Something went wrong
        print('LC Device List Generator. (C) Advanced Micro Devices.')
        print('Usage:')
        print('    gen_devices <path to llc executable> <output folder>')
