//////////////////////////////////////////////////////////////////////////////
//
//  Copyright (C) 2007-2012 Infinite Z, Inc.  All Rights Reserved.
//
//  File:       ZSStylusShape.cs
//  Content:    Interface class for maintains the stylus object itself.
//  SVN Info:   $Id$
//
//////////////////////////////////////////////////////////////////////////////

using UnityEngine;
using System.Collections;
using System.Collections.Generic;
using System;

/// <summary>
/// Class for maintaining the shape of a stylus tool.
/// While ZSStylusSelector interfaces the stylus with the scene, ZSStylusShape can make the stylus itself dynamic.
/// The stylus can be made to change shape, color, or behavior here based on overlaps generated by ZSStylusSelector.
/// </summary>
public abstract class ZSStylusShape : MonoBehaviour
{
  /// <summary>
  /// The default color of objects in the m_colorCodedObjects list.
  /// </summary>
  public Color m_defaultColor = Color.white;

  /// <summary>
  /// When the stylus overlaps with objects in the scene, this color will be applied to all objects in the m_colorCodedObjects list.
  /// </summary>
  public Color m_hoveredColor = Color.green;

  /// <summary>
  /// The list of objects whose color will change based on overlap between the stylus and objects in the scene.
  /// </summary>
  public GameObject[] m_colorCodedObjects;

  /// <summary>
  /// The current color of all objects in the m_colorCodedObjects list.
  /// </summary>
  public virtual Color color
  {
    set
    {
      foreach (GameObject go in m_colorCodedObjects)
        go.renderer.material.SetColor("_Emission", value);
      m_color = value;
    }
    get { return m_color; }
  }

  /// <summary>
  /// The "epicenter" of the stylus.  Defined in local space.
  /// If multiple objects collide with the stylus, the one closest to this point will be preferred in hovering and selection.
  /// This also determines the length of the raycast.
  /// If the hotspot is at (0, 0, 0), the ray will be infinite, in +Z direction.
  /// </summary>
  public virtual Vector3 hotSpot { get { return m_hotSpot; } }

  protected Vector3 m_hotSpot = Vector3.zero;
  protected Color m_color;

  /// <summary>
  /// This callback will be called whenever the stylus begins intersecting with an object.
  /// </summary>
  public virtual void OnHoverBegin(GameObject gameObject, Vector3 point) { color = m_hoveredColor; }

  /// <summary>
  /// This callback will be called after each frame during which the stylus is intersecting with an object.
  /// </summary>
  public virtual void OnHoverStay(GameObject gameObject, Vector3 point) {}

  /// <summary>
  /// This callback will be called whenever the stylus finishes intersecting with an object.
  /// </summary>
  public virtual void OnHoverEnd(GameObject gameObject) { color = m_defaultColor; }

  /// <summary>
  /// This callback will be called whenever the stylus begins dragging an object.
  /// </summary>
  public virtual void OnDragBegin(GameObject gameObject) {}

  /// <summary>
  /// This callback will be called whenever the stylus finishes dragging an object.
  /// </summary>
  public virtual void OnDragEnd(GameObject[] gameObjects) {}
}
