# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2023 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseDeclaredVarsMoreThanAssignments", "")] Param()

#region CSharp_code
$Dflat = @"
using System;
using System.Runtime.InteropServices;

namespace IntelEthernetCmdlets
{
    class Util
    {
        public static bool CheckBit(UInt16 Value, UInt16 Bit)
        {
            if ((Value & Bit) == Bit)
            {
                return true;
            }
            return false;
        }
    }

    public class DcbConstants
    {
        public const Int32 MAX_TRAFFIC_CLASS = 8;
        public const Int32 MAX_USER_PRIORITY = 8;
        public const Int32 DCBX_MAX_APPS = 32;
        public const Int32 DCBX_APPS_SIZE = DCBX_MAX_APPS * 4;
    }

    public class DcbDataRegFlagValue
    {
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_CAP = 0x0001;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_EN = 0x0002;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_FW_INC = 0x0004;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_NPAR = 0x0008;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_ISCSI = 0x0010;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_MAIN_VSI = 0x0020;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_FCOE_VSI = 0x0040;
    }

    public class DcbxModeValue
    {
        public const Byte DCBX_MODE_CEE = 0x1;
        public const Byte DCBX_MODE_IEEE = 0x2;
    }

    public enum RegVer
    {
        VER_INVALID,
        I40E,
        ICE
    }

    public class IntlLan_GetDcbData
    {
        public IntlLan_GetDcbData()
        {
            LocalConfig = new IntlLan_GetDcbDataDcbxConfig();
            LocalConfig.ConfigName = "LocalConfig";

            RemoteConfig = new IntlLan_GetDcbDataDcbxConfig();
            RemoteConfig.ConfigName = "RemoteConfig";

            DesiredConfig = new IntlLan_GetDcbDataDcbxConfig();
            DesiredConfig.ConfigName = "DesiredConfig";
        }
        public String Name { get; set; }
        public UInt16 Flags { get; set; }
        public Boolean DcbCapable { get; set; }
        public Boolean DcbEnabled { get; set; }

        public UInt16 DcbxStatus { get; set; }

        public IntlLan_GetDcbDataDcbxConfig LocalConfig { get; set; }
        public IntlLan_GetDcbDataDcbxConfig RemoteConfig { get; set; }
        public IntlLan_GetDcbDataDcbxConfig DesiredConfig { get; set; }

        public IIntlLan_GetDcbDataRegisters DcbRegs { get; set; }
        public RegVer RegVer { get; set; }
    }

    public class IntlLan_GetDcbDataDcbxConfig
    {
        public IntlLan_GetDcbDataDcbxConfig()
        {
            EtsCfg = new IntlLan_GetDcbDataEts();
            EtsCfg.TableName = "EtsCfg";
            EtsRec = new IntlLan_GetDcbDataEts();
            EtsRec.TableName = "EtsRec";
            Pfc = new IntlLan_GetDcbDataPfc();
        }
        public String Name { get; set; }
        public Byte DcbxMode { get; set; }
        public Byte AppMode { get; set; }
        public UInt32 CeeTlvStatus { get; set; }
        public UInt32 NumApps { get; set; }

        public String ConfigName { get; set; }

        public IntlLan_GetDcbDataEts EtsCfg { get; set; }
        public IntlLan_GetDcbDataEts EtsRec { get; set; }
        public IntlLan_GetDcbDataPfc Pfc { get; set; }
        public IntlLan_GetDcbDataAppPriority[] AppPriority { get; set; }
    }

    public class IntlLan_GetDcbDataEts
    {
        public IntlLan_GetDcbDataEts()
        {
            UP = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            BW = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            TSA = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            PfcEnableBits = new Boolean[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public String TableName { get; set; }
        public Boolean Willing { get; set; }
        public Byte CBS { get; set; }
        public Byte MaxTCs { get; set; }
        public Boolean[] PfcEnableBits { get; set; }
        public Byte[] UP { get; set; }
        public Byte[] BW { get; set; }
        public Byte[] TSA { get; set; }
    }

    public class IntlLan_GetDcbDataPfc
    {
        public IntlLan_GetDcbDataPfc()
        {
            PfcEnableBits = new Boolean[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public Boolean Willing { get; set; }
        public Byte MBC { get; set; }
        public Byte PFCcap { get; set; }
        public Byte PfcEnable { get; set; }
        public Boolean[] PfcEnableBits { get; set; }
    }

    public class IntlLan_GetDcbDataAppPriority
    {
        public Byte Priority { get; set; }
        public Byte Selector { get; set; }
        public UInt16 ProtocolId { get; set; }
    }

    public interface IIntlLan_GetDcbDataRegisters { }

    public class IntlLan_GetDcbDataRegistersI40e : IIntlLan_GetDcbDataRegisters
    {
        public IntlLan_GetDcbDataRegistersI40e()
        {
            I40E_PRT_TCTUPR = new UInt32[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public UInt32 I40E_PRTDCB_GENC { get; set; }
        public UInt32 I40E_PRTDCB_TC2PFC { get; set; }
        public UInt32 I40E_PRTDCB_MFLCN { get; set; }
        public UInt32 I40E_PRTDCB_TUP2TC { get; set; }
        public UInt32 I40E_PRTDCB_RUP2TC { get; set; }
        public UInt32[] I40E_PRT_TCTUPR { get; set; }
        public UInt32 I40E_VSI_TUPR_MainVsi { get; set; }
        public UInt32 I40E_VSI_RUPR_MainVsi { get; set; }
        public UInt32 I40E_VSI_TUPR_FCoEVsi { get; set; }
        public UInt32 I40E_VSI_RUPR_FCoEVsi { get; set; }
    }

    public class IntlLan_GetDcbDataRegistersIce : IIntlLan_GetDcbDataRegisters
    {
        public IntlLan_GetDcbDataRegistersIce()
        {
            PrtTcTuprVal = new UInt32[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public UInt32 PrtDcbGencVal { get; set; }
        public UInt32 GlDcbTc2PfcVal { get; set; }
        public UInt32 PrtMacHsecCtlTxPauseEnableVal { get; set; }
        public UInt32 PrtMacHsecCtlRxPauseEnableVal { get; set; }
        public UInt32 PrtDcbTup2TcVal { get; set; }
        public UInt32 PrtDcbRup2TcVal { get; set; }
        public UInt32[] PrtTcTuprVal { get; set; }
        public UInt32 VsiTuprMainVsiVal { get; set; }
        public UInt32 VsiRuprMainVsiVal { get; set; }
    }

    public unsafe interface IGetDcb
    {
        IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes);
    }

    #region 100G DCB
    public unsafe class GetDcbIce : IGetDcb
    {
        private const Int32 ICE_DSCP_NUM_VAL = 64;
        private const Int32 DSCP_MAPPED_SIZE = 8;


        private struct ice_dcb_ets_cfg
        {
            public Byte willing;
            public Byte cbs;
            public Byte maxtcs;
            public fixed Byte prio_table[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tcbwtable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tsatable[DcbConstants.MAX_TRAFFIC_CLASS];
        }


        private struct ice_dcb_pfc_cfg
        {
            public Byte willing;
            public Byte mbc;
            public Byte pfccap;
            public Byte pfcenable;
        }


        private struct ice_dcb_app_priority_table
        {
            public UInt16 prot_id;
            public Byte priority;
            public Byte selector;
        }


        private struct ice_dcbx_cfg
        {
            public Byte dcbx_mode;
            public Byte app_mode;
            public UInt32 numapps;
            public UInt32 tlv_status;
            public ice_dcb_ets_cfg etscfg;
            public ice_dcb_ets_cfg etsrec;
            public ice_dcb_pfc_cfg pfc;
            public fixed Byte app[DcbConstants.DCBX_APPS_SIZE];
        }


        private struct DcbRegisters
        {
            public UInt32 PrtDcbGencVal;
            public UInt32 GlDcbTc2PfcVal;
            public UInt32 PrtMacHsecCtlTxPauseEnableVal;
            public UInt32 PrtMacHsecCtlRxPauseEnableVal;
            public UInt32 PrtDcbTup2TcVal;
            public UInt32 PrtDcbRup2TcVal;
            public fixed UInt32 PrtTcTuprVal[8];
            public UInt32 VsiTuprMainVsiVal;
            public UInt32 VsiRuprMainVsiVal;
        }


        private struct DcbGetGenData
        {
            public UInt16 Flags;
            public UInt16 DcbxStatus;
            public ice_dcbx_cfg LocalConfig;
            public ice_dcbx_cfg RemoteConfig;
            public ice_dcbx_cfg DesiredConfig;
            public DcbRegisters DcbRegs;
        }

        public IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes)
        {
            var pData = GCHandle.Alloc(bytes, GCHandleType.Pinned);
            var result = (DcbGetGenData)Marshal.PtrToStructure(pData.AddrOfPinnedObject(), typeof(DcbGetGenData));
            pData.Free();
            DcbGetGenData* pDcbData = (DcbGetGenData*)&result;
            return CreatePsObject(strPortName, *pDcbData);
        }

        private IntlLan_GetDcbData CreatePsObject(String strPortName, DcbGetGenData DcbData)
        {
            IntlLan_GetDcbData GetDcbData = new IntlLan_GetDcbData
            {
                Name = strPortName,
                Flags = DcbData.Flags,
                DcbCapable = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_CAP)),
                DcbEnabled = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_EN)),
                DcbxStatus = DcbData.DcbxStatus,
            };

            CopyDcbxConfig(strPortName, DcbData.LocalConfig, GetDcbData.LocalConfig);
            CopyDcbxConfig(strPortName, DcbData.RemoteConfig, GetDcbData.RemoteConfig);
            CopyDcbxConfig(strPortName, DcbData.DesiredConfig, GetDcbData.DesiredConfig);

            GetDcbData.DcbRegs = new IntlLan_GetDcbDataRegistersIce();
            GetDcbData.RegVer = RegVer.ICE;
            CopyDcbRegs(strPortName, DcbData.DcbRegs, GetDcbData.DcbRegs);
            return GetDcbData;
        }


        private void CopyDcbxConfig(String strPortName, ice_dcbx_cfg FromDcbxConfig, IntlLan_GetDcbDataDcbxConfig ToDcbxConfig)
        {
            ToDcbxConfig.Name = strPortName;
            ToDcbxConfig.DcbxMode = FromDcbxConfig.dcbx_mode;
            ToDcbxConfig.AppMode = FromDcbxConfig.app_mode;
            ToDcbxConfig.CeeTlvStatus = FromDcbxConfig.tlv_status;
            ToDcbxConfig.NumApps = FromDcbxConfig.numapps;
            if (ToDcbxConfig.NumApps > DcbConstants.DCBX_MAX_APPS)
            {
                ToDcbxConfig.NumApps = DcbConstants.DCBX_MAX_APPS;
            }

            CopyPfcConfig(strPortName, FromDcbxConfig.pfc, ToDcbxConfig.Pfc);

            CopyEtsConfig(strPortName, FromDcbxConfig.etscfg, ToDcbxConfig.EtsCfg);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsCfg.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            CopyEtsConfig(strPortName, FromDcbxConfig.etsrec, ToDcbxConfig.EtsRec);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsRec.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            ToDcbxConfig.AppPriority = new IntlLan_GetDcbDataAppPriority[ToDcbxConfig.NumApps];

            CopyApps(FromDcbxConfig.app, ToDcbxConfig.NumApps, ToDcbxConfig.AppPriority);
        }


        private void CopyEtsConfig(String strPortName, ice_dcb_ets_cfg FromEtsConfig, IntlLan_GetDcbDataEts ToEtsConfig)
        {
            ToEtsConfig.Name = strPortName;
            ToEtsConfig.Willing = (FromEtsConfig.willing == 1);
            ToEtsConfig.CBS = FromEtsConfig.cbs;
            ToEtsConfig.MaxTCs = FromEtsConfig.maxtcs;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToEtsConfig.UP[i] = FromEtsConfig.prio_table[i];
                ToEtsConfig.BW[i] = FromEtsConfig.tcbwtable[i];
                ToEtsConfig.TSA[i] = FromEtsConfig.tsatable[i];
            }
        }


        private void CopyPfcConfig(String strPortName, ice_dcb_pfc_cfg FromPfcConfig, IntlLan_GetDcbDataPfc ToPfcConfig)
        {
            ToPfcConfig.Name = strPortName;
            ToPfcConfig.Willing = (FromPfcConfig.willing == 1);
            ToPfcConfig.MBC = FromPfcConfig.mbc;
            ToPfcConfig.PFCcap = FromPfcConfig.pfccap;
            ToPfcConfig.PfcEnable = FromPfcConfig.pfcenable;
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToPfcConfig.PfcEnableBits[i] = ((ToPfcConfig.PfcEnable & (1 << i)) == (1 << i));
            }
        }


        private void CopyApps(Byte* FromApps, UInt32 NumApps, IntlLan_GetDcbDataAppPriority[] ToApps)
        {
            ice_dcb_app_priority_table* AppTable = (ice_dcb_app_priority_table*)FromApps;
            for (int i = 0; i < NumApps; i++)
            {
                // Since this array's elements are a class, and classes are ref types, we need
                // to explicitly allocate the elements.
                ToApps[i] = new IntlLan_GetDcbDataAppPriority();

                if (AppTable->prot_id > 0)
                {
                    ToApps[i].Priority = AppTable->priority;
                    ToApps[i].Selector = AppTable->selector;
                    ToApps[i].ProtocolId = AppTable->prot_id;
                }

                AppTable++;
            }
        }


        private void CopyDcbRegs(String strPortName, DcbRegisters FromDcbRegs, IIntlLan_GetDcbDataRegisters ToDcbRegs)
        {
            IntlLan_GetDcbDataRegistersIce toReg = (IntlLan_GetDcbDataRegistersIce)ToDcbRegs;

            toReg.Name = strPortName;
            toReg.PrtDcbGencVal = FromDcbRegs.PrtDcbGencVal;
            toReg.GlDcbTc2PfcVal = FromDcbRegs.GlDcbTc2PfcVal;
            toReg.PrtMacHsecCtlTxPauseEnableVal = FromDcbRegs.PrtMacHsecCtlTxPauseEnableVal;
            toReg.PrtMacHsecCtlRxPauseEnableVal = FromDcbRegs.PrtMacHsecCtlRxPauseEnableVal;
            toReg.PrtDcbTup2TcVal = FromDcbRegs.PrtDcbTup2TcVal;
            toReg.PrtDcbRup2TcVal = FromDcbRegs.PrtDcbRup2TcVal;
            toReg.VsiTuprMainVsiVal = FromDcbRegs.VsiTuprMainVsiVal;
            toReg.VsiRuprMainVsiVal = FromDcbRegs.VsiRuprMainVsiVal;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                toReg.PrtTcTuprVal[i] = FromDcbRegs.PrtTcTuprVal[i];
            }
        }
    }
    #endregion

    #region 40G DCB
    public unsafe class GetDcbI40e : IGetDcb
    {
        private struct i40e_dcb_ets_config
        {
            public Byte willing;
            public Byte cbs;
            public Byte maxtcs;
            public fixed Byte prioritytable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tcbwtable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tsatable[DcbConstants.MAX_TRAFFIC_CLASS];
        }

        private struct i40e_dcb_pfc_config
        {
            public Byte willing;
            public Byte mbc;
            public Byte pfccap;
            public Byte pfcenable;
        }

        private struct i40e_dcb_app_priority_table
        {
            public Byte priority;
            public Byte selector;
            public UInt16 protocolid;
        }

        private struct i40e_dcbx_config
        {
            public Byte dcbx_mode;
            public Byte app_mode;
            public UInt32 numapps;
            public UInt32 tlv_status;
            public i40e_dcb_ets_config etscfg;
            public i40e_dcb_ets_config etsrec;
            public i40e_dcb_pfc_config pfc;
            public fixed Byte app[DcbConstants.DCBX_APPS_SIZE];
        }

        private struct DcbRegisters
        {
            public UInt32 I40E_PRTDCB_GENC;
            public UInt32 I40E_PRTDCB_TC2PFC;
            public UInt32 I40E_PRTDCB_MFLCN;
            public UInt32 I40E_PRTDCB_TUP2TC;
            public UInt32 I40E_PRTDCB_RUP2TC;
            public fixed UInt32 I40E_PRT_TCTUPR[8];
            public UInt32 I40E_VSI_TUPR_MainVsi;
            public UInt32 I40E_VSI_RUPR_MainVsi;
            public UInt32 I40E_VSI_TUPR_FCoEVsi;
            public UInt32 I40E_VSI_RUPR_FCoEVsi;
        }

        private struct DcbGetGenData
        {
            public UInt16 Flags;
            public UInt16 DcbxStatus;
            public i40e_dcbx_config LocalConfig;
            public i40e_dcbx_config RemoteConfig;
            public i40e_dcbx_config DesiredConfig;
            public DcbRegisters DcbRegs;
        }

        public IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes)
        {
            var pData = GCHandle.Alloc(bytes, GCHandleType.Pinned);
            var result = (DcbGetGenData)Marshal.PtrToStructure(pData.AddrOfPinnedObject(), typeof(DcbGetGenData));
            pData.Free();
            DcbGetGenData* pDcbData = (DcbGetGenData*)&result;
            return CreatePsObject(strPortName, *pDcbData);
        }

        private IntlLan_GetDcbData CreatePsObject(String strPortName, DcbGetGenData DcbData)
        {
            IntlLan_GetDcbData GetDcbData = new IntlLan_GetDcbData
            {
                Name = strPortName,
                Flags = DcbData.Flags,
                DcbCapable = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_CAP)),
                DcbEnabled = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_EN)),
                DcbxStatus = DcbData.DcbxStatus,
            };

            CopyDcbxConfig(strPortName, DcbData.LocalConfig, GetDcbData.LocalConfig);
            CopyDcbxConfig(strPortName, DcbData.RemoteConfig, GetDcbData.RemoteConfig);
            CopyDcbxConfig(strPortName, DcbData.DesiredConfig, GetDcbData.DesiredConfig);

            GetDcbData.DcbRegs = new IntlLan_GetDcbDataRegistersI40e();
            GetDcbData.RegVer = RegVer.I40E;
            CopyDcbRegs(strPortName, DcbData.DcbRegs, GetDcbData.DcbRegs);
            return GetDcbData;
        }

        private void CopyDcbxConfig(String strPortName, i40e_dcbx_config FromDcbxConfig, IntlLan_GetDcbDataDcbxConfig ToDcbxConfig)
        {
            ToDcbxConfig.Name = strPortName;
            ToDcbxConfig.DcbxMode = FromDcbxConfig.dcbx_mode;
            ToDcbxConfig.AppMode = FromDcbxConfig.app_mode;
            ToDcbxConfig.CeeTlvStatus = FromDcbxConfig.tlv_status;
            ToDcbxConfig.NumApps = FromDcbxConfig.numapps;
            if (ToDcbxConfig.NumApps > DcbConstants.DCBX_MAX_APPS)
            {
                ToDcbxConfig.NumApps = DcbConstants.DCBX_MAX_APPS;
            }

            CopyPfcConfig(strPortName, FromDcbxConfig.pfc, ToDcbxConfig.Pfc);

            CopyEtsConfig(strPortName, FromDcbxConfig.etscfg, ToDcbxConfig.EtsCfg);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsCfg.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            CopyEtsConfig(strPortName, FromDcbxConfig.etsrec, ToDcbxConfig.EtsRec);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsRec.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            ToDcbxConfig.AppPriority = new IntlLan_GetDcbDataAppPriority[ToDcbxConfig.NumApps];

            CopyApps(FromDcbxConfig.app, ToDcbxConfig.NumApps, ToDcbxConfig.AppPriority);
        }


        private void CopyEtsConfig(String strPortName, i40e_dcb_ets_config FromEtsConfig, IntlLan_GetDcbDataEts ToEtsConfig)
        {
            ToEtsConfig.Name = strPortName;
            ToEtsConfig.Willing = (FromEtsConfig.willing == 1);
            ToEtsConfig.CBS = FromEtsConfig.cbs;
            ToEtsConfig.MaxTCs = FromEtsConfig.maxtcs;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToEtsConfig.UP[i] = FromEtsConfig.prioritytable[i];
                ToEtsConfig.BW[i] = FromEtsConfig.tcbwtable[i];
                ToEtsConfig.TSA[i] = FromEtsConfig.tsatable[i];
            }
        }


        private void CopyPfcConfig(String strPortName, i40e_dcb_pfc_config FromPfcConfig, IntlLan_GetDcbDataPfc ToPfcConfig)
        {
            ToPfcConfig.Name = strPortName;
            ToPfcConfig.Willing = (FromPfcConfig.willing == 1);
            ToPfcConfig.MBC = FromPfcConfig.mbc;
            ToPfcConfig.PFCcap = FromPfcConfig.pfccap;
            ToPfcConfig.PfcEnable = FromPfcConfig.pfcenable;
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToPfcConfig.PfcEnableBits[i] = ((ToPfcConfig.PfcEnable & (1 << i)) == (1 << i));
            }
        }


        private void CopyApps(Byte* FromApps, UInt32 NumApps, IntlLan_GetDcbDataAppPriority[] ToApps)
        {
            i40e_dcb_app_priority_table* AppTable = (i40e_dcb_app_priority_table*)FromApps;
            for (int i = 0; i < NumApps; i++)
            {
                // Since this array's elements are a class, which and classes are ref types, we need
                // to explicitly allocate the elements.
                ToApps[i] = new IntlLan_GetDcbDataAppPriority();

                if (AppTable->protocolid > 0)
                {
                    ToApps[i].Priority = AppTable->priority;
                    ToApps[i].Selector = AppTable->selector;
                    ToApps[i].ProtocolId = AppTable->protocolid;
                }

                AppTable++;
            }
        }


        private void CopyDcbRegs(String strPortName, DcbRegisters FromDcbRegs, IIntlLan_GetDcbDataRegisters ToDcbRegs)
        {
            IntlLan_GetDcbDataRegistersI40e toReg = (IntlLan_GetDcbDataRegistersI40e)ToDcbRegs;

            toReg.Name = strPortName;
            toReg.I40E_PRTDCB_GENC = FromDcbRegs.I40E_PRTDCB_GENC;
            toReg.I40E_PRTDCB_TC2PFC = FromDcbRegs.I40E_PRTDCB_TC2PFC;
            toReg.I40E_PRTDCB_MFLCN = FromDcbRegs.I40E_PRTDCB_MFLCN;
            toReg.I40E_PRTDCB_TUP2TC = FromDcbRegs.I40E_PRTDCB_TUP2TC;
            toReg.I40E_PRTDCB_RUP2TC = FromDcbRegs.I40E_PRTDCB_RUP2TC;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                toReg.I40E_PRT_TCTUPR[i] = FromDcbRegs.I40E_PRT_TCTUPR[i];
            }

            toReg.I40E_VSI_TUPR_MainVsi = FromDcbRegs.I40E_VSI_TUPR_MainVsi;
            toReg.I40E_VSI_RUPR_MainVsi = FromDcbRegs.I40E_VSI_RUPR_MainVsi;
            toReg.I40E_VSI_TUPR_FCoEVsi = FromDcbRegs.I40E_VSI_TUPR_FCoEVsi;
            toReg.I40E_VSI_RUPR_FCoEVsi = FromDcbRegs.I40E_VSI_RUPR_FCoEVsi;
        }
    }
    #endregion
}
"@
#endregion CSharp_code

# SIG # Begin signature block
# MIIt8QYJKoZIhvcNAQcCoIIt4jCCLd4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCM5Jlr7XMXI4xy
# ovQHeqTjsb8STCSzX/UJOOHIVbKPIqCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCG1QwghtQAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgu19QzRfWaNZNOIRgoJ1f
# zLiPI0ZMlgiQKGGU6ay0LCkwDQYJKoZIhvcNAQEBBQAEggGAdb9SrV2JRTb/h1Ga
# 1LJ8PQ0UKoN2ZkDhpqrgHWGAgrljxs2sw+vGKxznN9ojgw55qwW5YPQ/NsocfhVe
# QXveR5LoY2Xxy/Lq4ozNyVjiaWdLtP1/YGaYt0ZdtZuAkz+/FZd+ckwrNBEl7/JV
# ORFrJzwvemekg8VmHjjAZkHaAUOSknr+xHo4KBkqrBVYJnw3GUXh+6vsise8uVmx
# Mt0T/GH23kNH9V/DyRRLGr6RcI1Qjy1EIoVr8YlVGSahMUeEXQh55ZPxfbSk/Kx8
# oQZCiykJDnrpE/x8o/KOlK/aLqcliSL4Wk0e+m9dkQz+luR+kDolAkl0Soebayfm
# P7DJuOGxtdo/Jtzo32Y15cld2iy96CqGxDedqt5Eeqjnzxy2PytdR9Hk1vmeeaEn
# 9fmveNdN3C7z03aj/jprdZJ4TFLn46/M+LYCugTaZ1+KjnYShC6dlEGNA5TDgrJy
# Y8ZZdGIFZ87cv8KtuLvGDJI0Rg5GNq9iSYp+BcvywY9mkW42oYIYzTCCGMkGCisG
# AQQBgjcDAwExghi5MIIYtQYJKoZIhvcNAQcCoIIYpjCCGKICAQMxDzANBglghkgB
# ZQMEAgIFADCB8wYLKoZIhvcNAQkQAQSggeMEgeAwgd0CAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQgLgOEiCJjdWserrdXl2dyc2S0Z9L7vUQM2Aa40a59
# A0MCFD+cdr4uKzyI+pvNdEqrtyzxKl7yGA8yMDI0MTIwNDE4MzAzNlqgcqRwMG4x
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBp
# bmcgU2lnbmVyIFIzNaCCEv8wggZdMIIExaADAgECAhA6UmoshM5V5h1l/MwS2OmJ
# MA0GCSqGSIb3DQEBDAUAMFUxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdv
# IExpbWl0ZWQxLDAqBgNVBAMTI1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcg
# Q0EgUjM2MB4XDTI0MDExNTAwMDAwMFoXDTM1MDQxNDIzNTk1OVowbjELMAkGA1UE
# BhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEwMC4GA1UEAxMnU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBTaWdu
# ZXIgUjM1MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAjdFn9MFIm739
# OEk6TWGBm8PY3EWlYQQ2jQae45iWgPXUGVuYoIa1xjTGIyuw3suUSBzKiyG0/c/Y
# n++d5mG6IyayljuGT9DeXQU9k8GWWj2/BPoamg2fFctnPsdTYhMGxM06z1+Ft0Ba
# v8ybww21ii/faiy+NhiUM195+cFqOtCpJXxZ/lm9tpjmVmEqpAlRpfGmLhNdkqiE
# uDFTuD1GsV3jvuPuPGKUJTam3P53U4LM0UCxeDI8Qz40Qw9TPar6S02XExlc8X1Y
# siE6ETcTz+g1ImQ1OqFwEaxsMj/WoJT18GG5KiNnS7n/X4iMwboAg3IjpcvEzw4A
# ZCZowHyCzYhnFRM4PuNMVHYcTXGgvuq9I7j4ke281x4e7/90Z5Wbk92RrLcS35hO
# 30TABcGx3Q8+YLRy6o0k1w4jRefCMT7b5mTxtq5XPmKvtgfPuaWPkGZ/tbxInyND
# A7YgOgccULjp4+D56g2iuzRCsLQ9ac6AN4yRbqCYsG2rcIQ5INTyI2JzA2w1vsAH
# PRbUTeqVLDuNOY2gYIoKBWQsPYVoyzaoBVU6O5TG+a1YyfWkgVVS9nXKs8hVti3V
# pOV3aeuaHnjgC6He2CCDL9aW6gteUe0AmC8XCtWwpePx6QW3ROZo8vSUe9AR7mMd
# u5+FzTmW8K13Bt8GX/YBFJO7LWzwKAUCAwEAAaOCAY4wggGKMB8GA1UdIwQYMBaA
# FF9Y7UwxeqJhQo1SgLqzYZcZojKbMB0GA1UdDgQWBBRo76QySWm2Ujgd6kM5LPQU
# ap4MhTAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAK
# BggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUF
# BwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwSgYDVR0fBEMw
# QTA/oD2gO4Y5aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGlt
# ZVN0YW1waW5nQ0FSMzYuY3JsMHoGCCsGAQUFBwEBBG4wbDBFBggrBgEFBQcwAoY5
# aHR0cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5n
# Q0FSMzYuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAN
# BgkqhkiG9w0BAQwFAAOCAYEAsNwuyfpPNkyKL/bJT9XvGE8fnw7Gv/4SetmOkjK9
# hPPa7/Nsv5/MHuVus+aXwRFqM5Vu51qfrHTwnVExcP2EHKr7IR+m/Ub7PamaeWfl
# e5x8D0x/MsysICs00xtSNVxFywCvXx55l6Wg3lXiPCui8N4s51mXS0Ht85fkXo3a
# uZdo1O4lHzJLYX4RZovlVWD5EfwV6Ve1G9UMslnm6pI0hyR0Zr95QWG0MpNPP0u0
# 5SHjq/YkPlDee3yYOECNMqnZ+j8onoUtZ0oC8CkbOOk/AOoV4kp/6Ql2gEp3bNC7
# DOTlaCmH24DjpVgryn8FMklqEoK4Z3IoUgV8R9qQLg1dr6/BjghGnj2XNA8ujta2
# JyoxpqpvyETZCYIUjIs69YiDjzftt37rQVwIZsfCYv+DU5sh/StFL1x4rgNj2t8G
# ccUfa/V3iFFW9lfIJWWsvtlC5XOOOQswr1UmVdNWQem4LwrlLgcdO/YAnHqY52Qw
# nBLiAuUnuBeshWmfEb5oieIYMIIGFDCCA/ygAwIBAgIQeiOu2lNplg+RyD5c9Mfj
# PzANBgkqhkiG9w0BAQwFADBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5n
# IFJvb3QgUjQ2MB4XDTIxMDMyMjAwMDAwMFoXDTM2MDMyMTIzNTk1OVowVTELMAkG
# A1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAxMjU2Vj
# dGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEB
# AQUAA4IBjwAwggGKAoIBgQDNmNhDQatugivs9jN+JjTkiYzT7yISgFQ+7yavjA6B
# g+OiIjPm/N/t3nC7wYUrUlY3mFyI32t2o6Ft3EtxJXCc5MmZQZ8AxCbh5c6WzeJD
# B9qkQVa46xiYEpc81KnBkAWgsaXnLURoYZzksHIzzCNxtIXnb9njZholGw9djnjk
# TdAA83abEOHQ4ujOGIaBhPXG2NdV8TNgFWZ9BojlAvflxNMCOwkCnzlH4oCw5+4v
# 1nssWeN1y4+RlaOywwRMUi54fr2vFsU5QPrgb6tSjvEUh1EC4M29YGy/SIYM8ZpH
# admVjbi3Pl8hJiTWw9jiCKv31pcAaeijS9fc6R7DgyyLIGflmdQMwrNRxCulVq8Z
# pysiSYNi79tw5RHWZUEhnRfs/hsp/fwkXsynu1jcsUX+HuG8FLa2BNheUPtOcgw+
# vHJcJ8HnJCrcUWhdFczf8O+pDiyGhVYX+bDDP3GhGS7TmKmGnbZ9N+MpEhWmbiAV
# PbgkqykSkzyYVr15OApZYK8CAwEAAaOCAVwwggFYMB8GA1UdIwQYMBaAFPZ3at0/
# /QET/xahbIICL9AKPRQlMB0GA1UdDgQWBBRfWO1MMXqiYUKNUoC6s2GXGaIymzAO
# BgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggr
# BgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwTAYDVR0fBEUwQzBBoD+gPYY7aHR0
# cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nUm9v
# dFI0Ni5jcmwwfAYIKwYBBQUHAQEEcDBuMEcGCCsGAQUFBzAChjtodHRwOi8vY3J0
# LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3RhbXBpbmdSb290UjQ2LnA3
# YzAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcN
# AQEMBQADggIBABLXeyCtDjVYDJ6BHSVY/UwtZ3Svx2ImIfZVVGnGoUaGdltoX4hD
# skBMZx5NY5L6SCcwDMZhHOmbyMhyOVJDwm1yrKYqGDHWzpwVkFJ+996jKKAXyIIa
# Uf5JVKjccev3w16mNIUlNTkpJEor7edVJZiRJVCAmWAaHcw9zP0hY3gj+fWp8MbO
# ocI9Zn78xvm9XKGBp6rEs9sEiq/pwzvg2/KjXE2yWUQIkms6+yslCRqNXPjEnBnx
# uUB1fm6bPAV+Tsr/Qrd+mOCJemo06ldon4pJFbQd0TQVIMLv5koklInHvyaf6vAT
# JP4DfPtKzSBPkKlOtyaFTAjD2Nu+di5hErEVVaMqSVbfPzd6kNXOhYm23EWm6N2s
# 2ZHCHVhlUgHaC4ACMRCgXjYfQEDtYEK54dUwPJXV7icz0rgCzs9VI29DwsjVZFpO
# 4ZIVR33LwXyPDbYFkLqYmgHjR3tKVkhh9qKV2WCmBuC27pIOx6TYvyqiYbntinmp
# Oqh/QPAnhDgexKG9GX/n1PggkGi9HCapZp8fRwg8RftwS21Ln61euBG0yONM6noD
# 2XQPrFwpm3GcuqJMf0o8LLrFkSLRQNwxPDDkWXhW+gZswbaiie5fd/W2ygcto78X
# CSPfFWveUOSZ5SqK95tBO8aTHmEa4lpJVD7HrTEn9jb1EGvxOb1cnn0CMIIGgjCC
# BGqgAwIBAgIQNsKwvXwbOuejs902y8l1aDANBgkqhkiG9w0BAQwFADCBiDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBD
# aXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVT
# RVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMjEwMzIyMDAw
# MDAwWhcNMzgwMTE4MjM1OTU5WjBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1w
# aW5nIFJvb3QgUjQ2MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAiJ3Y
# uUVnnR3d6LkmgZpUVMB8SQWbzFoVD9mUEES0QUCBdxSZqdTkdizICFNeINCSJS+l
# V1ipnW5ihkQyC0cRLWXUJzodqpnMRs46npiJPHrfLBOifjfhpdXJ2aHHsPHggGsC
# i7uE0awqKggE/LkYw3sqaBia67h/3awoqNvGqiFRJ+OTWYmUCO2GAXsePHi+/JUN
# Aax3kpqstbl3vcTdOGhtKShvZIvjwulRH87rbukNyHGWX5tNK/WABKf+Gnoi4cmi
# sS7oSimgHUI0Wn/4elNd40BFdSZ1EwpuddZ+Wr7+Dfo0lcHflm/FDDrOJ3rWqauU
# P8hsokDoI7D/yUVI9DAE/WK3Jl3C4LKwIpn1mNzMyptRwsXKrop06m7NUNHdlTDE
# MovXAIDGAvYynPt5lutv8lZeI5w3MOlCybAZDpK3Dy1MKo+6aEtE9vtiTMzz/o2d
# YfdP0KWZwZIXbYsTIlg1YIetCpi5s14qiXOpRsKqFKqav9R1R5vj3NgevsAsvxsA
# nI8Oa5s2oy25qhsoBIGo/zi6GpxFj+mOdh35Xn91y72J4RGOJEoqzEIbW3q0b2iP
# uWLA911cRxgY5SJYubvjay3nSMbBPPFsyl6mY4/WYucmyS9lo3l7jk27MAe145GW
# xK4O3m3gEFEIkv7kRmefDR7Oe2T1HxAnICQvr9sCAwEAAaOCARYwggESMB8GA1Ud
# IwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0GA1UdDgQWBBT2d2rdP/0BE/8W
# oWyCAi/QCj0UJTAOBgNVHQ8BAf8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zATBgNV
# HSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBF
# oEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRp
# ZmljYXRpb25BdXRob3JpdHkuY3JsMDUGCCsGAQUFBwEBBCkwJzAlBggrBgEFBQcw
# AYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# Dr5lQe1oRLjlocXUEYfktzsljOt+2sgXke3Y8UPEooU5y39rAARaAdAxUeiX1ktL
# J3+lgxtoLQhn5cFb3GF2SSZRX8ptQ6IvuD3wz/LNHKpQ5nX8hjsDLRhsyeIiJsms
# 9yAWnvdYOdEMq1W61KE9JlBkB20XBee6JaXx4UBErc+YuoSb1SxVf7nkNtUjPfcx
# uFtrQdRMRi/fInV/AobE8Gw/8yBMQKKaHt5eia8ybT8Y/Ffa6HAJyz9gvEOcF1VW
# XG8OMeM7Vy7Bs6mSIkYeYtddU1ux1dQLbEGur18ut97wgGwDiGinCwKPyFO7Apcm
# VJOtlw9FVJxw/mL1TbyBns4zOgkaXFnnfzg4qbSvnrwyj1NiurMp4pmAWjR+Pb/S
# IduPnmFzbSN/G8reZCL4fvGlvPFk4Uab/JVCSmj59+/mB2Gn6G/UYOy8k60mKcma
# AZsEVkhOFuoj4we8CYyaR9vd9PGZKSinaZIkvVjbH/3nlLb0a7SBIkiRzfPfS9T+
# JesylbHa1LtRV9U/7m0q7Ma2CQ/t392ioOssXW7oKLdOmMBl14suVFBmbzrt5V5c
# QPnwtd3UOTpS9oCG+ZZheiIvPgkDmA8FzPsnfXW5qHELB43ET7HHFHeRPRYrMBKj
# kb8/IN7Po0d0hQoF4TeMM+zYAJzoKQnVKOLg8pZVPT8xggSRMIIEjQIBATBpMFUx
# CzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMT
# I1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0EgUjM2AhA6UmoshM5V5h1l
# /MwS2OmJMA0GCWCGSAFlAwQCAgUAoIIB+TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcN
# AQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI0MTIwNDE4MzAzNlowPwYJKoZIhvcNAQkE
# MTIEMBmHcoKSNBDkzp0EuQN3zbCRY11SV4lYX/jLXr3i+qEyV4y7gF8vCwmp6gjF
# eyEYfDCCAXoGCyqGSIb3DQEJEAIMMYIBaTCCAWUwggFhMBYEFPhgmBmm+4gs9+hS
# l/KhGVIaFndfMIGHBBTGrlTkeIbxfD1VEkiMacNKevnC3TBvMFukWTBXMQswCQYD
# VQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0
# aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2AhB6I67aU2mWD5HIPlz0
# x+M/MIG8BBSFPWMtk4KCYXzQkDXEkd6SwULaxzCBozCBjqSBizCBiDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5
# MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJU
# cnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDbCsL18Gzrno7PdNsvJ
# dWgwDQYJKoZIhvcNAQEBBQAEggIAcxQeH3+dkwKufiQ5v91A213EBhibriZUxFdG
# +e9E6lMdIJG9a+i9JeS5/BoEUmAtGF1FQTz0ef/psYgYH/KXMpOBew5i4W3chjif
# YvHe2qqUFElECoKCvh8MOIOkGwsAVo7FoSUQzfymHuCOu+h9IQ5eX4hF2cTjdU0N
# s2nDgga+uf647UE4jNjVUF07IvKIkwYKfLiIqCHqKZtWnbMN8kwIfDPKNIS13MkB
# tKWExivxJew52zXz9WB2awd82MzzQnWM8KDDRVUj83w2pKnvZKwWxdFPoHlWFqsi
# GWdU4S6CW+5hkpTpBL5O+Uf9zFJ0as1U2RFrTEVnsWRW2VING10AZ29XPykHV5N7
# qhxgfXqU3GSyIsjrRHHJoy2cc4GEKMcrLjRbHyz9JcwNiz7v8P18RbKH2UY1qNmh
# c7M2mH/NcR+wF9mZ7CvplWH4wXV/pL32tvsrHs4W7cVMxFRACwgnwRvEL0i0HIsZ
# o+VTmPfJnMBSAdMT3gt+Ibe+ZzIkqVPwSAIBWv9PWYNwy/3RPn7DoD8cPc91Bg8k
# tEQiGqsVhXVvD2WOA2oN7quHs2GI/QISxOaCtWLPM+FOSUPBNFy7oyCOVuQqcjZR
# 3mLNzkFYOfsq+6ON8kngPVTnsraYSBLclLz27/Jix5gBU+yIG6NknEB4MSDCoakW
# /zr2qRk=
# SIG # End signature block
