# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")] Param()

# Functions common to all cmdlets
Function ValidateGetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    if ((-Not $Adapters) -and $AdapterName)
    {
        foreach ($n in $AdapterName)
        {
            $TmpAdapterNames = @()
            try
            {
                $PhysicalAdapterArray = $script:PnpDevice.Where({ $_.Name -Like $n })
                $IntelAdapterArray = $PhysicalAdapterArray.Where({ $_.Manufacturer -eq "Intel" })

                if ($IntelAdapterArray)
                {
                    $TmpAdapterNames = $IntelAdapterArray.Name
                }
                elseif ($PhysicalAdapterArray)
                {
                    # Add non-Intel devices for warning display
                    $TmpAdapterNames = $PhysicalAdapterArray.Name
                }
            }
            catch
            {
                # Failed due to Adapters passed to Name parameter
                $AdapterNames = @()
                $ErrorMessages.Value += $Messages.InvalidParams
                break
            }

            if (-Not $TmpAdapterNames)
            {
                $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                continue
            }
            $AdapterNames += $TmpAdapterNames
        }
    }
    elseif ((-Not $AdapterName) -and $Adapters)
    {
        foreach ($a in $Adapters)
        {
            if (CheckPropertyExists $a "CreationClassName")
            {
                if ($a.CreationClassName -eq "MSFT_NetAdapter")
                {
                    $AdapterNames += $a.InterfaceDescription
                }
                elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                    $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                    $a.CreationClassName -eq "Win32_PnpEntity")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
            elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
            {
                $AdapterNames += $a.Name
            }
            else
            {
                $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
            }
        }
    }
    elseif (-Not ($AdapterName -and $Adapters))
    {
        $AdapterNames = ($script:PnpDevice.Where({ $_.Manufacturer -eq "Intel" })).Name
    }
    elseif ($AdapterName -and $Adapters)
    {
        $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
    }

    return $AdapterNames
}

Function ValidateSetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    do
    {
        if ($AdapterName -and $Adapters)
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
            break
        }
        elseif ($AdapterName)
        {
            foreach ($n in $AdapterName)
            {
                $TmpAdapterNames = @()
                try
                {
                    $PhysicalAdapterArray = $script:PnpDevice.Where({ $_.Name -Like $n })
                    $IntelAdapterArray = $PhysicalAdapterArray.Where({ $_.Manufacturer -eq "Intel" })

                    if ($IntelAdapterArray)
                    {
                        $TmpAdapterNames = $IntelAdapterArray.Name
                    }
                    elseif ($PhysicalAdapterArray)
                    {
                        # Add non-Intel devices for warning display
                        $TmpAdapterNames = $PhysicalAdapterArray.Name
                    }
                }
                catch
                {
                    # Failed due to Adapters passed to Name parameter
                    $AdapterNames = @()
                    $ErrorMessages.Value += $Messages.InvalidParams
                    break
                }

                if (-Not $TmpAdapterNames)
                {
                    $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                    continue
                }
                $AdapterNames += $TmpAdapterNames
            }
        }
        elseif ($Adapters)
        {
            foreach ($a in $Adapters)
            {
                if (CheckPropertyExists $a "CreationClassName")
                {
                    if ($a.CreationClassName -eq "MSFT_NetAdapter")
                    {
                        $AdapterNames += $a.InterfaceDescription
                    }
                    elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                        $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                        $a.CreationClassName -eq "Win32_PnpEntity")
                    {
                        $AdapterNames += $a.Name
                    }
                    else
                    {
                        $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                    }
                }
                elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterOrName
            break
        }
    } while ($false)

    return $AdapterNames
}

Function ValidatePathParams([ref]$LogPath, $UseDefaultPath, $LogName, [ref]$ErrorMessages)
{
    $Result = $true

    try
    {
        if ($UseDefaultPath)
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\" + $LogName

            if (-not (Test-Path -Path $DefaultPath -ErrorAction Stop))
            {
                New-Item -Path $ENV:LOCALAPPDATA -Name "\Intel\Wired Networking" -ItemType "directory" -ErrorAction SilentlyContinue
            }
            $LogPath.Value = $DefaultPath
        }
        else
        {
            $LogPath.Value = $Path
        }

        $PathRoot = GetPSDriveRoot $LogPath.Value
        if (-Not [string]::IsNullOrEmpty($PathRoot))
        {
            $PathRoot = $PathRoot.TrimEnd("\")
            $strPathNoQualifier = Split-Path $LogPath.Value -NoQualifier
            $LogPath.Value = $PathRoot + $strPathNoQualifier
        }

        $isPathFile = Test-Path -Path $LogPath.Value -PathType Leaf -ErrorAction Stop

        if (($isPathFile) -and (-not $Append) -and (-not $Force))
        {
            $ErrorMessages.Value += $Messages.LogmanFileExists -f $AdapterName
            $Result = $false
        }
        elseif (-not $isPathFile)
        {
            if (Test-Path -Path $LogPath.Value -ErrorAction Stop)
            {
                $ErrorMessages.Value += $Messages.FolderFileNameExits
                $Result = $false
            }
            else
            {
                $strAbsolutePath = [IO.Path]::GetFullPath($LogPath.Value)
                $strParentFolder = Split-Path -Path $strAbsolutePath

                if (-Not (Test-Path -Path $strParentFolder -ErrorAction Stop))
                {
                    $ErrorMessages.Value += $Messages.PathIncorrect
                    $Result = $false
                }
            }
        }
    }
    catch
    {
        $ErrorMessages.Value += $Messages.PathIncorrect
        $Result = $false
    }
    return $Result
}

function GetPSDriveRoot($Path)
{
    $strQualifier = Split-Path -Path $Path -Qualifier -ErrorAction SilentlyContinue
    if (-Not [string]::IsNullOrEmpty($strQualifier))
    {
        $strPSDriveName = $strQualifier.TrimEnd(":")
        $CurrentPSDrive = Get-PSDrive -Name $strPSDriveName -ErrorAction SilentlyContinue
        if ($null -ne $CurrentPSDrive)
        {
            return $CurrentPSDrive.Root
        }
    }

    return $null
}

function InvokeCimMethod($ClassName, $InstanceName = "", $MethodName, $params = @{}, $Namespace = "root\wmi")
{
    $query = "Select * from $ClassName"
    if ($InstanceName)
    {
        $query += " where instancename like '$InstanceName'"
    }

    Invoke-CimMethod -Query $query -MethodName $MethodName -Arguments $params -Namespace $Namespace -ErrorAction SilentlyContinue
}

function GetIntelEthernetDevices($AdditionalDriverArray = $null, $RemoveDriverArray = $null)
{
    # List of drivers that should have majority of cmdlet support
    $SupportedDrivers = @('icea', 'iceb', 'scea', 'sceb', 'i40ea', 'i40eb', 'ixw')

    # Add additional driver names that should be supported for the cmdlet calling this
    if ($AdditionalDriverArray)
    {
        $SupportedDrivers += $AdditionalDriverArray
    }

    # Remove driver names that should NOT be supported for the cmdlet calling this
    if ($RemoveDriverArray)
    {
        $SupportedDrivers = $SupportedDrivers | Where-Object { $_ -notin $RemoveDriverArray }
    }

    $script:SupportedAdapters = @()
    $script:PnpDevice = @(Get-PnpDevice | Where-Object { $_.Class -eq "Net" } -ErrorAction SilentlyContinue)
    if ($script:PnpDevice)
    {
        foreach ($Driver in $SupportedDrivers)
        {
            $script:SupportedAdapters += $script:PnpDevice.Where({ $_.Service -like $Driver })
        }
    }
}

function GetSupportedAdaptersByDriver([array]$strSupportedDriverArray)
{
    return $script:SupportedAdapters.Where({ $_.Service -in $strSupportedDriverArray })
}

function GetSupportedAdapters($AdapterNames, [ref]$WarningMessages)
{
    $SupportedAdapterNames = @()
    $AdapterNames = $AdapterNames | Sort-Object
    foreach ($a in $AdapterNames)
    {
        if ($script:SupportedAdapters.Name -Contains $a)
        {
            $SupportedAdapterNames += $a
        }
        else
        {
            $WarningMessages.Value += $Messages.NoCmdletSupport -f $a
        }
    }

    return $SupportedAdapterNames
}

Function GetOSSpecificDriverName($DriverName)
{
    $WIN10_RS5_BUILD = 17682
    $VER_NT_SERVER = 3

    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iMinorVersion = $OSVersion.Minor
    $iBuildNumber = $OSVersion.Build

    $strOSSpecificDriverName = ''

    if (10 -eq $iMajorVersion -and 0 -eq $iMinorVersion)
    {
        if ($VER_NT_SERVER -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
        {
            $bIsWin2016ServerAndHigher = $true
        }
    }

    if ($bIsWin2016ServerAndHigher)
    {
        if (IsWin2022ServerOrLater)
        {
            $strOSSpecificDriverName = $DriverName
        }
        elseif ($iBuildNumber -ge $WIN10_RS5_BUILD)
        {
            $strOSSpecificDriverName = $DriverName + "68"
        }
        else
        {
            $strOSSpecificDriverName = $DriverName + "65"
        }
    }

    return $strOSSpecificDriverName
}

function GetDriverInfParentFolder($strDriverName, $Verb)
{
    $strDriverStorePathArray = Get-WindowsDriver -Online -ErrorAction SilentlyContinue | Select-Object OriginalFileName

    if ($null -eq $strDriverStorePathArray)
    {
        switch ($Verb)
        {
            Start {$script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
            Register
            {
                $script:ErrorMessagesRegister += $Messages.RegisterCountersFailure -f $a
                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                break
            }
        }
    }

    # The system keeps a history of installed drivers, Get the latest installed driver inf path
    $strPathToInf = Get-ChildItem -Path $strDriverStorePathArray.OriginalFileName.Where({ $_ -like "*$strDriverName*" }) | Sort-Object -Descending -Property "CreationTime" | Select-Object -First 1

    [System.IO.Path]::GetDirectoryName($strPathToInf)
}

function CheckDeviceError($AdapterName)
{
    $SupportedAdapter = $script:SupportedAdapters.Where({ $_.FriendlyName -eq $AdapterName })

    if ($SupportedAdapter)
    {
        # if the device is not 'working properly'
        if ([Int32]$SupportedAdapter.ConfigManagerErrorCode -ne 0)
        {
            $PreErrorActionPreference = $global:ErrorActionPreference
            $global:ErrorActionPreference = 'SilentlyContinue'
            # Workaround due to ProblemDescription being empty by default - change current path
            Push-Location -Path (Get-Module -Name PnPDevice).ModuleBase
            $StatusMsg = $AdapterName + ": " + $SupportedAdapter.ProblemDescription
            # Reset path
            Pop-Location
            $global:ErrorActionPreference = $PreErrorActionPreference
            $StatusMsg
        }
    }
}

function ValidateSingleAdapter([array]$PipelineInput, [array]$AdapterName, [ref]$ErrorMessages)
{
    $Result = $false

    do
    {
        if ($PipelineInput.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        if ($AdapterName.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        $Result = $true
    } while ($false)

    return $Result
}

function IsWin2022ServerOrLater()
{
    $WIN2022_BUILD = 20298
    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iBuildNumber = $OSVersion.Build

    if (($iMajorVersion -eq 10) -and ($iBuildNumber -ge $WIN2022_BUILD) -and -not (IsOperatingSystemClientBased))
    {
        return $true
    }

    return $false
}

function IsOperatingSystemClientBased()
{
    $VER_NT_WORKSTATION = 1
    return ($VER_NT_WORKSTATION -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
}

function CheckPropertyExists($TestObject, $strPropertyName)
{
    $bReturnValue = $true
    try
    {
        return (($TestObject | Get-Member -MemberType "Property" -ErrorAction Stop).Name -contains $strPropertyName)
    }
    catch
    {
        $bReturnValue = $false
    }
    return $bReturnValue
}

function GetAdapterSettings($AdapterName, $DisplayName, $RegistryKeyword)
{
    $AdapterSettings = @()
    $SettingsRet = @()
    if ([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword))
    {
        $AdapterSettings = GetAllAdapterSettings $AdapterName
        $SettingsRet += InitializeProfileSetting $AdapterName
    }
    else
    {
        # Need to handle Profile differently, this is an EthernetCmdlets created setting
        if ($DisplayName -eq 'Profile' -or $RegistryKeyword -eq 'PerformanceProfile')
        {
            $SettingsRet += InitializeProfileSetting $AdapterName

            # Remove Profile from passed in array to avoid GatherSettingEnumOutput (InitializeProfileSetting handles it uniquely)
            if ($null -ne $DisplayName)
            {
                $DisplayName = $DisplayName -ne 'Profile'
            }
            if ($null -ne $RegistryKeyword)
            {
                $RegistryKeyword = $RegistryKeyword -ne 'PerformanceProfile'
            }
        }

        $AdapterSettings = GetSettings $AdapterName $DisplayName $RegistryKeyword
    }

    foreach ($Setting in $AdapterSettings)
    {
        switch ([int]($Setting.DisplayParameterType))
        {
            {$_ -in [int][DisplayParameterType]::int,
            [int][DisplayParameterType]::long,
            [int][DisplayParameterType]::word,
            [int][DisplayParameterType]::dword}
            { $SettingsRet += GatherSettingIntOutput $a $Setting; break }
            {$_ -in [int][DisplayParameterType]::enum,
            [int][DisplayParameterType]::edit}
            {$SettingsRet += GatherSettingEnumOutput $a $Setting; break}
            default {$SettingsRet += GatherSettingEnumOutput $a $Setting; break}
        }
    }

    return $SettingsRet
}

function GetAllAdapterSettings($AdapterName)
{
    return $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName})
}

function GetSettings($AdapterName, $DisplayName, $RegistryKeyword)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            if ($DisplayName -eq 'Profile')
            {
                continue
            }
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName})
            if (-not $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            if ($RegistryKeyword -eq 'PerformanceProfile')
            {
                continue
            }
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword})
            if (-not $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    return $SettingArray
}

function GatherSettingEnumOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting.RegistryKeyword)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DescriptionMap  = $Setting.ValidDisplayValues
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        PossibleValues  = $Setting.ValidRegistryValues
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
    }
}

function GatherSettingIntOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting.RegistryKeyword)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingInt'
        Base            = $Setting.NumericParameterBaseValue
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
        Min             = $Setting.NumericParameterMinValue
        Max             = $Setting.NumericParameterMaxValue
        Step            = $Setting.NumericParameterStepValue
    }
}

function GetSettingMinihelp($RegistryKeyword)
{
    $SettingMiniHelp = $MiniHelp.$($RegistryKeyword)
    if ([string]::IsNullOrEmpty($SettingMiniHelp))
    {
        $SettingMiniHelp = $Messages.MiniHelpNotFound
    }

    return $SettingMiniHelp
}

function InitializeProfileSetting($AdapterName)
{
    # for our supported adapters, by default its safe to add: Standard Server, Web Server, and Low Latency
    # PossibleValues and DescriptionMap for Profile
    $ProfileValuesMap = @{ '2' = $Messages.StandardServer; '3' = $Messages.WebServer; '6' = $Messages.LowLatency }
    $DefaultValue = '2'

    $IntelDCBInstalled = $false
    $HyperVInstalled = $false

    $IntelDCB = Get-Service | Where-Object {$_.Name -eq "IntelDCB"}
    if ($null -ne $IntelDCB)
    {
        $IntelDCBInstalled = $true
    }

    # ID 20 represents Hyper-V
    $HyperV = Get-CimInstance -Namespace "root\cimv2" -ClassName "Win32_ServerFeature" | Where-Object {$_.ID -eq '20'}
    if ($null -ne $HyperV)
    {
        $HyperVInstalled = $true
    }

    if ($IntelDCBInstalled)
    {
        $ProfileValuesMap.Add('4', $Messages.StorageServer)
        $DefaultValue = '4'
    }
    if ($HyperVInstalled)
    {
        $ProfileValuesMap.Add('5', $Messages.VirtServer)
        $DefaultValue = '5'
    }
    if ($IntelDCBInstalled -and $HyperVInstalled)
    {
        $ProfileValuesMap.Add('7', $Messages.StorageVirt)
        $DefaultValue = '7'
    }

    $CurrentProfile = GetCurrentProfileValue $AdapterName
    $CurrentProfileRegistryValue = $null
    $CurrentProfileDisplayValue = $null
    if ($null -ne $CurrentProfile)
    {
        $CurrentProfileRegistryValue = $CurrentProfile.RegistryValue
        $CurrentProfileDisplayValue = $CurrentProfile.DisplayValue
        if ($CurrentProfileRegistryValue -eq 1)
        {
            $ProfileValuesMap.Add($CurrentProfileRegistryValue, $CurrentProfileDisplayValue)
        }
    }
    $ProfileValuesMap = $ProfileValuesMap.GetEnumerator() | Sort-Object -Property Name
    $SettingMiniHelp = GetSettingMinihelp("PerformanceProfile")
    $ParentId = $script:MSNetAdvProperty.Where({ $_.InterfaceDescription -eq $AdapterName}).InstanceID.split(':')[0]

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = 'PerformanceProfile'
        CurrentValue    = $CurrentProfile.RegistryValue
        DefaultValue    = $DefaultValue
        Description     = $Messages.Profile
        DescriptionMap  = $ProfileValuesMap.Value
        DisplayName     = $Messages.Profile
        DisplayValue    = $CurrentProfileDisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $ParentId
        PossibleValues  = $ProfileValuesMap.GetEnumerator().Name
        RegistryKeyword = 'PerformanceProfile'
        RegistryValue   = $CurrentProfileRegistryValue
    }
}

function GetCurrentProfileValue($AdapterName)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName

    if ($null -ne $AdapterRegKey.PerformanceProfile)
    {
        $ProfileRegValue = $AdapterRegKey.PerformanceProfile
        # Profile is already custom settings or in case one of the associated Profile settings got changed
        if ($ProfileRegValue -eq '1' -or ($false -eq (IsProfileStillApplied $ProfileRegValue $AdapterName)))
        {
            $ProfileRegValue = '1'
            $ProfileDisplayVal = $Messages.CustomSettings
        }
        else
        {
            $ProfileDisplayVal = $PROFILE_VALUE_TO_NAME[$ProfileRegValue]
        }

        return [PsCustomObject] @{
            # return the key from the hashtable
            DisplayValue = $ProfileDisplayVal
            RegistryValue = $ProfileRegValue }
    }
}

function IsProfileStillApplied($RegistryValue, $AdapterName)
{
    $TmpProfileName = $PROFILE_VALUE_TO_NAME[$RegistryValue]
    if ($null -ne $TmpProfileName)
    {
        $DriverFamily = $script:SupportedAdapters.Where({ $_.Name -eq $AdapterName }).Service
        $tmp = GetProfileSettingsFromXml $TmpProfileName $DriverFamily
        foreach ($ProfileSetting in $tmp.GetEnumerator())
        {
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq $ProfileSetting.Key})
            if ($TmpSetting.RegistryValue -ne $ProfileSetting.Value)
            {
                return $false
            }
        }
    }

    return $true
}

function GetProfileSettingsFromXml()
{
    Param
    (
        [parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [string]
        $ProfileToSet,

        [parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [ValidateSet("i40ea", "i40eb", "icea", "iceb", "scea", "sceb", "ixw")]
        [string]
        $AdapterFamily
    )

    $Profile_to_XmlProfile = @{
        "Standard Server" = "standard"
        "Web Server" = "web"
        "Storage Server" = "storage"
        "Virtualization Server (Hyper-V*)" = "hyperv"
        "Low Latency" = "lowlatency"
        "Storage + Virtualization" = "storagevirtualization"
    }

    $OSBuild = [Environment]::OSVersion.Version.Build

    # Each Key treated as a range of Build Numbers (14393 to 16299 is NDIS v6.60)
    $OSBuild_to_NDIS = [ordered]@{
        14393 = "ndis660"
        16299 = "ndis680"
    }

    $Adapter_to_ProfileString = @{
        i40ea = "40gig"
        i40eb = "40gig"
        icea  = "E8XX"
        iceb  = "E8XX"
        scea  = "E8XX"
        sceb  = "E8XX"
        ixw   = "10gig"
    }

    $Adapter_to_ProfileDeviceString = @{
        i40ea = "X710"
        i40eb = "X722"
    }

    [xml]$Xml = Get-Content -Path $PSScriptRoot\PerformanceProfiles.xml
    # validate XML against XSD before using it:
    try
    {
        $Xml.Schemas.Add('', "$PSScriptRoot\PerformanceProfiles.xsd") | Out-Null
        $Xml.Validate($null)
    }
    catch [System.Xml.Schema.XmlSchemaValidationException]
    {
        $script:ErrorMessagesProfile += $Messages.ProfileXmlError
        return
    }

    $PerfProfiles = $Xml.SelectNodes("//ProfileList/Profile")

    # Convert cmdlet parameters to strings from xml
    $AdapterFamilyProfile = $Adapter_to_ProfileString[$AdapterFamily]

    foreach ($CurrentBuildNumber in $OSBuild_to_NDIS.Keys)
    {
        if ($OSBuild -gt $CurrentBuildNumber)
        {
            $NDISVersion = $OSBuild_to_NDIS.$CurrentBuildNumber
        }
    }

    $ProfileName = $Profile_to_XmlProfile[$ProfileToSet]

    # Construct profile names
    # Top-level, eg 40gig_standard
    $TopLevelProfile = $AdapterFamilyProfile + "_" + $ProfileName
    Write-Verbose $TopLevelProfile

    # + NDIS version, eg 40gig_standard.ndis660
    $NdisProfile = $AdapterFamilyProfile + "_" + $ProfileName + "." + $NDISVersion
    Write-Verbose $NdisProfile

    # + device name, eg 40gig_standard.ndis660.X722
    # adding '*' at the end so it matches .no_npar too
    $DeviceProfile = $AdapterFamilyProfile + "_" + $ProfileName + "." + $NDISVersion + "." + $Adapter_to_ProfileDeviceString[$AdapterFamily] + "*"
    Write-Verbose $DeviceProfile

    $TopLevelProfile = $PerfProfiles | Where-Object { $_.Id -like  $TopLevelProfile }
    $NdisProfile     = $PerfProfiles | Where-Object { $_.Id -like  $NdisProfile }
    $DeviceProfile   = $PerfProfiles | Where-Object { $_.Id -like  $DeviceProfile }

    # assemble it all together - from generic to detailed profile
    # ie DeviceProfile settings should take precedence over those from TopLevelProfile
    $Profiles = @($TopLevelProfile, $NdisProfile, $DeviceProfile)
    $RetVal = @{}

    foreach ($Profile in $Profiles)
    {
        foreach($Setting in $Profile.Setting)
        {
            $RetVal[$Setting.id] = $Setting.'#text'
        }
    }

    return $RetVal
}

function GetAdapterPropertiesFromRegistry($AdapterName)
{
    # Individual Adapter GUID  - (Get-NetAdapter -InterfaceDescription $AdapterName | Where-Object {$_.InterfaceDescription -eq $AdapterName}).InterfaceGuid
    $AdapterInstanceID = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    $972Key = Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -ErrorAction SilentlyContinue
    $AdapterRegKey = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterInstanceID})
    return $AdapterRegKey
}

function SetAdapterPropertyInRegistry($AdapterName, $Property, $Value)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName
    Set-ItemProperty -Path $AdapterRegKey.PSPath -Name $Property $Value -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIt8gYJKoZIhvcNAQcCoIIt4zCCLd8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBO80c+EtQuOmQi
# 8sfyTGIdxhkFEOdal3TnyBx+w8waLKCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCG1UwghtRAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQg4jczZgptvImiO+GFAPZ0
# 3zJ8zDPMBPQj9c1d38ae2kwwDQYJKoZIhvcNAQEBBQAEggGAoxrgcxJEx5hns6Vg
# m0LAbdA6r95HCnwJkhxL3iEjpunShUVWm/8rfeUhdoy7DNzOqShvQVXnS7nwHFHo
# WRzChdEY2hzL8a/vzR1/bjrdCcA3Pr1DeH6ScutnYImQcogIil+Rj766TqKEXVMH
# DzGotOPilXKYPCUmhP/+gKdQ3N9RqlsHRtTAwkII2FJ/boGhhs9fX0bevvKr2BOu
# X82Cz5D4qHD/bQ8M5kDwXhYtJJlVR6XSJcYch7XPgbYxxYC5RhD1D3eltagw6MYA
# YL86D7SXKMEE6BCvdJ+N3Ry5gjYT/Ao+GUjXYKpU0qUj38EOzDqzpRJZHAb0KIpv
# 4nOHnL6WAPFFbNEHIGmXqU5UUErbi3saPOQkj67NllKE/gjtMcPFMTideppRHBw/
# V07jh3sL75kpGeTd8VJlqwfzjUMT6M8VPuiLWRtkNS+bWDzghri5UypJf8qmofla
# 1osPLfTYZwFh51SKJjLGDs/o6WyO8nJr9rEXHlX31HQANkUkoYIYzjCCGMoGCisG
# AQQBgjcDAwExghi6MIIYtgYJKoZIhvcNAQcCoIIYpzCCGKMCAQMxDzANBglghkgB
# ZQMEAgIFADCB9AYLKoZIhvcNAQkQAQSggeQEgeEwgd4CAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQgj8xrG8a1sYl5weqgpYbOhN2U2Gbv+mN9DODEZtlQ
# 8xoCFQCRwfxcvLeF6ux6My7WTimx3np9RxgPMjAyNDEyMDQxODMwMTBaoHKkcDBu
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMTAwLgYDVQQDEydTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1w
# aW5nIFNpZ25lciBSMzWgghL/MIIGXTCCBMWgAwIBAgIQOlJqLITOVeYdZfzMEtjp
# iTANBgkqhkiG9w0BAQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMSwwKgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5n
# IENBIFIzNjAeFw0yNDAxMTUwMDAwMDBaFw0zNTA0MTQyMzU5NTlaMG4xCzAJBgNV
# BAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgU2ln
# bmVyIFIzNTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAI3RZ/TBSJu9
# /ThJOk1hgZvD2NxFpWEENo0GnuOYloD11BlbmKCGtcY0xiMrsN7LlEgcyoshtP3P
# 2J/vneZhuiMmspY7hk/Q3l0FPZPBllo9vwT6GpoNnxXLZz7HU2ITBsTNOs9fhbdA
# Wr/Mm8MNtYov32osvjYYlDNfefnBajrQqSV8Wf5ZvbaY5lZhKqQJUaXxpi4TXZKo
# hLgxU7g9RrFd477j7jxilCU2ptz+d1OCzNFAsXgyPEM+NEMPUz2q+ktNlxMZXPF9
# WLIhOhE3E8/oNSJkNTqhcBGsbDI/1qCU9fBhuSojZ0u5/1+IjMG6AINyI6XLxM8O
# AGQmaMB8gs2IZxUTOD7jTFR2HE1xoL7qvSO4+JHtvNceHu//dGeVm5Pdkay3Et+Y
# Tt9EwAXBsd0PPmC0cuqNJNcOI0XnwjE+2+Zk8bauVz5ir7YHz7mlj5Bmf7W8SJ8j
# QwO2IDoHHFC46ePg+eoNors0QrC0PWnOgDeMkW6gmLBtq3CEOSDU8iNicwNsNb7A
# Bz0W1E3qlSw7jTmNoGCKCgVkLD2FaMs2qAVVOjuUxvmtWMn1pIFVUvZ1yrPIVbYt
# 1aTld2nrmh544Auh3tgggy/WluoLXlHtAJgvFwrVsKXj8ekFt0TmaPL0lHvQEe5j
# Hbufhc05lvCtdwbfBl/2ARSTuy1s8CgFAgMBAAGjggGOMIIBijAfBgNVHSMEGDAW
# gBRfWO1MMXqiYUKNUoC6s2GXGaIymzAdBgNVHQ4EFgQUaO+kMklptlI4HepDOSz0
# FGqeDIUwDgYDVR0PAQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAww
# CgYIKwYBBQUHAwgwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEF
# BQcCARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEoGA1UdHwRD
# MEEwP6A9oDuGOWh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1Rp
# bWVTdGFtcGluZ0NBUjM2LmNybDB6BggrBgEFBQcBAQRuMGwwRQYIKwYBBQUHMAKG
# OWh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGlu
# Z0NBUjM2LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20w
# DQYJKoZIhvcNAQEMBQADggGBALDcLsn6TzZMii/2yU/V7xhPH58Oxr/+EnrZjpIy
# vYTz2u/zbL+fzB7lbrPml8ERajOVbudan6x08J1RMXD9hByq+yEfpv1G+z2pmnln
# 5XucfA9MfzLMrCArNNMbUjVcRcsAr18eeZeloN5V4jwrovDeLOdZl0tB7fOX5F6N
# 2rmXaNTuJR8yS2F+EWaL5VVg+RH8FelXtRvVDLJZ5uqSNIckdGa/eUFhtDKTTz9L
# tOUh46v2JD5Q3nt8mDhAjTKp2fo/KJ6FLWdKAvApGzjpPwDqFeJKf+kJdoBKd2zQ
# uwzk5Wgph9uA46VYK8p/BTJJahKCuGdyKFIFfEfakC4NXa+vwY4IRp49lzQPLo7W
# ticqMaaqb8hE2QmCFIyLOvWIg4837bd+60FcCGbHwmL/g1ObIf0rRS9ceK4DY9rf
# BnHFH2v1d4hRVvZXyCVlrL7ZQuVzjjkLMK9VJlXTVkHpuC8K5S4HHTv2AJx6mOdk
# MJwS4gLlJ7gXrIVpnxG+aIniGDCCBhQwggP8oAMCAQICEHojrtpTaZYPkcg+XPTH
# 4z8wDQYJKoZIhvcNAQEMBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGlu
# ZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFUxCzAJ
# BgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMTI1Nl
# Y3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0EgUjM2MIIBojANBgkqhkiG9w0B
# AQEFAAOCAY8AMIIBigKCAYEAzZjYQ0GrboIr7PYzfiY05ImM0+8iEoBUPu8mr4wO
# gYPjoiIz5vzf7d5wu8GFK1JWN5hciN9rdqOhbdxLcSVwnOTJmUGfAMQm4eXOls3i
# QwfapEFWuOsYmBKXPNSpwZAFoLGl5y1EaGGc5LByM8wjcbSF52/Z42YaJRsPXY54
# 5E3QAPN2mxDh0OLozhiGgYT1xtjXVfEzYBVmfQaI5QL35cTTAjsJAp85R+KAsOfu
# L9Z7LFnjdcuPkZWjssMETFIueH69rxbFOUD64G+rUo7xFIdRAuDNvWBsv0iGDPGa
# R2nZlY24tz5fISYk1sPY4gir99aXAGnoo0vX3Okew4MsiyBn5ZnUDMKzUcQrpVav
# GacrIkmDYu/bcOUR1mVBIZ0X7P4bKf38JF7Mp7tY3LFF/h7hvBS2tgTYXlD7TnIM
# PrxyXCfB5yQq3FFoXRXM3/DvqQ4shoVWF/mwwz9xoRku05iphp22fTfjKRIVpm4g
# FT24JKspEpM8mFa9eTgKWWCvAgMBAAGjggFcMIIBWDAfBgNVHSMEGDAWgBT2d2rd
# P/0BE/8WoWyCAi/QCj0UJTAdBgNVHQ4EFgQUX1jtTDF6omFCjVKAurNhlxmiMpsw
# DgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMEwGA1UdHwRFMEMwQaA/oD2GO2h0
# dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ1Jv
# b3RSNDYuY3JsMHwGCCsGAQUFBwEBBHAwbjBHBggrBgEFBQcwAoY7aHR0cDovL2Ny
# dC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nUm9vdFI0Ni5w
# N2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3
# DQEBDAUAA4ICAQAS13sgrQ41WAyegR0lWP1MLWd0r8diJiH2VVRpxqFGhnZbaF+I
# Q7JATGceTWOS+kgnMAzGYRzpm8jIcjlSQ8JtcqymKhgx1s6cFZBSfvfeoyigF8iC
# GlH+SVSo3HHr98NepjSFJTU5KSRKK+3nVSWYkSVQgJlgGh3MPcz9IWN4I/n1qfDG
# zqHCPWZ+/Mb5vVyhgaeqxLPbBIqv6cM74Nvyo1xNsllECJJrOvsrJQkajVz4xJwZ
# 8blAdX5umzwFfk7K/0K3fpjgiXpqNOpXaJ+KSRW0HdE0FSDC7+ZKJJSJx78mn+rw
# EyT+A3z7Ss0gT5CpTrcmhUwIw9jbvnYuYRKxFVWjKklW3z83epDVzoWJttxFpujd
# rNmRwh1YZVIB2guAAjEQoF42H0BA7WBCueHVMDyV1e4nM9K4As7PVSNvQ8LI1WRa
# TuGSFUd9y8F8jw22BZC6mJoB40d7SlZIYfaildlgpgbgtu6SDsek2L8qomG57Yp5
# qTqof0DwJ4Q4HsShvRl/59T4IJBovRwmqWafH0cIPEX7cEttS5+tXrgRtMjjTOp6
# A9l0D6xcKZtxnLqiTH9KPCy6xZEi0UDcMTww5Fl4VvoGbMG2oonuX3f1tsoHLaO/
# Fwkj3xVr3lDkmeUqivebQTvGkx5hGuJaSVQ+x60xJ/Y29RBr8Tm9XJ59AjCCBoIw
# ggRqoAMCAQICEDbCsL18Gzrno7PdNsvJdWgwDQYJKoZIhvcNAQEMBQAwgYgxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkg
# Q2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVV
# U0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTIxMDMyMjAw
# MDAwMFoXDTM4MDExODIzNTk1OVowVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1Nl
# Y3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFt
# cGluZyBSb290IFI0NjCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAIid
# 2LlFZ50d3ei5JoGaVFTAfEkFm8xaFQ/ZlBBEtEFAgXcUmanU5HYsyAhTXiDQkiUv
# pVdYqZ1uYoZEMgtHES1l1Cc6HaqZzEbOOp6YiTx63ywTon434aXVydmhx7Dx4IBr
# Aou7hNGsKioIBPy5GMN7KmgYmuu4f92sKKjbxqohUSfjk1mJlAjthgF7Hjx4vvyV
# DQGsd5KarLW5d73E3ThobSkob2SL48LpUR/O627pDchxll+bTSv1gASn/hp6IuHJ
# orEu6EopoB1CNFp/+HpTXeNARXUmdRMKbnXWflq+/g36NJXB35ZvxQw6zid61qmr
# lD/IbKJA6COw/8lFSPQwBP1ityZdwuCysCKZ9ZjczMqbUcLFyq6KdOpuzVDR3ZUw
# xDKL1wCAxgL2Mpz7eZbrb/JWXiOcNzDpQsmwGQ6Stw8tTCqPumhLRPb7YkzM8/6N
# nWH3T9ClmcGSF22LEyJYNWCHrQqYubNeKolzqUbCqhSqmr/UdUeb49zYHr7ALL8b
# AJyPDmubNqMtuaobKASBqP84uhqcRY/pjnYd+V5/dcu9ieERjiRKKsxCG1t6tG9o
# j7liwPddXEcYGOUiWLm742st50jGwTzxbMpepmOP1mLnJskvZaN5e45NuzAHteOR
# lsSuDt5t4BBRCJL+5EZnnw0ezntk9R8QJyAkL6/bAgMBAAGjggEWMIIBEjAfBgNV
# HSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQU9ndq3T/9ARP/
# FqFsggIv0Ao9FCUwDgYDVR0PAQH/BAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wEwYD
# VR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcw
# RaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0
# aWZpY2F0aW9uQXV0aG9yaXR5LmNybDA1BggrBgEFBQcBAQQpMCcwJQYIKwYBBQUH
# MAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIB
# AA6+ZUHtaES45aHF1BGH5Lc7JYzrftrIF5Ht2PFDxKKFOct/awAEWgHQMVHol9ZL
# Syd/pYMbaC0IZ+XBW9xhdkkmUV/KbUOiL7g98M/yzRyqUOZ1/IY7Ay0YbMniIibJ
# rPcgFp73WDnRDKtVutShPSZQZAdtFwXnuiWl8eFARK3PmLqEm9UsVX+55DbVIz33
# Mbhba0HUTEYv3yJ1fwKGxPBsP/MgTECimh7eXomvMm0/GPxX2uhwCcs/YLxDnBdV
# VlxvDjHjO1cuwbOpkiJGHmLXXVNbsdXUC2xBrq9fLrfe8IBsA4hopwsCj8hTuwKX
# JlSTrZcPRVSccP5i9U28gZ7OMzoJGlxZ5384OKm0r568Mo9TYrqzKeKZgFo0fj2/
# 0iHbj55hc20jfxvK3mQi+H7xpbzxZOFGm/yVQkpo+ffv5gdhp+hv1GDsvJOtJinJ
# mgGbBFZIThbqI+MHvAmMmkfb3fTxmSkop2mSJL1Y2x/955S29Gu0gSJIkc3z30vU
# /iXrMpWx2tS7UVfVP+5tKuzGtgkP7d/doqDrLF1u6Ci3TpjAZdeLLlRQZm867eVe
# XED58LXd1Dk6UvaAhvmWYXoiLz4JA5gPBcz7J311uahxCweNxE+xxxR3kT0WKzAS
# o5G/PyDez6NHdIUKBeE3jDPs2ACc6CkJ1Sji4PKWVT0/MYIEkTCCBI0CAQEwaTBV
# MQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# EyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNgIQOlJqLITOVeYd
# ZfzMEtjpiTANBglghkgBZQMEAgIFAKCCAfkwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3
# DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yNDEyMDQxODMwMTBaMD8GCSqGSIb3DQEJ
# BDEyBDDc3x1yJbMJ8RzaBCpHa5vLEvnzbu3XjBqP7Esk927CnmlM78dxrEdJfayo
# MwmQFjYwggF6BgsqhkiG9w0BCRACDDGCAWkwggFlMIIBYTAWBBT4YJgZpvuILPfo
# UpfyoRlSGhZ3XzCBhwQUxq5U5HiG8Xw9VRJIjGnDSnr5wt0wbzBbpFkwVzELMAkG
# A1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2Vj
# dGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBSb290IFI0NgIQeiOu2lNplg+RyD5c
# 9MfjPzCBvAQUhT1jLZOCgmF80JA1xJHeksFC2scwgaMwgY6kgYswgYgxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0
# eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VS
# VHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5AhA2wrC9fBs656Oz3TbL
# yXVoMA0GCSqGSIb3DQEBAQUABIICAEot3XK5Jsq9ZYAWtougpfm2g/s2NWTx04rY
# ZoQZbdDjn3hSU0DvtxtGrtypSBBlU6aSX6xbYrCJTdLs6h4zpsWCZHSHbCT+dtgN
# kBFAF9GcbTG65XajA8Nt5irNdmZwZKMIU0gngYZ5qI5lvhO/niBjULNkFMDl5T1B
# s7qKw8vJrH8BMRS0H9TMtObzxFAfZ2Ue19gaUO5/vUOyIWh4WtsYCAwBXTzoZE+l
# fV6Wi+YLqq8D0i/WHJRyHPDEOLZu1kuKW5QvDcfg9/rO/GTns4jrOSGh9BVMlrrl
# TDw8Ad6EZAVwT20Dl3SHvS2y8cOgczclrO1DGLo5pGnPWLD0Xl92ZBAmbmMYJKVp
# tvZBtV6+KtMFXCcCBLmAC5sgEo9RFGpSaP5+YgTs3Gqstlc6tE1c0d3zChwkmWL5
# cCzzUIuxzOLiJ+qLuLP724ltZpZB8oSg4tLCsKB8s+mwPv61SvFF/gLfrlP0/HGF
# UunuxTxsUpHN/UEil+spqjyaqkbz2rjGC04Rxt3yapfaKq5yrNvkHMiGQofHB1Ut
# YrizX2l8shH7MLkk1b1GUynrjjVm2jZ11JPWLX8bTBt8y70QnxLRe1EgLC8GPQ4a
# iOEJ4/U0YynCoSYLHnmYsTS5UR+gRGIr5GaBBjlFNeU2fYxZ8rmK8vSHnMdknS8B
# zKQzipVr
# SIG # End signature block
