/**
 * C4DLink
 *
 * To generate documentation, install Node.js and 'mr-doc' (https://mr-doc.github.io/).
 *
 * 'npm install -g mr-doc'
 *
 * And run...:
 *
 * 'mr-doc -s res/assets/js/ -o qst_doc/ -n "Quick Start Dialog"'
 */
var cdata = "";

var c4dlink = {
    /**
     * Sets up C4DLink - call this after $(document).ready.
     *
     * @param {Bool} convertLinks - auto-converts links with target '_blank' to open via OS-default browser
     */
    init: function (convertLinks) {
        convertLinks = typeof convertLinks !== 'undefined' ? convertLinks : false;
        c4dlink.addHandler("*[data-c4d_cmd=blank]", c4dlink.openBrowser);
        c4dlink.addHandler("*[data-c4d_cmd=_blank]", c4dlink.openBrowser);
        c4dlink.addHandler("*[data-c4d_cmd=activate]", c4dlink.activate);
        c4dlink.addHandler("*[data-c4d_cmd=callcommand]", c4dlink.callCommand);
        c4dlink.addHandler("*[data-c4d_cmd=open]", c4dlink.openFile);
        if (convertLinks) {
            c4dlink.convertLinks();
        }
    },
    /**
     * Auto-converts link with target '_blank'
     */
    convertLinks: function () {
        c4dlink.addHandler("a[target=_blank]", c4dlink.openBrowser);
    },
    /**
     * Switches to item width id.
     *
     * @param {String} selector - element selector
     * @param {String} cmd - command
     * @param {Object} cb - callback
     */
    addHandler: function (selector, cb) {
        $(selector).click(function (e) {
            e.preventDefault();
            var value = "false";
            var hrefData = $(this).attr("href");
            if (typeof hrefData !== "undefined") {
                value = hrefData;
            }
            var dataValue = $(this).data("c4d_value");
            if (typeof dataValue !== "undefined") {
                value = dataValue;
            }
            var closeDlg = false;
            var dataClose = $(this).data("c4d_close");
            if (typeof dataClose === "boolean") {
                closeDlg = dataClose;
            }
            cb(value, closeDlg);
        });
    },
    /**
     * Opens the OS-default browser.
     *
     * USAGE:
     * -----------------------------------------------------------------------------------------------------------------------
     * Auto-convert existing links with 'target=_blank' using c4dlink.init(true) or c4dlink.convertLinks();
     * Alternatively you can decorate the anchor tag with the 'data-c4d_cmd="_blank"' attribute.
     * Call directly using c4dlink.openBrowser("google.de");
     *
     * Hint: Decorate anchor with 'data-c4d_close' attribute to close dialog after execution.
     *
     * -----------------------------------------------------------------------------------------------------------------------
     *
     * @param {String} value - url
     */
    openBrowser: function (value, closeDlg) {
        closeDlg = typeof closeDlg !== 'undefined' ? closeDlg : false;
        window.location.href = c4dlink.getURL('blank', value, closeDlg);
    },
    /**
     * Opens a *.C4D file.
     *
     * USAGE:
     * -----------------------------------------------------------------------------------------------------------------------
     * Decorate an anchor with the 'data-c4d_cmd="open"' attribute and either provide the file-link via 'data-c4d_value' or just use the href attribute.
     * Call directly using c4dlink.openFile("file.c4d");
     *
     * Hint: Decorate anchor with 'data-c4d_close' attribite to close dialog after execution.
     *
     * -----------------------------------------------------------------------------------------------------------------------
     *
     * @param {String} value - url
     */
    openFile: function (value, closeDlg) {
        closeDlg = typeof closeDlg !== 'undefined' ? closeDlg : false;
        window.location.href = c4dlink.getURL('open', value, closeDlg);
    },
    /**
     * Enables / disables the dialog.
     *
     * USAGE:
     * -----------------------------------------------------------------------------------------------------------------------
     * Decorate an anchor with the 'data-c4d_cmd="enable"' attribute.
     * Call directly using c4dlink.enable();
     *
     * Hint: Decorate anchor with 'data-c4d_close' attribite to close dialog after execution.
     *
     * -----------------------------------------------------------------------------------------------------------------------
     *
     * @param {Boolean} enabled - true if enabled
     */
    enable: function (enabled, closeDlg) {
        closeDlg = typeof closeDlg !== 'undefined' ? closeDlg : false;
        var url = c4dlink.getURL('enable', String(enabled), closeDlg);
        var state = url.substr(url.indexOf("&enabled=") + 9, url.length);
        state = state.substr(0, state.indexOf("&"));
        url = url.replace("enabled=" + state, "enabled=" + String(enabled));
        window.location.href = url;
    },
    /**
     * Enables / disables new feature highlighting.
     *
     * USAGE:
     * -----------------------------------------------------------------------------------------------------------------------
     * Decorate an anchor with the 'data-c4d_cmd="highlight"' attribute.
     * Call directly using c4dlink.highlight();
     *
     * Hint: Decorate anchor with 'data-c4d_close' attribite to close dialog after execution.
     *
     * -----------------------------------------------------------------------------------------------------------------------
     *
     * @param {Boolean} enabled - true if enabled
     */
    highlight: function (enabled, closeDlg) {
        closeDlg = typeof closeDlg !== 'undefined' ? closeDlg : false;
        var url = c4dlink.getURL('highlight', String(enabled), closeDlg);
        var state = url.substr(url.indexOf("&highlight=") + 11, url.length);
        if (state.indexOf("&") !== -1) {
            state = state.substr(0, state.indexOf("&"));
        }
        url = url.replace("highlight=" + state, "highlight=" + String(enabled));
        window.location.href = url;
    },
    /**
     * Executes remote CallCommand().
     *
     * USAGE:
     * -----------------------------------------------------------------------------------------------------------------------
     * Decorate an anchor with the 'data-c4d_cmd="callcommand"' attribute and either provide the ID via 'data-c4d_value' or just use the href attribute.
     * Call directly using c4dlink.callCommand(ID);
     *
     * Hint: Decorate anchor with 'data-c4d_close' attribite to close dialog after execution.
     *
     * -----------------------------------------------------------------------------------------------------------------------
     *
     * @param {String} value - url
     */
    callCommand: function (value, closeDlg) {
        closeDlg = typeof closeDlg !== 'undefined' ? closeDlg : false;
        window.location.href = c4dlink.getURL('callcommand', value, closeDlg);
    },
    /**
     * Adds return parameters to URL.
     *
     * @param {String} action - action to execute
     * @param {String} value - value to encode
     * @param {Bool} closeDlg - closes dialog after execution
     */
    getURL: function (action, value, closeDlg) {
        closeDlg = typeof closeDlg !== 'undefined' ? closeDlg : false;
        var recentList = c4dlink.getRecentlList();
        var recent = "";
        for (var i = 0; i < recentList.length; i++) {
            recent += "&" + i + "=" + c4dlink.b64EncodeUnicode(recentList[i]);
        }
        return window.location.href.match(/^[^?&]+/)[0] + "?version=" + c4dlink.getVersion() + "&perpetual=" + String(c4dlink.getPerpetual()) + "&action=" + action + "&value=" + c4dlink.b64EncodeUnicode(value) + "&close=" + String(closeDlg) + "&enabled=" + String(c4dlink.getEnabled()) + "&highlight=" + String(c4dlink.getHighlight()) + "&with_redshift=" + String(c4dlink.getRedshift()) + "&with_python=" + String(c4dlink.getPython()) + recent + cdata;
    },
    /**
     * Returns the list of recently opened files.
     *
     * @returns {Array} The list of recently opened files.
     */
    getRecentlList: function () {
        var recent = [];
        var url = window.location.href;
        var tmp = "";
        for (var i = 0; i < 10; i++) {
            if (url.indexOf(i + "=") !== -1) {
                url = url.substr(url.indexOf(i + "=") + 2, url.length);
                tmp = url;
                if (tmp.indexOf("&") !== -1) {
                    tmp = tmp.substr(0, tmp.indexOf("&"));
                }
                try {
                    tmp = c4dlink.b64DecodeUnicode(tmp);
                }catch (e){
                    
                }
                recent.push(tmp);
            } else {
                break;
            }
        }
        return recent;
    },
    /**
     * Returns the client's dialog enabled state.
     *
     * @returns {bool} The client's dialog enabled state.
     */
    getEnabled: function () {
        var enabled = false;
        var url = window.location.href;
        if (url.indexOf("&enabled=") !== -1) {
            url = url.substr(url.indexOf("&enabled=") + 9, url.length);
            if (url.indexOf("&") !== -1) {
                url = url.substr(0, url.indexOf("&"));
            }
            if (url === "true") {
                enabled = true;
            }
        }
        return enabled;
    },
    /**
     * Returns the client's new feature highlighting state.
     *
     * @returns {bool} The client's new feature highlighting state.
     */
    getHighlight: function () {
        var highlight = false;
        var url = window.location.href;
        if (url.indexOf("&highlight=") !== -1) {
            url = url.substr(url.indexOf("&highlight=") + 11, url.length);
            if (url.indexOf("&") !== -1) {
                url = url.substr(0, url.indexOf("&"));
            }
            if (url === "true") {
                highlight = true;
            }
        }
        return highlight;
    },
    /**
     * Returns the client's Redshift state.
     *
     * @returns {bool} The client's Redshift state.
     */
    getRedshift: function () {
        var redshift = false;
        var url = window.location.href;
        if (url.indexOf("&with_redshift=") !== -1) {
            url = url.substr(url.indexOf("&with_redshift=") + 15, url.length);
            if (url.indexOf("&") !== -1) {
                url = url.substr(0, url.indexOf("&"));
            }
            if (url === "true") {
                redshift = true;
            }
        }
        return redshift;
    },
    /**
     * Returns the client's Python state.
     *
     * @returns {bool} The client's Python state.
     */
    getPython: function () {
        var python = false;
        var url = window.location.href;
        if (url.indexOf("&with_python=") !== -1) {
            url = url.substr(url.indexOf("&with_python=") + 13, url.length);
            if (url.indexOf("&") !== -1) {
                url = url.substr(0, url.indexOf("&"));
            }
            if (url === "true") {
                python = true;
            }
        }
        return python;
    },
    /**
     * Gets the original type value.
     *
     * @returns {bool} The client's type.
     */
    getType: function () {
        var type = "full";
        var url = window.location.href;
        if (url.indexOf("&type=") !== -1) {
            type = url.substr(url.indexOf("&type=") + 6, url.length);
            if (type.indexOf("&") !== -1) {
                type = type.substr(0, type.indexOf("&"));
            }
        }
        return type;
    },
    /**
     * Gets the version string.
     *
     * @returns {String} The client's version.
     */
    getVersion: function () {
        var version = "";
        var url = window.location.href;
        if (url.indexOf("?version=") !== -1) {
            url = url.substr(url.indexOf("?version=") + 9, url.length);
            if (url.indexOf("&") !== -1) {
                version = url.substr(0, url.indexOf("&"));
            }
        }
        return version;
    },
    /**
     * Checks for perpetual.
     *
     * @returns {Boolean} true if perpetual.
     */
    getPerpetual: function () {
        var perpetual = false;
        var url = window.location.href;
        if (url.indexOf("&perpetual=") !== -1) {
            url = url.substr(url.indexOf("&perpetual=") + 11, url.length);
            if (url.indexOf("&") !== -1) {
                perpetual = url.substr(0, url.indexOf("&")) == "true" ? true : false;
            }
        }
        return perpetual;
    }, 
    /**
     * Queries for an available update.
     *
     * @param {Function} checkUpdateCB - the update callback (needs to take <data> obj)
     */
    checkUpdate: function (checkUpdateCB) {
        if(c4dlink.getVersion().length == 0) return;
        
        $.ajax({ 
            type: 'GET', 
            url: "https://www.cineversity.com/quickstart/" + c4dlink.getVersion().substr(0, 2) + "_update.json", 
            dataType: 'json',
            success: checkUpdateCB
        });
    },
    /**
     * Queries for an available languages.
     *
     * @param {Function} checkLanguagesCB - the callback (needs to take <data> obj)
     */
    checkLanguages: function (checkLanguagesCB) {
        if(c4dlink.getVersion().length == 0) return;
        
        $.ajax({ 
            type: 'GET', 
            url: "https://www.cineversity.com/quickstart/" + c4dlink.getVersion().substr(0, 2) + "_languages.json", 
            dataType: 'json',
            success: checkLanguagesCB
        });
    },
    /**
     * Encodes a base64 encoded unicode string.
     *
     * @param {String} str - string to encode
     * @returns {String} The encoded string.
     */
    b64EncodeUnicode: function (str) {
        return btoa(encodeURIComponent(str).replace(/%([0-9A-F]{2})/g, function (match, p1) {
            return String.fromCharCode('0x' + p1);
        }));
    },
    /**
     * Decodes a base64 encoded unicode string.
     *
     * @param {String} str - string to decode
     * @returns {String} The decoded string.
     */
    b64DecodeUnicode: function (str) {
        return decodeURIComponent(Array.prototype.map.call(atob(str), function (c) {
            return '%' + ('00' + c.charCodeAt(0).toString(16)).slice(-2)
        }).join(''));
    }
};