#version 150
out vec4 FragColor;
in vec4 v_texcoord;

uniform vec3      iResolution;     // viewport resolution (in pixels)
uniform float     iTime;     // shader playback time (in seconds)
uniform vec4      iMouse;          // mouse pixel coords. xy: current (if MLB down), zw: click
uniform int      iFrame;
uniform sampler2D iChannel0;


//
// Source: https://www.shadertoy.com/view/WlSXRW
//

// The MIT License
// Copyright © 2019 Inigo Quilez
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following conditions: The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

// Intersection of a ray with a rounded cube, testing a single
// corner (sphere) instead of 8, and only 3 edges (cylinders)
// instead of 12. There might be a better (more compact and
// efficient) way to do it, but this is where I landed. However
// the code to compute the surface normal is particularly
// elegant.

// Other intersectors: http://iquilezles.org/www/articles/intersectors/intersectors.htm
//
// Box:             https://www.shadertoy.com/view/ld23DV
// Triangle:        https://www.shadertoy.com/view/MlGcDz
// Capsule:         https://www.shadertoy.com/view/Xt3SzX
// Ellipsoid:       https://www.shadertoy.com/view/MlsSzn
// Sphere:          https://www.shadertoy.com/view/4d2XWV
// Capped Cylinder: https://www.shadertoy.com/view/4lcSRn
// Disk:            https://www.shadertoy.com/view/lsfGDB
// Capped Cone:     https://www.shadertoy.com/view/llcfRf
// Rounded Box:     https://www.shadertoy.com/view/WlSXRW
// Rounded Cone:    https://www.shadertoy.com/view/MlKfzm
// Torus:           https://www.shadertoy.com/view/4sBGDy
// Sphere4:         https://www.shadertoy.com/view/3tj3DW
// Goursat:         https://www.shadertoy.com/view/3lj3DW
    
#define AA 2  // reduce this to 1 if you have a slow machine


float roundedboxIntersect( in vec3 ro, in vec3 rd, in vec3 size, in float rad )
{
	// bounding box
    vec3 m = 1.0/rd;
    vec3 n = m*ro;
    vec3 k = abs(m)*(size+rad);
    vec3 t1 = -n - k;
    vec3 t2 = -n + k;
	float tN = max( max( t1.x, t1.y ), t1.z );
	float tF = min( min( t2.x, t2.y ), t2.z );
	if( tN > tF || tF < 0.0) return -1.0;
    float t = tN;

    // convert to first octant
    vec3 pos = ro+t*rd;
    vec3 s = sign(pos);
    ro  *= s;
    rd  *= s;
    pos *= s;
        
    // faces
    bvec3 cen = bvec3( pos.x<size.x && pos.y<size.y,
                       pos.y<size.y && pos.z<size.z,
                       pos.z<size.z && pos.x<size.x );
    if( any(cen) )
    {
        return t;
    }

    // some precomputation
    vec3 oc = ro - size;
    vec3 dd = rd*rd;
	vec3 oo = oc*oc;
    vec3 od = oc*rd;
    float ra2 = rad*rad;

    t = 1e20;        

    // corner
    {
    float b = od.x + od.y + od.z;
	float c = oo.x + oo.y + oo.z - ra2;
	float h = b*b - c;
	if( h>0.0 ) t = -b-sqrt(h);
    }
    // edge X
    {
	float a = dd.y + dd.z;
	float b = od.y + od.z;
	float c = oo.y + oo.z - ra2;
	float h = b*b - a*c;
	if( h>0.0 )
    {
	  h = (-b-sqrt(h))/a;
      if( h>0.0 && h<t && abs(ro.x+rd.x*h)<size.x ) t = h;
    }
	}
    // edge Y
    {
	float a = dd.z + dd.x;
	float b = od.z + od.x;
	float c = oo.z + oo.x - ra2;
	float h = b*b - a*c;
	if( h>0.0 )
    {
	  h = (-b-sqrt(h))/a;
      if( h>0.0 && h<t && abs(ro.y+rd.y*h)<size.y ) t = h;
    }
	}
    // edge Z
    {
	float a = dd.x + dd.y;
	float b = od.x + od.y;
	float c = oo.x + oo.y - ra2;
	float h = b*b - a*c;
	if( h>0.0 )
    {
	  h = (-b-sqrt(h))/a;
      if( h>0.0 && h<t && abs(ro.z+rd.z*h)<size.z ) t = h;
    }
	}

    if( t>1e19 ) t=-1.0;
    
	return t;
}

// normal of a rounded box
vec3 roundedboxNormal( in vec3 pos, in vec3 siz, in float rad )
{
    return sign(pos)*normalize(max(abs(pos)-siz,0.0));
}


//======================================================

mat4 rotate( vec3 v, float angle )
{
    float s = sin( angle );
    float c = cos( angle );
    float ic = 1.0 - c;

    return mat4( v.x*v.x*ic + c,     v.y*v.x*ic - s*v.z, v.z*v.x*ic + s*v.y, 0.0,
                 v.x*v.y*ic + s*v.z, v.y*v.y*ic + c,     v.z*v.y*ic - s*v.x, 0.0,
                 v.x*v.z*ic - s*v.y, v.y*v.z*ic + s*v.x, v.z*v.z*ic + c,     0.0,
			     0.0,                0.0,                0.0,                1.0 );
}

mat4 translate( float x, float y, float z )
{
    return mat4( 1.0, 0.0, 0.0, 0.0,
				 0.0, 1.0, 0.0, 0.0,
				 0.0, 0.0, 1.0, 0.0,
				 x,   y,   z,   1.0 );
}

vec3 pconvert( in mat4 mat, in vec3 v ) { return (mat*vec4(v,1.0)).xyz; }
vec3 nconvert( in mat4 mat, in vec3 v ) { return (mat*vec4(v,0.0)).xyz; }

//======================================================


struct
{
   mat4  world_to_obj;
   mat4  obj_to_world;
   vec3  size;
   float radius;
}box;

vec2 intersect( in vec3 ro, in vec3 rd )
{
    vec2 res = vec2(1e20,-1.0);
    
    // plane
    {
    float t = (0.0-ro.y)/rd.y;
    if( t>0.0 ) res = vec2(t,1.0);
    }

    // rounded box
    {
    
    // convert from ray to box space
    vec3 rdd = nconvert(box.world_to_obj, rd );
    vec3 roo = pconvert(box.world_to_obj, ro );
    // intersect in box space
    float t = roundedboxIntersect(roo,rdd,box.size,box.radius);
    if( t>0.0 && t<res.x ) res = vec2(t,2.0);
    }
    
    return res;
}

// texture sampling
vec4 boxmap( sampler2D sam, in vec3 p, in vec3 n, in float k )
{
    vec3 m = pow( abs(n), vec3(k) );
	vec4 x = texture( sam, p.yz );
	vec4 y = texture( sam, p.zx );
	vec4 z = texture( sam, p.xy );
	return (x*m.x + y*m.y + z*m.z)/(m.x+m.y+m.z);
}

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    // camera movement	
    float an = 0.15*(iTime-10.0);
    vec3 ta = vec3( 0.0, 1.0, 0.0 );
    vec3 ro = ta + vec3( 2.0*cos(an), 0.2, 2.0*sin(an) );
    // camera matrix
    vec3 ww = normalize( ta - ro );
    vec3 uu = normalize( cross(ww,vec3(0.6,8.0,0.0) ) );
    vec3 vv =          ( cross(uu,ww));

    // rounded box animation
    box.obj_to_world = translate( 0.0, 1.0, 0.0 ) * 
                       rotate( normalize(vec3(1.0,1.0,0.1)), iTime ); 
    box.world_to_obj = inverse( box.obj_to_world );
    box.size = vec3(0.5,0.4,0.3);
    box.radius = 0.2;

    // render
    vec3 tot = vec3(0.0);
    
    #if AA>1
    for( int m=0; m<AA; m++ )
    for( int n=0; n<AA; n++ )
    {
        // pixel coordinates
        vec2 o = vec2(float(m),float(n)) / float(AA) - 0.5;
        vec2 p = (2.0*(fragCoord+o)-iResolution.xy)/iResolution.y;
        #else    
        vec2 p = (2.0*fragCoord-iResolution.xy)/iResolution.y;
        #endif
    
	    // create view ray
        vec3 rd = normalize( p.x*uu + p.y*vv + 1.6*ww );

        // raymarch
        vec2 tm = intersect( ro, rd );
        vec3 col = vec3(0.6,0.75,0.85) - 0.97*rd.y;
        if( tm.y>0.0 )
        {
            // shading
            vec3 pos = ro + tm.x*rd;
            vec3 nor;
            vec3 mate;
            
            if( tm.y<1.5 ) // floor
            {
                nor = vec3(0.0,1.0,0.0);
                mate = 0.35*texture( iChannel0, 0.2*pos.xz ).xyz;
            }
            else // rounded cube
            {
                // convert position from world to box space
                vec3 bpos = pconvert(box.world_to_obj,pos);
                // compute normal in box space
                vec3 bnor = roundedboxNormal(bpos,box.size,box.radius);
                // convert normal from box to world space
                nor = nconvert(box.obj_to_world,bnor);
                
                mate = 0.35*boxmap( iChannel0, bpos, bnor, 8.0 ).xyz;
            }
            
            // lighting
            vec3 lig = normalize(vec3(0.8,0.4,-0.6));
            float dif = clamp( dot(nor,lig), 0.0, 1.0 );
            vec3  hal = normalize(lig-rd);
            float sha = step( intersect( pos+0.01*nor, lig ).y, 0.0 );
            float amb = 0.6 + 0.4*nor.y;
            float bou = clamp(0.3-0.7*nor.y,0.0,1.0);
            float spe = clamp(dot(nor,hal),0.0,1.0);
            col  = 4.0*vec3(1.00,0.80,0.60)*dif*sha;
            col += 2.0*vec3(0.20,0.30,0.40)*amb;
            col += 2.0*vec3(0.30,0.20,0.10)*bou;
			col *= mate;            
            col += 0.3*pow(spe,8.0)*dif*sha*(0.04+0.96*pow(clamp(dot(lig,hal),0.0,1.0),5.0));
            col = mix( col, vec3(0.6,0.7,0.8), 1.0-exp(-0.001*tm.x*tm.x) );           
        }
        
        // vignetting        
        col *= 1.0 - 0.1*dot(p,p);

	    tot += col;
    #if AA>1
    }
    tot /= float(AA*AA);
    #endif

    // gamma
    tot = pow( tot, vec3(0.45) );
    
    tot = clamp(tot,0.0,1.0);
        
	// grading
    tot = tot*tot*(3.0-2.0*tot);

    fragColor = vec4( tot, 1.0 );
}

void main( void ){vec4 color = vec4(0.0,0.0,0.0,1.0); mainImage(color, gl_FragCoord.xy);color.w = 1.0;FragColor = color;}  
