// StereoSample.c
// NVIDIA Corporation
// 
// 2002-02-05 Detlef Roettger: Based on my sample done for ELSA
//            (which can also be found in the StereoGraphics developer relations download)

/* ##### INCLUDES 	*/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <gl\gl.h>
#include <gl\glu.h>

#include "resource.h"


/* ##### DEFINES 	*/

#define QUADBUFFERED_STEREO 1
// Uncomment the following line to get a VerticalInterleave stereo sample which doesn't need a PFD_STEREO format.
// #undef  QUADBUFFERED_STEREO 

#define X 0.525731112119133606f
#define Z 0.850650808352039932f

#define MORPH_MIN 0.25f
#define MORPH_MAX 1.5f

// some very good looking defaults for this demo

#define EYE_OFFSET  0.0358333f  // default viewpoint separation
#define EYE_ADJUST -0.0123611f  // default horizontal image shift adjustment

#define PULL_BACK 2.2f          // z distance from object center

/* ##### TYPEDEFS	*/

/* ##### VARIABLES */

char szAppName[]  = "OpenGL_Stereo_Sample";
char szAppTitle[] = "OpenGL Stereo Sample";

HINSTANCE hInstMain;
HACCEL    hAccel;

static HDC   hdc   = NULL;
static HGLRC hglrc = NULL;

char szMessage[2048];
char szInfo[256];

GLboolean bStereo = GL_FALSE;
GLboolean bFullscreen = GL_FALSE;

BOOL bUpdateStencil = TRUE; // update at least once
int nScreenOffset = 0;

BOOL bLButtonDown = FALSE;
BOOL bRButtonDown = FALSE;

BOOL bPseudoMonoscopic = FALSE;

int nWidth;
int nHeight;
float aspectViewport = 1.0f;

GLfloat eyeOffset = EYE_OFFSET; // init stereo separation with default settings
GLfloat eyeAdjust = EYE_ADJUST;

POINT ptLast;
POINT ptCurrent;

UINT      uiTimer = 0;
GLfloat   angleX = 0.0f;
GLfloat   angleY = 0.0f;
GLfloat   angleZ = 0.0f;
GLfloat   translateZ = 0.0f;
GLfloat   incZ = 0.005f;

// Object data

GLfloat   factorMorph = 1.0;
GLfloat   incMorph    = 0.015f;

static GLfloat vIco[12][3] = 
{
  {  -X, 0.0f,    Z},
  {   X, 0.0f,    Z},
  {  -X, 0.0f,   -Z},
  {   X, 0.0f,   -Z},
  {0.0f,    Z,    X},
  {0.0f,    Z,   -X},
  {0.0f,   -Z,    X},
  {0.0f,   -Z,   -X},
  {   Z,    X, 0.0f},
  {  -Z,    X, 0.0f},
  {   Z,   -X, 0.0f},
  {  -Z,   -X, 0.0f}
};

static int idxIco[20][3] = 
{
  { 0,  1,  4}, //  0
  { 0,  4,  9}, //  1
  { 0,  9, 11}, //  2
  { 0,  6,  1}, //  3
  { 0, 11,  6}, //  4
  { 1,  6, 10}, //  5
  { 1, 10,  8}, //  6
  { 1,  8,  4}, //  7
  { 2,  3,  7}, //  8
  { 2,  5,  3}, //  9
  { 2,  9,  5}, // 10
  { 2, 11,  9}, // 11
  { 2,  7, 11}, // 12
  { 3,  5,  8}, // 13
  { 3,  8, 10}, // 14
  { 3, 10,  7}, // 15
  { 4,  5,  9}, // 16
  { 4,  8,  5}, // 17
  { 6,  7, 10}, // 18
  { 6, 11,  7}  // 19
};

static GLfloat vDode[20][3];

static int idxDode[12][5] = 
{
  { 0,  1,  2,  4,  3}, //  0
  { 0,  3,  5,  6,  7}, //  1
  { 9,  8, 12, 11, 10}, //  2
  { 9, 13, 14, 15,  8}, //  3
  { 0,  7, 17, 16,  1}, //  4
  { 9, 10, 16, 17, 13}, //  5
  {18,  5,  3,  4, 19}, //  6
  {15, 18, 19, 12,  8}, //  7
  { 6, 14, 13, 17,  7}, //  8
  { 1, 16, 10, 11,  2}, //  9
  { 5, 18, 15, 14,  6}, // 10
  { 2, 11, 12, 19,  4}  // 11
};


#ifndef QUADBUFFERED_STEREO

GLubyte patternEven[128] =
{
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 
  0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55
};

GLubyte patternOdd[128] =
{
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 
  0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA
};

#endif // QUADBUFFERED_STEREO

/* ##### PROTOTYPES */
void ParseCommandLine(LPSTR lpCmdLine);

int CALLBACK WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow);

LRESULT CALLBACK WndProc(HWND   hwnd,
                         UINT   message,
                         WPARAM wParam,
                         LPARAM lParam);

BOOL InitApplication(HINSTANCE hInstance);

BOOL InitInstance(HINSTANCE hInstance,
                  int       nCmdShow);

void Display_QuadbufferedStereo(void);
void Display_QuadbufferedMono(void);
void Display_VerticalInterleave(void);

void CommandHandler(HWND hwnd, int command);

void MaterialCreate(void);
void LightCreate(void);
void Icosahedron(void);

void InitDodecahedron(void);
void Dodecahedron(void);
void DodecahedronMorph(void);


/* ##### FUNCTIONS */

void ParseCommandLine(LPSTR lpCmdLine)
{
    char* token;

    token = strtok(lpCmdLine, " ");
    while (token != NULL)
    {
        if (!strcmp(token, "-fullscreen"))
        {
            bFullscreen = GL_TRUE;
        }
        token = strtok(NULL, " ");
    }
}



int CALLBACK WinMain( HINSTANCE hInstance,
                      HINSTANCE hPrevInstance,
                      LPSTR     lpCmdLine,
                      int       nCmdShow)
{
  MSG msg;

  // Override default parameteres with values from the commandline.
  ParseCommandLine(lpCmdLine);

  if (!hPrevInstance)
  {
    /* Other instances of app running? */
    if (!InitApplication(hInstance))
    {
      /* Initialize shared things */
      return (FALSE);     /* Exits if unable to initialize */
    }
  }

  /* Perform initializations that apply to a specific instance */
  if (!InitInstance(hInstance, nCmdShow))
  {
    return (FALSE);
  }

  hAccel = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDR_ACCELERATOR1));

  /* Acquire and dispatch messages until a WM_QUIT message is received. */
  while (GetMessage(&msg,     /* message structure */
                    NULL,     /* handle of window receiving the message */
                    0,        /* lowest message to examine */
                    0))       /* highest message to examine */
  {
    if (!TranslateAccelerator(msg.hwnd, hAccel, &msg))
    {
      TranslateMessage(&msg);   /* Translates virtual key codes */
      DispatchMessage(&msg);    /* Dispatches message to window */
    }
  }

  return (msg.wParam);    /* Returns the value from PostQuitMessage */
}


BOOL InitApplication(HINSTANCE hInstance)
{
  WNDCLASS  wc;

  /* 
    Fill in window class structure with parameters that describe the
    main window.
  */
  wc.style         = CS_OWNDC | CS_HREDRAW | CS_VREDRAW; /* Class style(s). */
  wc.lpfnWndProc   = (WNDPROC) WndProc;             /* Window Procedure */
  wc.cbClsExtra    = 0;                             /* No per-class extra data. */
  wc.cbWndExtra    = 0;                             /* No per-window extra data. */
  wc.hInstance     = hInstance;                     /* Owner of this class */
  wc.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON2)); /* Icon */
  wc.hCursor       = LoadCursor(NULL, IDC_ARROW);   /* Cursor */
  wc.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);   /* Default color */
  wc.lpszMenuName  = NULL; // MAKEINTRESOURCE(IDR_MENU1);    /* No Menu */
  wc.lpszClassName = szAppName;                     /* Name to register as */

  /* Register the window class and return success/failure code. */
  return (RegisterClass(&wc));
}


BOOL InitInstance(HINSTANCE hInstance,
                  int       nCmdShow)
{
  HWND  hwnd;   /* Main window handle. */

  /* Save the instance handle in static variable, which will be used in */
  /* many subsequence calls from this application to Windows. */

  hInstMain = hInstance; /* Store instance handle in our global variable */

  /* Create a main window for this application instance. */
  if (bFullscreen)
  {
      hwnd = CreateWindow(szAppName,            /* See RegisterClass() call.   */
                          szAppTitle,           /* Text for window title bar.  */
                          WS_POPUP | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,  /* Window style.     */
                          0,
                          0,
                          GetSystemMetrics(SM_CXSCREEN),
                          GetSystemMetrics(SM_CYSCREEN),
                          NULL,                 /* Overlapped windows have no parent.     */
                          NULL,                 /* Use the window class menu.             */
                          hInstance,            /* This instance owns this window.        */
                          NULL);                /* We don't use any data in our WM_CREATE */
  }
  else
  {
      hwnd = CreateWindow(szAppName,            /* See RegisterClass() call.   */
                          szAppTitle,           /* Text for window title bar.  */
                          WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,  /* Window style.     */
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          NULL,                 /* Overlapped windows have no parent.     */
                          NULL,                 /* Use the window class menu.             */
                          hInstance,            /* This instance owns this window.        */
                          NULL);                /* We don't use any data in our WM_CREATE */
  }

  /* If window could not be created, return "failure" */
  if (!hwnd)
  {
    return (FALSE);
  }

  /* Make the window visible; update its client area; and return "success" */
  ShowWindow(hwnd, nCmdShow); /* Show the window */
  UpdateWindow(hwnd);         /* Sends WM_PAINT message */

  return (TRUE);
}


LRESULT CALLBACK WndProc(HWND   hwnd,
                         UINT   uMessage,
                         WPARAM wParam,
                         LPARAM lParam)
{
  PIXELFORMATDESCRIPTOR pfd;
  int                   iStereoPixelFormats = 0;
  int                   iPixelFormat;
  int                   offset;
  POINT                 point;

  switch (uMessage)
	{
    case WM_CREATE:
      hdc = GetDC(hwnd);

      // Enumerate all pixelformats and count stereo pfds
      iPixelFormat = DescribePixelFormat(hdc, 1, sizeof(PIXELFORMATDESCRIPTOR), &pfd);
      while (iPixelFormat)
      {
        DescribePixelFormat(hdc, iPixelFormat, sizeof(PIXELFORMATDESCRIPTOR), &pfd);
        if (pfd.dwFlags & PFD_STEREO)
        {
          // With additional comparison rules we could break here and
          // use this pixelformat rather than calling ChoosePixelFormat
          // later and let it decide what we use.
          iStereoPixelFormats++;
        }
        iPixelFormat--;
      }
      
      ZeroMemory(&pfd, sizeof(PIXELFORMATDESCRIPTOR));      
      pfd.nSize           = sizeof(PIXELFORMATDESCRIPTOR); 
      pfd.nVersion        = 1; 
      pfd.dwFlags         = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
#ifdef QUADBUFFERED_STEREO
      pfd.dwFlags         |= PFD_STEREO; // Display_QuadbufferedStereo() needs PFD_STEREO enabled.
#endif // QUADBUFFERED_STEREO
      pfd.iPixelType      = PFD_TYPE_RGBA; 
      pfd.cColorBits      = 24; 
      pfd.cRedBits        = 0; 
      pfd.cRedShift       = 0; 
      pfd.cGreenBits      = 0; 
      pfd.cGreenShift     = 0; 
      pfd.cBlueBits       = 0; 
      pfd.cBlueShift      = 0; 
      pfd.cAlphaBits      = 0;
      pfd.cAlphaShift     = 0; 
      pfd.cAccumBits      = 0; 
      pfd.cAccumRedBits   = 0; 
      pfd.cAccumGreenBits = 0; 
      pfd.cAccumBlueBits  = 0; 
      pfd.cAccumAlphaBits = 0; 
      pfd.cDepthBits      = 24; 
#ifdef QUADBUFFERED_STEREO
      pfd.cStencilBits    = 0;
#else // QUADBUFFERED_STEREO
      pfd.cStencilBits    = 8;  // At least one bit needed for the VerticalInterleave method.
#endif // QUADBUFFERED_STEREO
      pfd.cAuxBuffers     = 0; 
      pfd.iLayerType      = PFD_MAIN_PLANE;
      pfd.bReserved       = 0; 
      pfd.dwLayerMask     = 0;
      pfd.dwVisibleMask   = 0; 
      pfd.dwDamageMask    = 0;

      iPixelFormat = ChoosePixelFormat(hdc, &pfd);
      if (iPixelFormat != 0)
      {
        // Get the actual settings of the pixelformat.
        // This is used in SetPixelFormat and for further references.
        DescribePixelFormat(hdc, iPixelFormat, sizeof(PIXELFORMATDESCRIPTOR), &pfd);

        if (SetPixelFormat(hdc, iPixelFormat, &pfd))
        {

          hglrc = wglCreateContext(hdc);
          if (hglrc != NULL)
          {
            if (wglMakeCurrent(hdc, hglrc))
            {
              /* ======================================================================== */
              // OPENGL DRIVER QUERY START

              wsprintf(szMessage,
                       "Current OpenGL Driver:\n\n%s\n%s\n%s\n\n%d stereo pixelformats exported\n",
                       glGetString(GL_VENDOR),
                       glGetString(GL_RENDERER),
                       glGetString(GL_VERSION),
                       iStereoPixelFormats);

#ifdef QUADBUFFERED_STEREO
              if (0 == (pfd.dwFlags & PFD_STEREO))
              {
                strcat(szMessage, "\nNote:\tApplication isn't running\n\twith a stereo pixelformat!\n");
              }

              glGetBooleanv(GL_STEREO, &bStereo); // query stereo support!
              if (!bStereo)
              {
                strcat(szMessage, "\nGL_STEREO == GL_FALSE\n");
              }
              else
              {
                strcat(szMessage, "\nGL_STEREO == GL_TRUE\n");
              }
#endif // QUADBUFFERED_STEREO
              MessageBox(GetFocus(), szMessage, szAppTitle, MB_SYSTEMMODAL | MB_OK | MB_ICONINFORMATION);

              // OPENGL DRIVER QUERY END
              /* ======================================================================== */

              /* ======================================================================== */
              // OPENGL INITIALIZATION START

              glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

              LightCreate();
              Icosahedron();
              MaterialCreate();

              glEnable(GL_DEPTH_TEST);
              glDepthFunc(GL_LESS);

              // glFrontFace(GL_CCW);  // default
              // glCullFace(GL_BACK);  // default
              glEnable(GL_CULL_FACE);
              glShadeModel(GL_FLAT);

              InitDodecahedron();

              // OPENGL INITIALIZATION END
              /* ======================================================================== */

              // wglMakeCurrent(hdc, NULL);
            }
            else
            {
              wsprintf(szMessage, "wglMakeCurrent returned FALSE.\nGetlastError() == %ld\n", GetLastError());
              MessageBox(GetFocus(), szMessage, szAppTitle, MB_SYSTEMMODAL | MB_OK | MB_ICONSTOP);
            }
          }
          else
          {
            wsprintf(szMessage, "wglCreateContext returned NULL.\nGetlastError() == %ld\n", GetLastError());
            MessageBox(GetFocus(), szMessage, szAppTitle, MB_SYSTEMMODAL | MB_OK | MB_ICONSTOP);
          }
        }
        else
        {
          wsprintf(szMessage, "SetPixelFormat returned FALSE.\nGetlastError() == %ld\n", GetLastError());
          MessageBox(GetFocus(), szMessage, szAppTitle, MB_SYSTEMMODAL | MB_OK | MB_ICONSTOP);
        }
      }
      else
      {
        wsprintf(szMessage, "ChoosePixelFormat returned 0.\nGetlastError() == %ld\n", GetLastError());
        MessageBox(GetFocus(), szMessage, szAppTitle, MB_SYSTEMMODAL | MB_OK | MB_ICONSTOP);
      }
      
      uiTimer = SetTimer(hwnd, 1, 1000/60, NULL); // max. 60 Hz
      return 0;
      
    case WM_TIMER:
      angleX += 0.6f;
      if (angleX > 360.0f)
      {
        angleX -= 360.0f;
      }

      angleY += 1.0f;
      if (angleY > 360.0f)
      {
        angleY -= 360.0f;
      }

      angleZ += 0.4f;
      if (angleZ > 360.0f)
      {
        angleZ -= 360.0f;
      }

      translateZ += incZ;
      if (translateZ > 0.5f)
      {
        translateZ = 0.5f;
        incZ = -incZ;
      }
      else if (translateZ < -0.5f)
      {
        translateZ = -0.5f;
        incZ = -incZ;
      }

      factorMorph += incMorph;
      if (factorMorph > MORPH_MAX)
      {
        factorMorph = MORPH_MAX;
        incMorph = -incMorph;
      }
      else if (factorMorph < MORPH_MIN)
      {
        factorMorph = MORPH_MIN;
        incMorph = -incMorph;
      }
      InvalidateRect(hwnd, NULL, FALSE);
      return 0;

    case WM_ERASEBKGND:
      // Important! Prevents GDI from clearing the window client area.
      return 1;

    case WM_SIZE:
      nWidth  = LOWORD(lParam); // width of client area 
      nHeight = HIWORD(lParam); // height of client area 
      glViewport(0, 0, nWidth, nHeight);
      if (nHeight > 0)
      {
        aspectViewport = (float) nWidth / (float) nHeight;
      }
      else
      {
        aspectViewport = 1.0f;
      }
      // FALL THROUGH!!!

    case WM_MOVE:
      bUpdateStencil = TRUE;
      point.x = 0;
      point.y = 0;
      ClientToScreen(hwnd, &point);
      nScreenOffset = point.x;
      return 0;

    case WM_PAINT:
#ifdef QUADBUFFERED_STEREO
      if (bPseudoMonoscopic)
      {
        Display_QuadbufferedMono();
      }
      else
      {
        Display_QuadbufferedStereo();
        SwapBuffers(hdc);
      }
#else // QUADBUFFERED_STEREO
      Display_VerticalInterleave();
      SwapBuffers(hdc);
#endif // QUADBUFFERED_STEREO
      ValidateRect(hwnd, NULL); // Validate the update region of the whole window.
      return 0;

    case WM_LBUTTONDOWN:
      bLButtonDown = TRUE;
      ptLast.x = (long) ((signed short) (lParam & 0xFFFF));
      ptLast.y = (long) lParam >> 16;
      SetCapture(hwnd);
      return 0;

    case WM_RBUTTONDOWN:
      bRButtonDown = TRUE;
      ptLast.x = (long) ((signed short) (lParam & 0xFFFF));
      ptLast.y = (long) lParam >> 16;
      SetCapture(hwnd);
      return 0;

    case WM_LBUTTONUP:
      bLButtonDown = FALSE;
      ReleaseCapture();
      return 0;

    case WM_RBUTTONUP:
      bRButtonDown = FALSE;
      ReleaseCapture();
      return 0;
    
    case WM_MBUTTONUP: // Toggle stereo mono mode.
      bPseudoMonoscopic = !bPseudoMonoscopic;
      InvalidateRect(hwnd, NULL, FALSE);
      return 0;

    case WM_MOUSEMOVE:
      if (bLButtonDown)
      {
        ptCurrent.x = (long) ((signed short) (lParam & 0xFFFF));
        ptCurrent.y = (long) lParam >> 16;
        offset = ptCurrent.x - ptLast.x;
        if (nWidth > 0)
        {
          eyeOffset += 0.2f * (float) offset / (float) nWidth;
        }
        ptLast = ptCurrent;
        InvalidateRect(hwnd, NULL, FALSE);
      }
      else if (bRButtonDown)
      {
        ptCurrent.x = (long) ((signed short) (lParam & 0xFFFF));
        ptCurrent.y = (long) lParam >> 16;
        offset = ptCurrent.x - ptLast.x;
        if (nWidth > 0)
        {
          eyeAdjust += 0.2f * (float) offset / (float) nWidth;
        }
        ptLast = ptCurrent;
        InvalidateRect(hwnd, NULL, FALSE);
      }
      return 0;

    case WM_COMMAND:
      {
        int command;

        command = LOWORD(wParam);
        if (command == IDCANCEL)
        {
          PostMessage(hwnd, WM_DESTROY, 0, 0L);
        }
        else
        {
          CommandHandler(hwnd, command);
        }
        InvalidateRect(hwnd, NULL, FALSE);
      }
      return 0;
        
    case WM_DESTROY:  /* message: window being destroyed */
      if (hglrc != NULL)
      {
        if (hdc != NULL)
        {
          /* ======================================================================== */
          // OPENGL DE-INITIALIZATION START
  
          // Delete DisplayLists etc. here
          glDeleteLists(1, 1);
          // OPENGL DE-INITIALIZATION END
          /* ======================================================================== */
          wglMakeCurrent(hdc, NULL);
          ReleaseDC(hwnd, hdc);
        }
        wglDeleteContext(hglrc);
      }
      PostQuitMessage(0);
      break;
	}
	return (DefWindowProc(hwnd, uMessage, wParam, lParam));
}


// Main drawing routine for Pixelformats with PFD_STEREO support.
// Contains the code for both eyes.
void Display_QuadbufferedStereo(void)
{
  // glViewport() call is done in WM_SIZE message

  // Draw the image for the left eye.
  glDrawBuffer(GL_BACK_LEFT);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Setup the projection for the left eye
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(aspectViewport * -0.75 - eyeAdjust,
            aspectViewport *  0.75 - eyeAdjust,
            -0.75, 0.75, 0.65, 4.0);
  glTranslatef(eyeOffset, 0.0f, 0.0f);  
  glTranslatef(0.0f, 0.0f, -PULL_BACK); 

  // Setup the transformation matrix for the object.
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glRotatef(angleZ, 0.0f, 0.0f, 1.0f);
  glRotatef(angleY, 0.0f, 1.0f, 0.0f);
  glRotatef(angleX, 1.0f, 0.0f, 0.0f);
  glTranslatef(0.0f, 0.0f, translateZ);
  
  DodecahedronMorph();
  glCallList(1);
          
  // Select back right buffer to recieve drawing.
  glDrawBuffer(GL_BACK_RIGHT);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  // Draw the image for the right eye.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(aspectViewport * -0.75 + eyeAdjust,
            aspectViewport *  0.75 + eyeAdjust,
            -0.75, 0.75, 0.65, 4.0);
  glTranslatef(-eyeOffset, 0.0f, 0.0f); 
  glTranslatef(0.0f, 0.0f, -PULL_BACK);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glRotatef(angleZ, 0.0f, 0.0f, 1.0f);
  glRotatef(angleY, 0.0f, 1.0f, 0.0f);
  glRotatef(angleX, 1.0f, 0.0f, 0.0f);
  glTranslatef(0.0f, 0.0f, translateZ);
  
  DodecahedronMorph();
  glCallList(1);
}

// Main drawing routine for Pixelformats with PFD_STEREO support.
// Contains the code for both eyes.
void Display_QuadbufferedMono(void)
{
  // glViewport() call is done in WM_SIZE message

  // Draw the image for both eyes
  // glDrawBuffer(GL_BACK);
  glDrawBuffer(GL_FRONT);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Setup the projection for both eyes.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(aspectViewport * -0.75,
            aspectViewport *  0.75,
            -0.75, 0.75, 0.65, 4.0);
  // glTranslatef(0.0, 0.0f, 0.0f);  
  glTranslatef(0.0f, 0.0f, -PULL_BACK); 

  // Setup the transformation matrix for the object.
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glRotatef(angleZ, 0.0f, 0.0f, 1.0f);
  glRotatef(angleY, 0.0f, 1.0f, 0.0f);
  glRotatef(angleX, 1.0f, 0.0f, 0.0f);
  glTranslatef(0.0f, 0.0f, translateZ);
  
  DodecahedronMorph();
  glCallList(1);
}



#ifndef QUADBUFFERED_STEREO

void UpdateStencil(void)
{
  // Draw vertical stripes into the stencil buffer to partition the left and right buffer.
  // If the first client pixel is on an even screen pixel, use the even pattern.

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0.0, (double) nWidth, 0.0, (double) nHeight, -1.0, 1.0);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);

  glClear(GL_STENCIL_BUFFER_BIT);
  glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
  glStencilFunc(GL_ALWAYS, 1, ~0);
  glEnable(GL_STENCIL_TEST);

  if ((nScreenOffset & 1) == 0)
  {
    glPolygonStipple(patternEven);
  }
  else
  {
    glPolygonStipple(patternOdd);
  }
  glEnable(GL_POLYGON_STIPPLE);
  glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);

  glRecti(0, 0, nWidth, nHeight);
  
  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_POLYGON_STIPPLE);
  glEnable(GL_LIGHTING);
}


// Main drawing routine for ECOMO 4D with stencil buffer filled via PolygonStipple (RECOMMENDED METHOD!!!)
void Display_VerticalInterleave(void)
{
  // glViewport() call is done in WM_SIZE message

  glDrawBuffer(GL_BACK);

  // This is a quick and dirty method to update the stencil only if the window has been moved or resized.
  // The CORRECT thing to do, is to redraw the stencil on each WM_PAINT in the paint region.
  // If this is NOT done, the stencil buffer MAY BE ERASED by overlapping applications!
  // (Try two of these samples simultaneously and you'll see.)
  if (bUpdateStencil)
  {
    UpdateStencil(); // Fast method to get the correct pattern into the stencil buffer.
    bUpdateStencil = FALSE;
  }

  // Clear both back buffers in one step.
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Select back "left" buffer to recieve drawing
  // The left image lies in the 0-stripes.
  glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
  glStencilFunc(GL_EQUAL, 0, ~0); // glDrawBuffer(GL_BACK_LEFT);
  glEnable(GL_STENCIL_TEST);

  // Setup the projection for the left eye
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(aspectViewport * -0.75 - eyeAdjust,
            aspectViewport *  0.75 - eyeAdjust,
            -0.75, 0.75, 0.65, 4.0);
  glTranslatef(eyeOffset, 0.0f, 0.0f);  
  glTranslatef(0.0f, 0.0f, -PULL_BACK); 

  // Setup the transformation matrix for the object.
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glRotatef(angleZ, 0.0f, 0.0f, 1.0f);
  glRotatef(angleY, 0.0f, 1.0f, 0.0f);
  glRotatef(angleX, 1.0f, 0.0f, 0.0f);
  glTranslatef(0.0f, 0.0f, translateZ);
  
  DodecahedronMorph();
  glCallList(1);
          
 
  // Select back "right" buffer to recieve drawing
  glStencilFunc(GL_NOTEQUAL, 0, ~0); // glDrawBuffer(GL_BACK_RIGHT);
  
  // Draw the image for the right eye.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glFrustum(aspectViewport * -0.75 + eyeAdjust,
            aspectViewport *  0.75 + eyeAdjust,
            -0.75, 0.75, 0.65, 4.0);
  glTranslatef(-eyeOffset, 0.0f, 0.0f); 
  glTranslatef(0.0f, 0.0f, -PULL_BACK);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glRotatef(angleZ, 0.0f, 0.0f, 1.0f);
  glRotatef(angleY, 0.0f, 1.0f, 0.0f);
  glRotatef(angleX, 1.0f, 0.0f, 0.0f);
  glTranslatef(0.0f, 0.0f, translateZ);
  
  DodecahedronMorph();
  glCallList(1);
}

#endif // QUADBUFFERED_STEREO


void CommandHandler(HWND hwnd, int command)
{
  switch (command)
  {
    case CMD_KEY_1:
      glEnable(GL_CULL_FACE);
      break;

    case CMD_KEY_2:
      glDisable(GL_CULL_FACE);
      break;
    
    case CMD_KEY_RETURN:
      eyeAdjust = EYE_ADJUST;
      eyeOffset = EYE_OFFSET;
      InvalidateRect(hwnd, NULL, FALSE);
      break;

    case CMD_KEY_ESCAPE:
      PostMessage(hwnd, WM_DESTROY, 0, 0L);
      break;
  }
}




static void TriangleNormalVector(GLfloat a[3], GLfloat b[3], GLfloat c[3], GLfloat n[3])
{
  int i;
  GLfloat d;
  GLfloat v1[3];
  GLfloat v2[3];

  for (i = 0; i < 3; i++)
  {
    v1[i] =  a[i] - b[i];
    v2[i] =  b[i] - c[i];
  }

  n[0] = v1[1] * v2[2] - v1[2] * v2[1];
  n[1] = v1[2] * v2[0] - v1[0] * v2[2];
  n[2] = v1[0] * v2[1] - v1[1] * v2[0];

  d = (GLfloat) sqrt(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]);
  if (fabs(d) > 1.0e-8)
  {
    d = 1.0f / d;
    n[0] *= d;
    n[1] *= d;
    n[2] *= d;
  }
}

void Icosahedron(void)
{
  int i;
  GLfloat norm[3];

  glNewList(1, GL_COMPILE);
    glBegin(GL_TRIANGLES);
    for (i = 0; i < 20; i++)
    {
      TriangleNormalVector(vIco[idxIco[i][0]], vIco[idxIco[i][1]], vIco[idxIco[i][2]], norm);
      glNormal3fv(norm);
      glVertex3fv(vIco[idxIco[i][0]]);
      glVertex3fv(vIco[idxIco[i][1]]);
      glVertex3fv(vIco[idxIco[i][2]]);
    }
    glEnd();
  glEndList();
}


void InitDodecahedron(void)
{
  int     i;
  GLfloat n[3];
  GLfloat d;

  for (i = 0; i < 20; i++)
  {
    n[0] = vIco[idxIco[i][0]][0] + vIco[idxIco[i][1]][0] + vIco[idxIco[i][2]][0];
    n[1] = vIco[idxIco[i][0]][1] + vIco[idxIco[i][1]][1] + vIco[idxIco[i][2]][1];
    n[2] = vIco[idxIco[i][0]][2] + vIco[idxIco[i][1]][2] + vIco[idxIco[i][2]][2];
    
    d = (GLfloat) sqrt(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]);
    if (fabs(d) > 1.0e-8)
    {
      d = 1.0f / d;
      n[0] *= d;
      n[1] *= d;
      n[2] *= d;
    }
    vDode[i][0] = n[0];
    vDode[i][1] = n[1];
    vDode[i][2] = n[2];
  }
}


void Dodecahedron(void)
{
  int i;

  for (i = 0; i < 12; i++)
  {
    glBegin(GL_POLYGON);
      glNormal3fv(vIco[i]);                // Icosahedron coordinate is exactly the normal vector. 
      glVertex3fv(vDode[idxDode[i][0]]);
      glVertex3fv(vDode[idxDode[i][1]]);
      glVertex3fv(vDode[idxDode[i][2]]);
      glVertex3fv(vDode[idxDode[i][3]]);
      glVertex3fv(vDode[idxDode[i][4]]);
    glEnd();
  }
}

void DodecahedronMorph(void)
{
  int i;
  GLfloat v[3];
  GLfloat norm[3];

  for (i = 0; i < 12; i++)
  {
    glBegin(GL_TRIANGLE_FAN);
      v[0] = factorMorph * vIco[i][0];
      v[1] = factorMorph * vIco[i][1];
      v[2] = factorMorph * vIco[i][2];
      TriangleNormalVector(v, vDode[idxDode[i][0]], vDode[idxDode[i][1]], norm);
      glNormal3fv(norm);
      glVertex3fv(v); // center
      glVertex3fv(vDode[idxDode[i][0]]);
      glVertex3fv(vDode[idxDode[i][1]]);
      TriangleNormalVector(v, vDode[idxDode[i][1]], vDode[idxDode[i][2]], norm);
      glNormal3fv(norm);
      glVertex3fv(vDode[idxDode[i][2]]);
      TriangleNormalVector(v, vDode[idxDode[i][2]], vDode[idxDode[i][3]], norm);
      glNormal3fv(norm);
      glVertex3fv(vDode[idxDode[i][3]]);
      TriangleNormalVector(v, vDode[idxDode[i][3]], vDode[idxDode[i][4]], norm);
      glNormal3fv(norm);
      glVertex3fv(vDode[idxDode[i][4]]);
      TriangleNormalVector(v, vDode[idxDode[i][4]], vDode[idxDode[i][0]], norm);
      glNormal3fv(norm);
      glVertex3fv(vDode[idxDode[i][0]]);
    glEnd();
  }
}



void MaterialCreate(void)
{
  GLfloat ambient[]  = {0.0f, 0.0f, 0.0f, 1.0f};
  GLfloat diffuse[]  = {0.764f, 0.831f, 0.129f, 1.0f}; // NVIDIA Corporate Design CMYK=(30.5, 0, 94, 0)
  GLfloat specular[] = {1.0f, 1.0f, 1.0f, 1.0f};
  GLfloat shininess  = 100.0f;

  glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, diffuse);
  glMaterialfv(GL_FRONT, GL_SPECULAR, specular);
  glMaterialf(GL_FRONT, GL_SHININESS, shininess);
}

// Create directional light
void LightCreate(void)
{
  GLfloat light0_ambient[]  = {1.0f, 1.0f, 1.0f, 1.0f};
  GLfloat light0_diffuse[]  = {1.0f, 1.0f, 1.0f, 1.0f};
  GLfloat light0_specular[] = {1.0f, 1.0f, 1.0f, 1.0f};
  GLfloat light0_position[] = {-5.0f, 5.0f, 5.0f, 1.0f};

  GLfloat light1_ambient[]  = {0.5f, 0.5f, 0.5f, 1.0f};
  GLfloat light1_diffuse[]  = {0.5f, 0.5f, 0.5f, 1.0f};
  GLfloat light1_specular[] = {0.5f, 0.5f, 0.5f, 1.0f};
  GLfloat light1_position[] = {5.0f, 0.0f, 0.0f, 1.0f};
  
  glLightfv(GL_LIGHT0, GL_AMBIENT, light0_ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light0_diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light0_specular);
  glLightfv(GL_LIGHT0, GL_POSITION, light0_position);
  
  glLightfv(GL_LIGHT1, GL_AMBIENT, light0_ambient);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, light0_diffuse);
  glLightfv(GL_LIGHT1, GL_SPECULAR, light0_specular);
  glLightfv(GL_LIGHT1, GL_POSITION, light1_position);

  glEnable(GL_LIGHTING);
  
  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
}

