﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Threading;
using System.IO.Ports;
using FTD2XX_NET;

/**************************************************************************
 * Tim Roberts, May 10, 2016
 * 
 * This z-test program uses a timer interrupt
 * to read data from the FTDI D2XX.DLL. It's not as elegant as a true interrupt
 * as it requires continual timer interrupts even when there is no data to receive.
 * Unfortunately, FTDI does not seem to provide a proper event handler in it's library
 * 
 * Aug 10, 2016
 * Added various menu boxes to convert mice from Z to S and to V. Tested and seems to be working.
 * It will also convert S to Z, but it cannot convert V to Z at this time because it is necessary
 * to open a serial port to do this, and the mouse would anyway need to be restarted. Better
 * to just use the v-test program to do this.
 * 
 * Sept 22, 2016
 * 
 * Changes Z-Type to V-Type, but J5 also needs to be removed on most S1-S3 mice.
 * ***************************************************************************/

namespace z_test
{
    public partial class Form1 : Form
    {
        // global variables
        private Int32 xglo, yglo, zglo;
        private Point point1, point2; // we need to keep track of end points for DrawLine
        private Pen black_pen = new Pen(Color.Black, 1); // init the pen for drawing
        private Graphics form_graph; // can't init here, init in Draw_Click
        private Color back_color; // can't init here, init in Draw_Click

        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public static bool _continue = true;

        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public string PortName = "Com1"; // default

        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public Form1()
        {
            InitializeComponent();
        }

        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public delegate void AddDataDelegate(String myString);
        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public AddDataDelegate myDelegate;

        private void Exit_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void Form1_Load(object sender, EventArgs e) { } // required

        /*********************************************************************************
         * HELPER FUNCTIONS to help implement clicks above
         * ********************************************************************************/

        private void Go_Click(object sender, EventArgs e)
        {
            // The Button control has the label READ on it.
            D2XXPortInit();
            InitGraphTimer();
        }

        // Create new instance of the FTDI device class
        FTDI myFtdiDevice = new FTDI();

        /****************************************************************************
         * Open D2XX port to mouse
         * ******************************************************************************/

        private void D2XXPortInit()
        {
            UInt32 ftdiDeviceCount = 0;
            FTDI.FT_STATUS ftStatus = FTDI.FT_STATUS.FT_OK;

            textBox1.Text = "";

            // Determine the number of FTDI devices connected to the machine
            ftStatus = myFtdiDevice.GetNumberOfDevices(ref ftdiDeviceCount);
            if (ftStatus == FTDI.FT_STATUS.FT_OK) // Check status
            {
                textBox1.Text = "Found device, status";
            }
            else
            {
                textBox1.Text = "Failed to find device, status";
                return;
            }

            // If no devices available, return
            if (ftdiDeviceCount == 0)
            {
                textBox1.Text = "No devices found";
                return;
            }

            // Allocate storage for device info list
            FTDI.FT_DEVICE_INFO_NODE[] ftdiDeviceList = new FTDI.FT_DEVICE_INFO_NODE[ftdiDeviceCount];

            // Populate our device list
            ftStatus = myFtdiDevice.GetDeviceList(ftdiDeviceList);

            // Open first device in our list by serial number
            ftStatus = myFtdiDevice.OpenBySerialNumber(ftdiDeviceList[0].SerialNumber);
            if (ftStatus != FTDI.FT_STATUS.FT_OK)
            {
                textBox1.Text = "Failed to open by serial number";
                return;
            }

            ftStatus = myFtdiDevice.SetBaudRate(9600);
            // Set data characteristics - Data bits, Stop bits, Parity
            ftStatus = myFtdiDevice.SetDataCharacteristics(FTDI.FT_DATA_BITS.FT_BITS_8, FTDI.FT_STOP_BITS.FT_STOP_BITS_1, FTDI.FT_PARITY.FT_PARITY_NONE);

            // Set flow control - set RTS/CTS flow control
            // ftStatus = myFtdiDevice.SetFlowControl(FTDI.FT_FLOW_CONTROL.FT_FLOW_RTS_CTS, 0x11, 0x13);

            // Set read timeout to 5 seconds, write timeout to infinite
//            ftStatus = myFtdiDevice.SetTimeouts(5000, 0);

        } // End of D2XXPortInit


        /*************************************************************
        * InitGraphTimer
        * Init graphics and timer
        * ***********************************************************/
        void InitGraphTimer()
        {
            // init graphics
            form_graph = this.CreateGraphics(); // get a handle for the form so we can draw
            back_color = this.BackColor; // get the form background color for clearing screen
            point1.X = 500; point1.Y = 300; // need to init starting point for drawing

            // init timer
            timer1.Interval = 10; // 10ms equals 100Hz, same rate as mouse data
            timer1.Tick += new EventHandler(timer1_Tick); // not done automatically by C#
            timer1.Enabled = true;
            this.myDelegate = new AddDataDelegate(AddDataMethod);
        }


        /*************************************************************
        * Process data sent by the timer event function
        * add new data to any leftover from last time
        * check if there is a CR in the data and process, else return
        * ***********************************************************/
        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public string local = ""; // store leftover values in global string

        ///<Summary>
        /// Gets the answer
        ///</Summary>
        public void AddDataMethod(String message) // new data
        {
            bool good;
            int index, len;
            string strs, strx, stry, strz;
            strs = ""; // already defined elsewhere
            int spos, xpos, ypos, zpos, vpos, rpos, cpos;
            int numb;
            local += message; // add new data to leftover data
            string mess="";

            // The data may have several data sets because we only read a few (100) times a second
            // loop trough each data set until there are none left
            // each data set will end with CR
Loop:       // loop until no more data sets
            //  look for CR at end of first data set
            index = local.IndexOf("\r");

            // separate first data set from global string
            if (index > -1)
            {
                mess = local.Substring(0, index + 1); // separate the first data set from the global string
                len = local.Length;
                if (index + 1 == len) local = ""; // no more data, make sure global string is empty
                else local = local.Substring(index + 1, len - index - 1); // remove first data set from global
                //textBox1.Text = local; // display leftover data for debugging
            }
            else
            {
                return;
            }

            // got data, so find out what it holds and process it
            spos = mess.IndexOf("S") + 1;
            xpos = mess.IndexOf("X") + 1;
            ypos = mess.IndexOf("Y") + 1;
            zpos = mess.IndexOf("Z") + 1;
            rpos = mess.IndexOf("\r") + 1; 
            vpos = mess.IndexOf("V"); // version number
            cpos = mess.IndexOf("C"); // calibration values
        
            if (spos > 0) // get the switch/button string
            {
                if (xpos > spos) strs = mess.Substring(spos, xpos - spos - 1);
                else if (ypos > spos) strs = mess.Substring(spos, ypos - spos - 1);
                else if (zpos > spos) strs = mess.Substring(spos, zpos - spos - 1);
                else if (rpos > spos) strs = mess.Substring(spos, rpos - spos - 1);
                else strs = mess.Substring(spos);
                Buttons.Text = String.Copy(strs);
            }

            if (xpos > 0) // get the x string
            {
                if (ypos > xpos) strx = mess.Substring(xpos, ypos - xpos - 1);
                else if (zpos > xpos) strx = mess.Substring(xpos, zpos - xpos - 1);
                else if (rpos > xpos) strx = mess.Substring(xpos, rpos - xpos - 1);
                else strx = mess.Substring(xpos);
                good = int.TryParse(strx, out numb);
                if (good) xglo += numb;
                strx = xglo.ToString();
                Xtext.Text = String.Copy(strx);
            }

            if (ypos > 0) // get the y string
            {
                if (zpos > ypos) stry = mess.Substring(ypos, zpos - ypos - 1);
                if (rpos > ypos) stry = mess.Substring(ypos, rpos - ypos - 1);
                else stry = mess.Substring(ypos);
                good = int.TryParse(stry, out numb);
                if (good) yglo -= numb;
                stry = yglo.ToString();
                Ytext.Text = String.Copy(stry);
            }

            if (zpos > 0) // get the z string
            {
                if (rpos > zpos) stry = mess.Substring(zpos, rpos - zpos - 1);
                strz = mess.Substring(zpos);
                good = int.TryParse(strz, out numb);
                if (good) zglo += numb;
                strz = zglo.ToString();
                Ztext.Text = String.Copy(strz);
            }

            if (vpos > 0) // get the version string
            {
                textBox1.Text = mess;
            }

            if (cpos > 0) // get the calibration string
            {
                textBox1.Text = mess;
            }

            // trace mouse movement on the screen
            // if button 2 pressed, clear the drawing
            if (strs == "2") form_graph.Clear(back_color); // clear the form if button 2 pressed
            point2.X = xglo / 2 + 500 + zglo / 2; // scale and shift data for display
            point2.Y = yglo / 2 + 300 - zglo / 2;
            form_graph.DrawLine(black_pen, point1, point2); // draw a line from last point to new point
            point1 = point2; // next time we start drawing from the new point

            goto Loop; // loop back to find more data

            // return is done at beginning of loop
        }

        /**********************************************************
        * The timer_tick is called each time the timer expires
        * Read data and send for processing via delegate
        * ***********************************************************/
        private void timer1_Tick(object sender, EventArgs e)
        {
            // Check the amount of data available to read
            UInt32 numBytesAvailable = 0;
            FTDI.FT_STATUS ftStatus;
            ftStatus = myFtdiDevice.GetRxBytesAvailable(ref numBytesAvailable);

            if (numBytesAvailable > 0) // if there is data, read it and send for processing.
                {
                string readData;
                UInt32 numBytesRead = 0;
                // Note that the Read method is overloaded, so can read string or byte array data
                ftStatus = myFtdiDevice.Read(out readData, numBytesAvailable, ref numBytesRead);
                textBox1.Invoke(this.myDelegate, new Object[] { readData });
                }
        }

        /***************************************************************
        * stop timer and close mouse connection
        * ***************************************************************/
        private void Close1_Click(object sender, EventArgs e)
        {
                timer1.Enabled = false; // turn off timer
                FTDI.FT_STATUS ftStatus = myFtdiDevice.Close(); // close device
                textBox1.Text = "";
        }

        /****************************************************************
         * Modify FTDI chip PID to 0xDBB8. This requires the modified Stealth driver.
         * This driver does not work with Win 8 onwards.
         * *************************************************************/
        private void Win7_Click(object sender, EventArgs e)
        {
            Close1_Click(sender, e); // make sure we are not drawing

            if (MessageBox.Show("NOT RECOMMENDED: This may require a new driver to be loaded, specifically version 2.06.00 or later from Stealth3D Mouse web site only. This works with Z-Type only, so if not a Z-Type, convert to new Z-Type first. Do you agree to this change?", "Question",
                MessageBoxButtons.YesNo) == DialogResult.Yes)
            {

                Cursor.Current = Cursors.WaitCursor;
                Program_Chip(0); // Win 7
                Cursor.Current = Cursors.Default;
                MessageBox.Show("Mouse modification is complete for use with Win2000 to Win7. Download driver version 2.06.00 or later from www.Stealth3DMouse.com and uncompress into a temporary directory. Unplug mouse and wait one minute before pluging in again.");
                Exit_Click(sender, e); // funny, but Application.Exit() does not work here, so go to Exit_Click instead
            }
            else
            {
                return; // user answered No.
            }

        } // End of Win 7 programming

        /****************************************************************
         * Modify FTDI chip PID to 0x6001 so allow use with unmodified FTDI driver.
         * This is required to use Win 8 and later, but still works with earlier operating systems.
         * *************************************************************/
        private void Win8_Click(object sender, EventArgs e)
        {
            bool result;
            Close1_Click(sender, e); // make sure we are not drawing

            if (MessageBox.Show("RECOMMENDED for all Stealth Z-Type mice for use with all Windows versions from Win2000 to Win 10. Also for converting V-Type and S-Type mouse to Z-Type. This may require a new driver to be loaded, specifically version 2.08.30 or later from FTDIchip website or www.Stealth3DMouse.com web site. Do you agree to this change?", "Question",
                MessageBoxButtons.YesNo) == DialogResult.Yes)
            {
                Cursor.Current = Cursors.WaitCursor;
                result = v2s(); // testing v to Z. // with v2s in, it does V to Z. With it out, it does S to Z.
                if (!result) { Application.Exit(); Exit_Click(sender, e); } // seems to be a problem with App.Exit
                Program_Chip(1); // Win 8
                Cursor.Current = Cursors.Default;

                MessageBox.Show("Mouse modification is complete for use with all Windows versions for S4. If converting an S1-V, S2-V or S3-V, you must additionally unplug and open mouse shell and install jumper J5. If necessary, it should be automatic, download driver version 2.08.30 or later and uncompress into a temporary directory. Unplug mouse and wait one minute before pluging in again.");
                Exit_Click(sender, e); // funny, but Application.Exit() does not work here, so go to Exit_Click instead
            }
            else
            {
                return; // user answered No.
            }
        } // End of Win 8-10 programming


        /****************************************************************
         * Change Z to V-Type
         * Change PID to 0x6001
         * Remove jumper J5 if present 
         * *************************************************************/
        private void VMouseButton_Click(object sender, EventArgs e)
        {
            Close1_Click(sender, e); // close device if open

            if (MessageBox.Show("Changing mouse to V-Type. S1, S2, S3 mice also require removing a jumper (J5) on the processor board. Click Yes, wait for this program to finish, then unplug mouse, open and remove jumper, then replug. With S4, there is nothing else required.", "Question",
                MessageBoxButtons.YesNo) == DialogResult.Yes)
            {
                Cursor.Current = Cursors.WaitCursor;
                Program_Chip(2); // 2 = V Type
                Cursor.Current = Cursors.Default;
                MessageBox.Show("Mouse has been changed to V type. For S1, S2, S3 mice unplug mouse, open and remove jumper J5. Close mouse and re-plug. Exit this program to continue. With S4, there is nothing else to do.\r\n");
                Exit_Click(sender, e); // funny, but Application.Exit() does not work here, so go to Exit_Click instead
            }
            else
            {
                return; // user answered No.
            }

        } // End of VMouseButton

        /****************************************************************
         * Change Z to S. New driver will be needed
         * Change PID to 0x6001
         * There is no change to J5 
         * *************************************************************/
        private void Smousebutton_Click(object sender, EventArgs e)
        {
            Close1_Click(sender, e); // close device if open

            if (MessageBox.Show("Changing this mouse to S-Type. Do you agree to this change?", "Question",
                MessageBoxButtons.YesNo) == DialogResult.Yes)
            {
                Cursor.Current = Cursors.WaitCursor;
                Program_Chip(3); // 3 = S Type
                Cursor.Current = Cursors.Default;
                MessageBox.Show("Mouse has been changed to S-type. Exit this program to continue.\r\n");
                Exit_Click(sender, e); // funny, but Application.Exit() does not work here, so go to Exit_Click instead
            }
            else
            {
                return; // user answered No.
            }

        } // End of Smousebutton

        /*****************************************************************
         * Program_Chip will program the FTDI chip to Win 7 or Win 8 type
         *  If the input value type is 0 = Z-Type Win 7, 1 = Z-Type Win 8, 2 = V-Type, 3 = S-Type
         *  ****************************************************************/
         ///<Summary>
        /// Gets the answer
        ///</Summary>
       public void Program_Chip(int type)
        {
            label2.Visible=true;

            UInt32 ftdiDeviceCount = 0;
            FTDI.FT_STATUS ftStatus = FTDI.FT_STATUS.FT_OK;

            // Create new instance of the FTDI device class
            FTDI myFtdiDevice = new FTDI();

            // Determine the number of FTDI devices connected to the machine
            ftStatus = myFtdiDevice.GetNumberOfDevices(ref ftdiDeviceCount);
            // Check status
            if (ftStatus == FTDI.FT_STATUS.FT_OK)
            {
                textBox1.Text = "Number of FTDI devices: " + ftdiDeviceCount.ToString();
                if (ftdiDeviceCount == 0) return;
            }
            else
            {
                textBox1.Text = "No devices installed";
                return;
            }

            // Allocate storage for device info list
            FTDI.FT_DEVICE_INFO_NODE[] ftdiDeviceList = new FTDI.FT_DEVICE_INFO_NODE[ftdiDeviceCount];

            // Populate our device list
            ftStatus = myFtdiDevice.GetDeviceList(ftdiDeviceList);

            if (ftStatus == FTDI.FT_STATUS.FT_OK)
            {
                for (UInt32 i = 0; i < ftdiDeviceCount; i++)
                {
                    textBox1.Text = "Device Index: " + i.ToString();
                    textBox1.Text += "Flags: " + String.Format("{0:x}", ftdiDeviceList[i].Flags);
                    textBox1.Text += "Type: " + ftdiDeviceList[i].Type.ToString();
                    textBox1.Text += "ID: " + String.Format("{0:x}", ftdiDeviceList[i].ID);
                    textBox1.Text += "Location ID: " + String.Format("{0:x}", ftdiDeviceList[i].LocId);
                    textBox1.Text += "Serial Number: " + ftdiDeviceList[i].SerialNumber.ToString();
                    textBox1.Text += "Description: " + ftdiDeviceList[i].Description.ToString();
                    // Make sure it's not someone elses chip before proceeding
                    //if ((ftdiDeviceList[i].ID != "403dbb8") && (ftdiDeviceList[i].ID != "4036001")) //not ours;
                    //if ((ftdiDeviceList[i].Description != "USB Serial Converter") && (ftdiDeviceList[i].Description != "Stealth Z-Mouse V01")) // not ours
 
                    Application.DoEvents();
                }
            }

            // Open first device in our list by serial number
            ftStatus = myFtdiDevice.OpenBySerialNumber(ftdiDeviceList[0].SerialNumber);
            if (ftStatus != FTDI.FT_STATUS.FT_OK)
            {
                textBox1.Text += "Failed to open device";
                return;
            }

            // Create our device EEPROM structure based on the type of device we have open
            // we used two different FTDI chips, this is the first, 232R.
 // 232R Device
            if (ftdiDeviceList[0].Type == FTDI.FT_DEVICE.FT_DEVICE_232R)
            {
                // We have an FT232R or FT245R so use FT232R EEPROM structure
                FTDI.FT232R_EEPROM_STRUCTURE myEEData = new FTDI.FT232R_EEPROM_STRUCTURE();
                // Read the device EEPROM
                try
                {
                    ftStatus = myFtdiDevice.ReadFT232REEPROM(myEEData);
                    if (ftStatus != FTDI.FT_STATUS.FT_OK)
                    {
                        myFtdiDevice.Close();
                        return;
                    }
                }
                catch (FTDI.FT_EXCEPTION)
                {
                    textBox1.Text = "Exception thrown when calling ReadFT232REEPROM";
                    return;
                }

                // Set PID
                if (type == 0)
                { myEEData.ProductID = 0xDBB8;}
                else
                { myEEData.ProductID = 0x6001;}

                // Set descriptions
                if (type > 1) // keep "Stealth" on updated mice 
                {
                    myEEData.Manufacturer = "FTDI";
                    myEEData.Description = "USB Serial Converter";
                }
                else
                {
                    myEEData.Manufacturer = "ABC-SD";
                    myEEData.Description = "Stealth Z-Mouse V01";
                }

                // fix Z-Type serial number. Others can have any serial number but we fix them
                myEEData.SerNumEnable = true; // serial number is fixed
                if (type > 1)
                {
                    myEEData.RIsD2XX = false; // load VCP
                    myEEData.SerialNumber = "PR123456"; // serial number for V-Type and S-Type
                }
                else
                {                    
                    myEEData.RIsD2XX = true; // load D2XX
                    myEEData.SerialNumber = "PR000001"; // serial number for Z-Type
                }
                myEEData.MaxPower = 500;
                myEEData.RemoteWakeup = false;
                myEEData.SelfPowered = false;

                // Write our modified data structure back to the device EEPROM
                try
                {
                    ftStatus = myFtdiDevice.WriteFT232REEPROM(myEEData);
                    if (ftStatus != FTDI.FT_STATUS.FT_OK)
                    {
                        myFtdiDevice.Close();
                        return;
                    }
                }
                catch (FTDI.FT_EXCEPTION)
                {
                    textBox1.Text = "Exception thrown when calling WriteFT232REEPROM";
                }
                // Write common EEPROM elements
                textBox1.Text = " EEPROM Contents for device at index 0:";
                textBox1.Text += " Vendor ID: " + String.Format("{0:x}", myEEData.VendorID);
                textBox1.Text += " Product ID: " + String.Format("{0:x}", myEEData.ProductID);
                textBox1.Text += " Manufacturer: " + myEEData.Manufacturer.ToString();
                textBox1.Text += " Manufacturer ID: " + myEEData.ManufacturerID.ToString();
                textBox1.Text += " Description: " + myEEData.Description.ToString();
                textBox1.Text += " Serial Number: " + myEEData.SerialNumber.ToString();
                textBox1.Text += " Max Power: " + myEEData.MaxPower.ToString() + "mA";
                textBox1.Text += " Self Powered: " + myEEData.SelfPowered.ToString();
                textBox1.Text += " Remote Wakeup Enabled: " + myEEData.RemoteWakeup.ToString();
                textBox1.Text += " Load D2XX: " + myEEData.RIsD2XX.ToString();
                Application.DoEvents();
            }

            // we used two different FTDI chips, this is the second, 232BM
   // 232BM Device
            else if (ftdiDeviceList[0].Type == FTDI.FT_DEVICE.FT_DEVICE_BM)
            {
                // We have an FT232B or FT245B so use FT232B EEPROM structure
                FTDI.FT232B_EEPROM_STRUCTURE myEEData = new FTDI.FT232B_EEPROM_STRUCTURE();
                ftStatus = myFtdiDevice.ReadFT232BEEPROM(myEEData);
                try
                {
                    ftStatus = myFtdiDevice.ReadFT232BEEPROM(myEEData);
                }
                catch (FTDI.FT_EXCEPTION)
                {
                    textBox1.Text = "Exception thrown when calling ReadFT232BEEPROM";
                }

                if (ftStatus != FTDI.FT_STATUS.FT_OK)
                {
                    // Wait for a key press
                    textBox1.Text = "Failed to read device EEPROM (error " + ftStatus.ToString() + ")";
                    myFtdiDevice.Close();
                    return;
                }

                // Set PID
                if (type == 0)
                { myEEData.ProductID = 0xDBB8; }
                else
                { myEEData.ProductID = 0x6001; }

                // fix Z-Type serial number. Others can have any serial number but we fix them
                myEEData.SerNumEnable = true; // serial number is fixed

                // Change description and serial number to write back to device
                if (type > 1)
                {
                    myEEData.Manufacturer = "FTDI";
                    myEEData.Description = "USB Serial Converter";
                    myEEData.SerialNumber = "PR123456"; // let driver give S/N
                }
                else
                {
                    myEEData.Manufacturer = "ABC-SD";
                    myEEData.Description = "Stealth Z-Mouse V01";
                    myEEData.SerialNumber = "PR000001"; // let driver give S/N
                }

                // The BM version does not have the D2XX / VCP selection, so it must be done manually with MProg

                myEEData.MaxPower = 500;
                myEEData.RemoteWakeup = false;
                myEEData.SelfPowered = false;

                // Write our modified data structure back to the device EEPROM
                // This can throw an exception if trying to write a device type that does not 
                // match the EEPROM structure being used, so should always use a 
                // try - catch block when calling
                try
                {
                    ftStatus = myFtdiDevice.WriteFT232BEEPROM(myEEData);
                    textBox1.Text += "Success changing S/N";
                }
                catch (FTDI.FT_EXCEPTION)
                {
                    textBox1.Text += "Exception thrown when calling WriteFT232BEEPROM";
                }

                if (ftStatus != FTDI.FT_STATUS.FT_OK)
                {
                    // Wait for a key press
                    textBox1.Text += "Failed to write device EEPROM (error " + ftStatus.ToString() + ")";
                    myFtdiDevice.Close();
                    return;
                }
                // Write common EEPROM elements to our console
                textBox1.Text = " EEPROM Contents for device at index 0:";
                textBox1.Text += " Vendor ID: " + String.Format("{0:x}", myEEData.VendorID);
                textBox1.Text += " Product ID: " + String.Format("{0:x}", myEEData.ProductID);
                textBox1.Text += " Manufacturer: " + myEEData.Manufacturer.ToString();
                textBox1.Text += " Manufacturer ID: " + myEEData.ManufacturerID.ToString();
                textBox1.Text += " Description: " + myEEData.Description.ToString();
                textBox1.Text += " Serial Number: " + myEEData.SerialNumber.ToString();
                textBox1.Text += " Max Power: " + myEEData.MaxPower.ToString() + "mA";
                textBox1.Text += " Self Powered: " + myEEData.SelfPowered.ToString();
                textBox1.Text += " Remote Wakeup Enabled: " + myEEData.RemoteWakeup.ToString();
                Application.DoEvents();

            }

            if (type == 2) // changing to V Type
            {
                uint num = 0; // must be uint, not int
                string a = "I"; // change to v type
                ftStatus = myFtdiDevice.Write(a, 1, ref num); // write 'I' to mouse to change protocol (works with S4 only)
                if (ftStatus != FTDI.FT_STATUS.FT_OK)
                textBox1.Text += "Write I error";                
            }

            // Use cycle port to force a re-enumeration of the device.  

            // Documentation says to just Cycle Port and Close will be called
            // For some reason, I need to close, reopen and then cycle
            // without this, either power needs to be cycled manually, or the optics calibration will be lost
            // or the Z/V switch will be lost 
            ftStatus = myFtdiDevice.Close();
            ftStatus = myFtdiDevice.OpenBySerialNumber(ftdiDeviceList[0].SerialNumber);
            ftStatus = myFtdiDevice.CyclePort();

            // Wait for device to be re-enumerated
            do
            {
                // The device will have the same location since it has not been 
                // physically unplugged, so we will keep trying to open it until it succeeds
                ftStatus = myFtdiDevice.OpenByLocation(ftdiDeviceList[0].LocId);
                Thread.Sleep(250);
            } while (ftStatus != FTDI.FT_STATUS.FT_OK);

            label2.Visible = false;
            ftStatus = myFtdiDevice.Close();
            Application.DoEvents();        
        } // End of Program_chip


        /* dummy proc */
        private void label1_Click(object sender, EventArgs e)
        {
        }


        /**************************************
         * Serial port section
         * ***********************************/
        bool port_open;
        string last_port = "";
        string ComA = "";
        string ComB = "";
        string ComC = "";


        // close the open com port
        private void ClosePort() // close port and timer off
        {
            timer1.Enabled = false;
            if (port_open)
            {
                serialPort1.Write("BEGIN E"); // write begin and end
                serialPort1.Close();
                port_open = false;
            }
        }

        // open the requested COM port
        private bool OpenPort(string name)  // open port and set parameters
        {
            if (port_open)
            {
                textBox1.Text = "Port open error";
                return true; // port already open, return
            }
            serialPort1.PortName = name;
            try // do these things, watching for errors
            {
                serialPort1.Open();
                port_open = true;
                serialPort1.BaudRate = 9600;//nvert displayed value to int
                serialPort1.Parity = Parity.None;
                serialPort1.StopBits = StopBits.One;
                serialPort1.DataBits = 8;
                serialPort1.Handshake = Handshake.None;
                serialPort1.RtsEnable = true;
                serialPort1.ReadTimeout = 500;
                serialPort1.WriteTimeout = 500;
            }
            catch { port_open = false; } // if error, do these things
            return port_open;
        }

        // init the Immersion type mouse
        private bool InitIMMC()
        {
            int i;
            string buff;

            if (port_open == false) return false;

            // check to see if the mouse is a V type which requires no action
            for (i = 0; i < 20; i++)
            {
                serialPort1.Write("\r");
                System.Threading.Thread.Sleep(5); // wait 5 ms
                buff = serialPort1.ReadExisting();
                if (buff.Contains("V"))
                {
                    serialPort1.Close();
                    port_open = false;
                    textBox1.Text = "Z-Type Mouse"; // no action needed
                    return false;
                }
            }
            for (i = 0; i < 50; i++)
            {
                serialPort1.Write("IMMC");
                System.Threading.Thread.Sleep(5); // wait 5 ms
                buff = serialPort1.ReadExisting();
                if (buff.Contains("C"))
                {
                    textBox1.Text = "IMMC Connected"; // Got it
                    return true;
                }
            }
            serialPort1.Close();
            port_open = false;
            textBox1.Text = "No IMMC";
            return false;

        } // end InitIMMC


 /**************************************************************
 * Change V-Type to S-Type
  * 1. Open COM and send Z, or remove jumper
  * 2. Run Program Chip to make it into Z-Type. Stop there if Z required
  * 3. Run Program chip again to make it S-Type.
 * **************************************************************/
        private bool v2s()
        {
            DialogResult result = MessageBox.Show(
            "For S1, S2, S3, click 'OK' if you have already removed J5, otherwise, click 'Cancel', unplug mouse, remove the jumper on J5, and re-plug. Then run this program again.\r\n" +
            "For S4 mice Click 'OK'", "", MessageBoxButtons.OKCancel);
            if (result == DialogResult.Cancel)
            {
                return false;
            }

                // Get a list of serial port names, put name in each button
                string[] ports = SerialPort.GetPortNames();
                try { ComA = ports[0]; }
                catch { }
                try { ComB = ports[1]; }
                catch { }
                try { ComC = ports[2]; }
                catch { }

                if (last_port.Length == 0) last_port = ComC;
                if (last_port.Length == 0) last_port = ComB;
                if (last_port.Length == 0) last_port = ComA;
            try { serialPort1.Close(); }
            catch { }
            try { if (OpenPort(last_port)) port_open = true; }
            catch { }
                if (port_open == false)
                {
                    MessageBox.Show("Error opening port " + last_port);
                    return false;
                }

                if (InitIMMC()) // init mouse
                {
                    serialPort1.Write("BEGIN"); // Begin an Immersion session to change protocol
                    System.Threading.Thread.Sleep(5); // wait 5 ms
                    serialPort1.Write("Z"); // change protocol to Z
                    textBox1.Text += "Z"; // report
                    serialPort1.Write("S"); // save calibration INCLUDING Z
                    serialPort1.Write("E"); // End the Immersion session
                }
                    serialPort1.Close();
                    port_open = false;
                
                return true;
        }






    } // End of Form1


} // namespace
