/*----------------------------------------------------------------------
*  math3d.cpp -- 3d maths implementation file
*
*----------------------------------------------------------------------
*
* (c) 1998-2007 3Dconnexion. All rights reserved. 
* Permission to use, copy, modify, and distribute this software for all
* purposes and without fees is hereby grated provided that this copyright
* notice appears in all copies.  Permission to modify this software is granted
* and 3Dconnexion will support such modifications only is said modifications are
* approved by 3Dconnexion.
*
*/
#include "stdafx.h"
#include "math3d.h"
#include <cmath>


void MathFrameTranslation( MathFrame *Frame, real X, real Y, real Z )
{
   Frame->MathTranslation[0] = -X;
   Frame->MathTranslation[1] = -Y;
   Frame->MathTranslation[2] = Z;
}

void MathFrameRotation( MathFrame *Frame, real Roll, real Pitch, real Yaw )
{
   real sa,ca, sb,cb, sc,cc;
   real sacc, casc, sasc, cacc;

   sa = (real)-sin(Yaw);	 ca = (real) cos(Yaw);
   sb = (real)-sin(Pitch); cb = (real) cos(Pitch);
   sc = (real) sin(Roll);	 cc = (real) cos(Roll);

   Frame->MathRotation[0][0] = ca*cb;
   Frame->MathRotation[1][0] =-sa*cb;
   Frame->MathRotation[2][0] = -sb;

   sacc = sa*cc;
   casc = ca*sc;
   sasc = sa*sc;
   cacc = ca*cc;

   Frame->MathRotation[0][1] = sacc - casc*sb;
   Frame->MathRotation[1][1] = cacc + sasc*sb;
   Frame->MathRotation[2][1] = -cb*sc;

   Frame->MathRotation[0][2] = sasc + cacc*sb;
   Frame->MathRotation[1][2] = casc - sacc*sb;
   Frame->MathRotation[2][2] = cb*cc;

}

void MathFrameMultiplication( MathFrame *FrameA, MathFrame *FrameB, MathFrame *FrameAB )
{
   MathFrame FrameResult;
   int m,n,k;

   for ( m=0; m<FRAMEROTm; ++m )
      for ( n=0; n<FRAMEROTn; ++n )
      {
         FrameResult.MathRotation[m][n] = (real) 0.0;
         for ( k=0; k<FRAMEROTn; ++k )
            FrameResult.MathRotation[m][n] += FrameA->MathRotation[m][k] *
            FrameB->MathRotation[k][n];
      };
   for ( m=0; m<FRAMETRA; ++m )
      FrameResult.MathTranslation[m] = FrameA->MathTranslation[m] +
      FrameB->MathTranslation[m];
   memcpy( FrameAB, &FrameResult, sizeof( FrameResult ) );
}

void MathFrameTransformation( MathFrame *Frame, MathPoint *PointIn, MathPoint *PointOut )
{
   MathPoint PointResult;
   int m,n;

   for ( m=0; m<FRAMEROTm; ++m )
   {
      PointResult.MathTranslation[m] = Frame->MathTranslation[m];
      for ( n=0; n<FRAMEROTn; ++n )
         PointResult.MathTranslation[m] += Frame->MathRotation[m][n] * PointIn->MathTranslation[n];
   };
   memcpy( PointOut, &PointResult, sizeof( PointResult ) );
}

void MathVectorSubtraction( MathPoint *Point1, MathPoint *Point2, MathPoint *Vector )
{
   Vector->MathTranslation[0] = Point1->MathTranslation[0] - Point2->MathTranslation[0];
   Vector->MathTranslation[1] = Point1->MathTranslation[1] - Point2->MathTranslation[1];
   Vector->MathTranslation[2] = Point1->MathTranslation[2] - Point2->MathTranslation[2];
}

real MathVectorScalar( MathPoint *Point1, MathPoint *Point2 )
{
   return Point1->MathTranslation[0]*Point2->MathTranslation[0] +
      Point1->MathTranslation[1]*Point2->MathTranslation[1] +
      Point1->MathTranslation[2]*Point2->MathTranslation[2];
}

void MathVectorProduct( MathPoint *Point1, MathPoint *Point2, MathPoint *Product )
{
   MathPoint ProductResult;

   ProductResult.MathTranslation[0] = Point1->MathTranslation[1] *
      Point2->MathTranslation[2] -
      Point1->MathTranslation[2] *
      Point2->MathTranslation[1];
   ProductResult.MathTranslation[1] = Point1->MathTranslation[2] *
      Point2->MathTranslation[0] -
      Point1->MathTranslation[0] *
      Point2->MathTranslation[2];
   ProductResult.MathTranslation[2] = Point1->MathTranslation[0] *
      Point2->MathTranslation[1] -
      Point1->MathTranslation[1] *
      Point2->MathTranslation[0];
   memcpy( Product, &ProductResult, sizeof( ProductResult ) );
}

void Math2DTransformation( MathPoint *PointIn, MathPoint *PointOut, real Focus )
{
   PointOut->MathTranslation[0] = Focus * PointIn->MathTranslation[0] / PointIn->MathTranslation[2];
   PointOut->MathTranslation[1] = Focus * PointIn->MathTranslation[1] / PointIn->MathTranslation[2];
   PointOut->MathTranslation[2] = (real) 0.0;
}

void Math2DCoordinate( MathPoint *PointIn, MathPoint *PointOut, MathVideo *Parameter )
{
   PointOut->MathTranslation[0] = PointIn->MathTranslation[0] *
      Parameter->VideoMaxX * Parameter->VideoRatio +
      Parameter->VideoMaxX/2;
   PointOut->MathTranslation[1] =-PointIn->MathTranslation[1] *
      Parameter->VideoMaxY +
      Parameter->VideoMaxY/2;
   PointOut->MathTranslation[2] = 0.0;
}



#define 	BROWN				RGB(0,255,255)
#define 	YELLOW 			RGB(255,255,0)
#define  GREEN				RGB(0,191,0)
#define  LIGHTRED			RGB(255,0,0)
#define	LIGHTMAGENTA   RGB(191,0,191)
#define	LIGHTBLUE		RGB(0,0,191)

#define CubeSize		1.0

MathPoint P1 = { { +CubeSize, -CubeSize, +CubeSize } };
MathPoint P2 = { { +CubeSize, -CubeSize, -CubeSize } };
MathPoint P3 = { { +CubeSize, +CubeSize, -CubeSize } };
MathPoint P4 = { { +CubeSize, +CubeSize, +CubeSize } };
MathPoint P5 = { { -CubeSize, -CubeSize, +CubeSize } };
MathPoint P6 = { { -CubeSize, -CubeSize, -CubeSize } };
MathPoint P7 = { { -CubeSize, +CubeSize, -CubeSize } };
MathPoint P8 = { { -CubeSize, +CubeSize, +CubeSize } };

MathPlane Pl1 = { { &P4, &P3, &P2, &P1 }, BROWN };
MathPlane Pl2 = { { &P8, &P4, &P1, &P5 }, YELLOW };
MathPlane Pl3 = { { &P8, &P5, &P6, &P7 }, GREEN };
MathPlane Pl4 = { { &P7, &P6, &P2, &P3 }, LIGHTRED };
MathPlane Pl5 = { { &P8, &P7, &P3, &P4 }, LIGHTMAGENTA };
MathPlane Pl6 = { { &P6, &P5, &P1, &P2 }, LIGHTBLUE };

#define FRAMEWIDE 800                   /* width of client area */
#define FRAMEHIGH 600                  /* height of client area */
#define WIDE	FRAMEWIDE
#define HIGH	FRAMEHIGH

HDC hHiddenDC=NULL;
HBITMAP hHiddenBitMap=NULL;

void DisplayPlane( MathPlane *Plane, MathFrame *Frame, MathVideo *Video )
{
   MathPoint Point3D[ PLANEPOINTS ];
   MathPoint Point2D[ PLANEPOINTS ];
   MathPoint Vector1, Vector2, VectorX;
   int loop;
   POINT Shape[6];
   HBRUSH ColorBrush, PrevBrush;
   HPEN ColorPen, PrevPen;

   for ( loop=0; loop<PLANEPOINTS; ++loop )
   {
      MathFrameTransformation( Frame, Plane->PlanePoints[loop], &Point3D[loop] );
      Math2DTransformation( &Point3D[loop], &Point2D[loop], Video->VideoFocus );
   };
   MathVectorSubtraction( Plane->PlanePoints[3], Plane->PlanePoints[0], &Vector1 );
   MathVectorSubtraction( Plane->PlanePoints[1], Plane->PlanePoints[0], &Vector2 );

   MathVectorSubtraction( &Point2D[3], &Point2D[0], &Vector1 );
   MathVectorSubtraction( &Point2D[1], &Point2D[0], &Vector2 );

   MathVectorProduct( &Vector1, &Vector2, &VectorX );
   if ( VectorX.MathTranslation[2] > (real) 0.0 )
   {
      for ( loop=0; loop<PLANEPOINTS; ++loop )
      {
         Math2DCoordinate( &Point2D[loop], &Point2D[loop], Video );
         Shape[loop].x = (int) Point2D[loop].MathTranslation[0];
         Shape[loop].y = (int) Point2D[loop].MathTranslation[1];
      };

      ColorPen = CreatePen(PS_SOLID,1, RGB(0,0,0) );
      ColorBrush = CreateSolidBrush( Plane->PlaneColor );
      PrevPen = static_cast<HPEN>(SelectObject(hHiddenDC,ColorPen));
      PrevBrush = static_cast<HBRUSH>(SelectObject(hHiddenDC,ColorBrush));

      Polygon( hHiddenDC, Shape, 4 );

      SelectObject(hHiddenDC,PrevPen);
      SelectObject(hHiddenDC,PrevBrush);
      DeleteObject(ColorPen);
      DeleteObject(ColorBrush);
   };
}

int InitHiddenWindow( HWND hWnd )
{
   HDC hDisplayDC;

   GlobalCompact((DWORD)-1L);

   hDisplayDC = GetDC(hWnd);
   hHiddenDC = CreateCompatibleDC(hDisplayDC);
   hHiddenBitMap = CreateCompatibleBitmap(hDisplayDC,FRAMEWIDE,FRAMEHIGH);
   if ( hHiddenBitMap == (HBITMAP) NULL )
      return FALSE;
   ReleaseDC( hWnd, hDisplayDC );

   return TRUE;
}

int DeleteHiddenWindow( HWND hWnd )
{
   if ( hHiddenBitMap != (HBITMAP) NULL )
   {
      DeleteObject( hHiddenBitMap );
      hHiddenBitMap = (HBITMAP) NULL;
   };
   if ( hHiddenDC != (HDC) NULL )
   {
      ReleaseDC( hWnd, hHiddenDC );
      hHiddenDC = (HDC) NULL;
   };

   return TRUE;
}

void DisplayCube( HWND hWnd, MathFrame *Frame, MathVideo *Video )
{
   HDC hDisplayDC;

   hDisplayDC = GetDC( hWnd );

   if (hHiddenDC == NULL || hHiddenBitMap == NULL)
      InitHiddenWindow (hWnd);

   SelectObject( hHiddenDC, hHiddenBitMap );
   PatBlt(hHiddenDC,0,0,FRAMEWIDE,FRAMEHIGH,WHITENESS);

   DisplayPlane( &Pl1, Frame, Video );
   DisplayPlane( &Pl2, Frame, Video );
   DisplayPlane( &Pl3, Frame, Video );
   DisplayPlane( &Pl4, Frame, Video );
   DisplayPlane( &Pl5, Frame, Video );
   DisplayPlane( &Pl6, Frame, Video );

   BitBlt( hDisplayDC, 0,0, WIDE,HIGH, hHiddenDC, 0,0, SRCCOPY );

   ReleaseDC( hWnd, hDisplayDC );
}
