/*----------------------------------------------------------------------
*  cube3d.cpp -- cube 3d implementation file
*
*----------------------------------------------------------------------
*
* (c) 1998-2007 3Dconnexion. All rights reserved. 
* Permission to use, copy, modify, and distribute this software for all
* purposes and without fees is hereby grated provided that this copyright
* notice appears in all copies.  Permission to modify this software is granted
* and 3Dconnexion will support such modifications only is said modifications are
* approved by 3Dconnexion.
*
*/
#include "stdafx.h"
#include "cube3d.h"
#include "math3d.h"
#include <atlstr.h>



/*| Definitions |*/
#define ScaleRotation		1024.0
#define ScaleTranslation	512.0

#define FRAMEWIDE 			800 	/* width and height of the frame */
#define FRAMEHIGH 			600

#define WINDOWWIDTH  		808		/* width and height of the window */
#define WINDOWHEIGHT 		646

#define NearPosition		-2.0

/*| Prototypes |*/
HWND InitClassWindow( HINSTANCE );
int InitHiddenWindow( HWND );
int DeleteHiddenWindow( HWND );
void DisplayCube( HWND, MathFrame *, MathVideo *);

/*| Global Variables |*/
HINSTANCE hInstanceGlobal;
HACCEL hAccel;
HWND MainhWnd;
HDC hDC;
HMENU hMenu;

int WindowX, WindowY, DisplayWidth, DisplayHeight;

MathVideo VGAVideo = { 8.3/12.3, FRAMEWIDE, FRAMEHIGH, 1.5 };
MathFrame FrameCube;

#define _3DxVersion	_T(" 3Dconnexion 2007")


UINT_PTR gTimerId = 0;
VOID CALLBACK _3DxTimerProc(HWND hwnd, UINT uMsg, UINT_PTR idEvent, DWORD dwTime);

/*|------------------------------------------------------------------|*/

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
   MSG Message;
   HWND hDesktopWnd;
   HDC hDCcaps;
   HFONT hfnt, hOldFont;

   HRESULT hr=::CoInitializeEx(NULL, COINIT_APARTMENTTHREADED );
   if (!SUCCEEDED(hr))
   {
      CString strError;
      strError.FormatMessage (_T("Error 0x%x"), hr);
      ::MessageBox (NULL, strError, _T("CoInitializeEx failed"), MB_ICONERROR|MB_OK);
      return FALSE;
   }

   hDesktopWnd = GetDesktopWindow();
   hDCcaps = GetDC(hDesktopWnd);
   DisplayWidth  = GetDeviceCaps(hDCcaps,HORZRES);
   DisplayHeight = GetDeviceCaps(hDCcaps,VERTRES);
   ReleaseDC(hDesktopWnd,hDCcaps); 

   WindowX= (DisplayWidth  - WINDOWWIDTH ) /2;
   WindowY= (DisplayHeight - WINDOWHEIGHT) /2;

   hMenu = LoadMenu(hInstance, (LPCTSTR) IDD_CUBE3DMENU);

   hInstanceGlobal = hInstance;
   MainhWnd = InitClassWindow(hInstance);
   if ( MainhWnd == FALSE )
      return FALSE;
   ShowWindow(MainhWnd, nCmdShow);


   if ( !InitHiddenWindow( MainhWnd ) )
      return FALSE;

   SetCursor( LoadCursor((HINSTANCE)NULL,IDC_WAIT) );

   MathFrameTranslation( &FrameCube, 0.0, 0.0, -10.0 );
   MathFrameRotation( &FrameCube, 0.0, 0.0, 0.0 );
   DisplayCube( MainhWnd, &FrameCube, &VGAVideo );

   hDC = GetDC( MainhWnd );              
   UpdateWindow(MainhWnd);

   SetCursor( LoadCursor((HINSTANCE)NULL,IDC_ARROW) );

   hDC= GetDC(MainhWnd);

   hfnt = static_cast<HFONT>(GetStockObject(ANSI_VAR_FONT));
   hOldFont = static_cast<HFONT>(SelectObject(hDC, hfnt));

   TextOut(hDC, 325, 270, _3DxVersion, (int)_tcslen(_3DxVersion) );

   SelectObject(hDC, hOldFont);

   ReleaseDC(MainhWnd,hDC);


   while( GetMessage(&Message,0,0,0) )
   {                
      if ( TranslateAccelerator(MainhWnd, hAccel, &Message) )
         continue;

      TranslateMessage(&Message);
      DispatchMessage(&Message);
   };
   return((int)Message.wParam);
}   

INT_PTR CALLBACK MagellanAboutDialog( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
   switch( msg )
   {
   case WM_INITDIALOG : return TRUE;
   case WM_COMMAND :
      switch( wParam )
      {
      case IDOK : 
         EndDialog( hWnd, 0 );
         return TRUE;
      };
   };
   return FALSE;
}

double Sensitivity = 1.0;
CComPtr<ISensor> g3DSensor;
CComPtr<IKeyboard> g3DKeyboard;
__int64 gKeyStates=0;

LRESULT FAR PASCAL MessageHandler(HWND hWnd, unsigned message,	WPARAM wParam, LPARAM lParam)
{
   INT_PTR lpDialog;

   switch (message)
   { 
   case WM_CREATE :
      {
         HRESULT hr;
         CComPtr<IUnknown> _3DxDevice;

         // Create the device object
         hr = _3DxDevice.CoCreateInstance(__uuidof(Device));
         if (SUCCEEDED(hr))
         {
            CComPtr<ISimpleDevice> _3DxSimpleDevice;

            hr = _3DxDevice.QueryInterface(&_3DxSimpleDevice);
            if (SUCCEEDED(hr))
            {
               // Get the interfaces to the sensor and the keyboard;
               g3DSensor = _3DxSimpleDevice->Sensor;
               g3DKeyboard = _3DxSimpleDevice->Keyboard;

               // Associate a configuration with this device
               _3DxSimpleDevice->LoadPreferences(_T("Cube3DPolling"));
               // Connect to the driver
               _3DxSimpleDevice->Connect();

               // Create timer used to poll the 3dconnexion device
               gTimerId = ::SetTimer(NULL, 0, 25, _3DxTimerProc);
            }
         }
      }
      break;


   case WM_COMMAND : 
      {
         switch(wParam)
         {
         case IDM_Exit:
            PostMessage( hWnd, WM_DESTROY, 0, 0L );
            break;

         case IDM_ABOUT:	 
            lpDialog = DialogBox( hInstanceGlobal, MAKEINTRESOURCE(IDD_ABOUT),
               MainhWnd, (DLGPROC)MagellanAboutDialog ); 
            break;
         }
      }
      break;


   case WM_PAINT: 
      DisplayCube( hWnd, &FrameCube, &VGAVideo );
      break;


   case WM_SYSCOMMAND:
      {
         switch( LOWORD(wParam) )
         {
         case SC_MAXIMIZE:
            MessageBox(hWnd, _T("Maximize cube window NOT implemented!"), _T("Cube 3D Demo"), MB_ICONWARNING|MB_OK );
            return TRUE;
         }
      }
      break;

   case WM_DESTROY: 
      {
         CComPtr<ISimpleDevice> _3DxDevice;

         // Kill the timer used to poll the sensor and keyboard
         if (gTimerId)
         {
            ::KillTimer(NULL, gTimerId);
            gTimerId = NULL;
         }

         // Release the sensor and keyboard interfaces
         if (g3DSensor)
         {
            g3DSensor->get_Device((IDispatch**)&_3DxDevice);
            g3DSensor.Release();
         }

         if (g3DKeyboard)
            g3DKeyboard.Release();



         if (_3DxDevice)
         {
            // Disconnect it from the driver
            _3DxDevice->Disconnect();
            _3DxDevice.Release();
         }

         DeleteHiddenWindow( hWnd );
         PostQuitMessage( 0 );
      }
      break;
   }
   return DefWindowProc(hWnd, message, wParam, lParam);
}


VOID CALLBACK ExecuteKeyFunction(int iFunction)
{
   switch (iFunction)
   {
   case 5:
      Sensitivity /= 2.0;
      break;

   case 6:
      Sensitivity *= 2.0;
      break;

   case 7:
      Sensitivity = 1.0;
      break;

   case 4: 
   case 31: 
      // Recenter the cube
      MathFrameTranslation( &FrameCube, 0.0, 0.0, -10. );
      MathFrameRotation( &FrameCube, 0., 0., 0. );
      if (MainhWnd)
         DisplayCube( MainhWnd, &FrameCube, &VGAVideo );

      break;
   }
}

// The timer callback is used to poll the 3d input device for change of keystates and
// the cap displacement values.
VOID CALLBACK _3DxTimerProc(HWND hwnd, UINT uMsg, UINT_PTR idEvent, DWORD dwTime)
{
   static DWORD s_dwLastDraw = 0;

   if (g3DKeyboard)
   {
      // Check if any change to the keyboard state
      try {
         long nKeys;
         nKeys = g3DKeyboard->Keys;
         long i;
         for (i=1; i<=nKeys; i++)
         {
            __int64 mask = (__int64)1<<(i-1);
            VARIANT_BOOL isPressed;
            isPressed = g3DKeyboard->IsKeyDown(i);
            if (isPressed == VARIANT_TRUE)
            {
               if (!(gKeyStates & mask))
               {
                  gKeyStates |= mask;
                  ExecuteKeyFunction (i);
               }
            }
            else
            {
               gKeyStates &= ~mask;
            }
         }
         // Test the special keys
         for (i=30; i<=31; i++)
         {
            __int64 mask = (__int64)1<<(i-1);
            VARIANT_BOOL isPressed;
            isPressed = g3DKeyboard->IsKeyDown(i);
            if (isPressed == VARIANT_TRUE)
            {
               if (!(gKeyStates & mask))
               {
                  gKeyStates |= mask;
                  ExecuteKeyFunction (i);
               }
            }
            else
            {
               gKeyStates &= ~mask;
            }
         }
      }
      catch (...) {
         // Some sort of exception handling
      }
   }
   if (g3DSensor)
   {
      try {

         CComPtr<IAngleAxis> pRotation = g3DSensor->Rotation;
         CComPtr<IVector3D> pTranslation = g3DSensor->Translation;

         // Check if the cap is still displaced
         if (pRotation->Angle > 0. || pTranslation->Length > 0.)
         {

            double timeFactor = 1.;

            DWORD dwNow = ::GetTickCount();
            if (s_dwLastDraw)
               timeFactor = (double)(dwNow-s_dwLastDraw)/g3DSensor->Period;
            s_dwLastDraw = dwNow;

            pTranslation->Length /= ScaleTranslation*Sensitivity;
            pRotation->Angle /= ScaleRotation*Sensitivity;

            MathFrame FrameTransRot;
            MathFrameTranslation(&FrameTransRot, pTranslation->X, pTranslation->Y, pTranslation->Z);


            MathFrameRotation( &FrameTransRot, 
               pRotation->X * pRotation->Angle, 
               pRotation->Y * pRotation->Angle,
               pRotation->Z * pRotation->Angle );

            MathFrameMultiplication( &FrameTransRot, &FrameCube, &FrameCube );
            if ( FrameCube.MathTranslation[2] > NearPosition )
               FrameCube.MathTranslation[2] = NearPosition;

            if (MainhWnd)
               DisplayCube( MainhWnd, &FrameCube, &VGAVideo );
         }
         else
            s_dwLastDraw = 0;

         pRotation.Release();
         pTranslation.Release();
      }
      catch (...)
      {
         // Some sort of exception handling
      }
   }
}

HWND InitClassWindow(HINSTANCE hInstance)
{
   WNDCLASS WndClass;
   HWND hWnd;

   memset( &WndClass, 0x00, sizeof( WndClass ) );
   WndClass.lpfnWndProc = (WNDPROC) MessageHandler;
   WndClass.hInstance = hInstance;
   WndClass.hIcon = LoadIcon( hInstance, (LPCTSTR)IDI_CUBE3D );
   WndClass.hCursor = LoadCursor( NULL,IDC_ARROW);
   WndClass.hbrBackground = CreateSolidBrush( RGB(255,255,255) );
   WndClass.lpszMenuName = NULL;
   WndClass.lpszClassName = _T("MAGELLANCUBE3DCLASS");
   WndClass.style = CS_HREDRAW | CS_VREDRAW;
   WndClass.cbClsExtra = 0;
   WndClass.cbWndExtra =0 ;
   if ( RegisterClass( &WndClass ) == FALSE )
      return (HWND) NULL;

   hWnd = CreateWindow(_T("MAGELLANCUBE3DCLASS"), _T("Cube 3D Polling Demo"),
      WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
      WindowX,WindowY, WINDOWWIDTH-7,WINDOWHEIGHT-7,
      0,hMenu, hInstance,(LPSTR)NULL );

   return hWnd;
}

