// CubeWindow.cpp : Implementation of CCubeWindow
/*
* (c) 1998-2006 3Dconnexion. All rights reserved. 
* Permission to use, copy, modify, and distribute this software for all
* purposes and without fees is hereby grated provided that this copyright
* notice appears in all copies.  Permission to modify this software is granted
* and 3Dconnexion will support such modifications only is said modifications are
* approved by 3Dconnexion.
*
*/
#include "stdafx.h"
///////////////////////////////////////////////////////////////////////////////////
// History
//
// $Id: CubeWindow.cpp 3304 2007-08-08 08:28:04Z markus_bonk $
//
// 22.05.07 MSB Changed the order LoadPreferences() and the hooks are executed so that
//              the VS71 compiler doesn't crash.
// 17.04.07 MSB Added call to LoadPreferences()
//              Corrected CCubeWindow::UninitializeCOM() which was creating another instance of the
//              Device component
// 05.12.06 MSB Initial Design
//
#include "CubeWindow.h"
#include "About.h"
#include <atlstr.h>
#include "math3d.h"
#define 	BROWN				RGB(0,255,255)
#define 	YELLOW 			RGB(255,255,0)
#define  GREEN				RGB(0,191,0)
#define  LIGHTRED			RGB(255,0,0)
#define	LIGHTMAGENTA   RGB(191,0,191)
#define	LIGHTBLUE		RGB(0,0,191)

#define CubeSize		1.0

MathPoint P1 = { { +CubeSize, -CubeSize, +CubeSize } };
MathPoint P2 = { { +CubeSize, -CubeSize, -CubeSize } };
MathPoint P3 = { { +CubeSize, +CubeSize, -CubeSize } };
MathPoint P4 = { { +CubeSize, +CubeSize, +CubeSize } };
MathPoint P5 = { { -CubeSize, -CubeSize, +CubeSize } };
MathPoint P6 = { { -CubeSize, -CubeSize, -CubeSize } };
MathPoint P7 = { { -CubeSize, +CubeSize, -CubeSize } };
MathPoint P8 = { { -CubeSize, +CubeSize, +CubeSize } };

MathPlane Pl1 = { { &P4, &P3, &P2, &P1 }, BROWN };
MathPlane Pl2 = { { &P8, &P4, &P1, &P5 }, YELLOW };
MathPlane Pl3 = { { &P8, &P5, &P6, &P7 }, GREEN };
MathPlane Pl4 = { { &P7, &P6, &P2, &P3 }, LIGHTRED };
MathPlane Pl5 = { { &P8, &P7, &P3, &P4 }, LIGHTMAGENTA };
MathPlane Pl6 = { { &P6, &P5, &P1, &P2 }, LIGHTBLUE };

#define FRAMEWIDE 800                   /* width of client area */
#define FRAMEHIGH 600                  /* height of client area */
#define WIDE	FRAMEWIDE
#define HIGH	FRAMEHIGH

HDC hHiddenDC=NULL;
HBITMAP hHiddenBitMap=NULL;
MathVideo VGAVideo = { 8.3/12.3, FRAMEWIDE, FRAMEHIGH, 1.5 };
MathFrame FrameCube;

void DisplayPlane( MathPlane *Plane, MathFrame *Frame, MathVideo *Video )
{
   MathPoint Point3D[ PLANEPOINTS ];
   MathPoint Point2D[ PLANEPOINTS ];
   MathPoint Vector1, Vector2, VectorX;
   int loop;
   POINT Shape[6];
   HBRUSH ColorBrush, PrevBrush;
   HPEN ColorPen, PrevPen;

   for ( loop=0; loop<PLANEPOINTS; ++loop )
   {
      MathFrameTransformation( Frame, Plane->PlanePoints[loop], &Point3D[loop] );
      Math2DTransformation( &Point3D[loop], &Point2D[loop], Video->VideoFocus );
   };
   MathVectorSubtraction( Plane->PlanePoints[3], Plane->PlanePoints[0], &Vector1 );
   MathVectorSubtraction( Plane->PlanePoints[1], Plane->PlanePoints[0], &Vector2 );

   MathVectorSubtraction( &Point2D[3], &Point2D[0], &Vector1 );
   MathVectorSubtraction( &Point2D[1], &Point2D[0], &Vector2 );

   MathVectorProduct( &Vector1, &Vector2, &VectorX );
   if ( VectorX.MathTranslation[2] > (real) 0.0 )
   {
      for ( loop=0; loop<PLANEPOINTS; ++loop )
      {
         Math2DCoordinate( &Point2D[loop], &Point2D[loop], Video );
         Shape[loop].x = (int) Point2D[loop].MathTranslation[0];
         Shape[loop].y = (int) Point2D[loop].MathTranslation[1];
      };

      ColorPen = CreatePen(PS_SOLID,1, RGB(0,0,0) );
      ColorBrush = CreateSolidBrush( Plane->PlaneColor );
      PrevPen = static_cast<HPEN>(SelectObject(hHiddenDC,ColorPen));
      PrevBrush = static_cast<HBRUSH>(SelectObject(hHiddenDC,ColorBrush));

      Polygon( hHiddenDC, Shape, 4 );

      SelectObject(hHiddenDC,PrevPen);
      SelectObject(hHiddenDC,PrevBrush);
      DeleteObject(ColorPen);
      DeleteObject(ColorBrush);
   };
}

int InitHiddenWindow( HWND hWnd )
{
   HDC hDisplayDC;

   GlobalCompact((DWORD)-1L);

   hDisplayDC = GetDC(hWnd);
   hHiddenDC = CreateCompatibleDC(hDisplayDC);

   RECT rect;
   GetClientRect(hWnd, &rect);

   hHiddenBitMap = CreateCompatibleBitmap(hDisplayDC,FRAMEWIDE, FRAMEHIGH);
   if ( hHiddenBitMap == (HBITMAP) NULL )
      return FALSE;
   ReleaseDC( hWnd, hDisplayDC );

   return TRUE;
}

int DeleteHiddenWindow( HWND hWnd )
{
   if ( hHiddenBitMap != (HBITMAP) NULL )
   {
      DeleteObject( hHiddenBitMap );
      hHiddenBitMap = (HBITMAP) NULL;
   };
   if ( hHiddenDC != (HDC) NULL )
   {
      ReleaseDC( hWnd, hHiddenDC );
      hHiddenDC = (HDC) NULL;
   };

   return TRUE;
}

void DisplayCube( HWND hWnd, MathFrame *Frame, MathVideo *Video )
{
   HDC hDisplayDC;

   hDisplayDC = GetDC( hWnd );

   if (hHiddenDC == NULL || hHiddenBitMap == NULL)
      InitHiddenWindow (hWnd);

   SelectObject( hHiddenDC, hHiddenBitMap );
   PatBlt(hHiddenDC,0,0,FRAMEWIDE,FRAMEHIGH,WHITENESS);

   DisplayPlane( &Pl1, Frame, Video );
   DisplayPlane( &Pl2, Frame, Video );
   DisplayPlane( &Pl3, Frame, Video );
   DisplayPlane( &Pl4, Frame, Video );
   DisplayPlane( &Pl5, Frame, Video );
   DisplayPlane( &Pl6, Frame, Video );

   BitBlt( hDisplayDC, 0,0, WIDE,HIGH, hHiddenDC, 0,0, SRCCOPY );

   ReleaseDC( hWnd, hDisplayDC );
}

///////////////////////////////////////////////////////////////////////////////////
// CCubeWindow
LRESULT CCubeWindow::OnAbout(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
   LRESULT result=0;

   CAbout *pAbout = new CAbout();
   pAbout->DoModal();
   delete pAbout;
   pAbout = NULL;

   return result;
}

#define WINDOWWIDTH 808
#define WINDOWHEIGHT 646
#define _3DxVersion	_T(" 3Dconnexion 2007")

LRESULT CCubeWindow::OnCreate(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
   LRESULT result=0;
   bHandled = FALSE;

   HWND hDesktopWnd = ::GetDesktopWindow();
   HDC hcaps = ::GetDC(hDesktopWnd);
   int DisplayWidth  = GetDeviceCaps(hcaps,HORZRES);
   int DisplayHeight = GetDeviceCaps(hcaps,VERTRES);
   ::ReleaseDC(hDesktopWnd, hcaps); 
   hcaps = NULL;

   int xpos= (DisplayWidth  - WINDOWWIDTH ) /2;
   int ypos= (DisplayHeight - WINDOWHEIGHT) /2;

   SetWindowPos (NULL, xpos, ypos, WINDOWWIDTH, WINDOWHEIGHT, SWP_NOZORDER);

   HMENU hmenu = GetMenu();
   if (hmenu)
      ::DestroyMenu (hmenu);
   m_hmenu = ::LoadMenu (_AtlBaseModule.GetResourceInstance(), MAKEINTRESOURCE(IDR_MAINMENU));
   SetMenu (m_hmenu);

   CString strText;
   strText.LoadString(IDS_TITLE);
   SetWindowText (strText);


   SetCursor( LoadCursor((HINSTANCE)NULL,IDC_WAIT) );

   InitHiddenWindow((HWND)*this);
   MathFrameTranslation( &FrameCube, 0.0, 0.0, -10.0 );
   MathFrameRotation( &FrameCube, 0.0, 0.0, 0.0 );
   DisplayCube( (HWND)*this, &FrameCube, &VGAVideo );
   UpdateWindow();

   HDC hdc = GetDC();              
   HFONT hfnt = static_cast<HFONT>(GetStockObject(ANSI_VAR_FONT));
   HFONT hOldFont = static_cast<HFONT>(SelectObject(hdc, hfnt));
   TextOut(hdc, 325, 270, _3DxVersion, (int)_tcslen(_3DxVersion) );
   SelectObject(hdc, hOldFont);
   ReleaseDC(hdc);
   hdc = NULL;

   SetCursor( LoadCursor((HINSTANCE)NULL,IDC_ARROW) );


   HRESULT hr=::CoInitializeEx(NULL, COINIT_APARTMENTTHREADED );
   if (!SUCCEEDED(hr))
   {
      CString strError;
      strError.FormatMessage(_T("Error 0x%x"), hr);
      ::MessageBox (NULL, strError, _T("CoInitializeEx failed"), MB_ICONERROR|MB_OK);

      bHandled = TRUE;

      // stop creation of window
      result = -1;
   }
   else
   {
      HRESULT hr = InitializeCOM();
      if (!SUCCEEDED(hr))
      {
         bHandled = TRUE;

         // stop creation of window
         result = -1;
      }
   }


   return result;
}

LRESULT CCubeWindow::OnExit(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
   LRESULT result = 0;
   DestroyWindow();
   return result;
}

LRESULT CCubeWindow::OnNcDestroy(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
   LRESULT result=0;
   bHandled = FALSE;

   UninitializeCOM();

   if (m_hmenu)
      DestroyMenu(m_hmenu);

   m_hmenu = NULL;

   PostQuitMessage(0);

   return result;
}

LRESULT CCubeWindow::OnPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
   LRESULT result=0;
   bHandled = FALSE;

   DisplayCube((HWND)*this, &FrameCube, &VGAVideo);

   return result;
}

HRESULT CCubeWindow::InitializeCOM()
{
   HRESULT hr;

   CComPtr<IUnknown> _3DxDevice;

   // Create the device object
   hr = _3DxDevice.CoCreateInstance(__uuidof(Device));

   if (SUCCEEDED(hr))
   {

      CComPtr<ISimpleDevice> _3DxSimpleDevice;

      hr = _3DxDevice.QueryInterface(&_3DxSimpleDevice);
      if (SUCCEEDED(hr))
      {

         // Set the preferences we want to use
         _3DxSimpleDevice->LoadPreferences(_T("AtlCude3D"));

         hr = __hook(&_ISimpleDeviceEvents::DeviceChange, 
                        _3DxSimpleDevice, 
                        &CCubeWindow::OnDeviceChange);

         // Get the interfaces to the sensor and the keyboard;
         hr = _3DxSimpleDevice->get_Sensor(&m_pISensor);

         hr = __hook(&_ISensorEvents::SensorInput, m_pISensor, 
                        &CCubeWindow::OnSensorInput);

         hr = _3DxSimpleDevice->get_Keyboard(&m_pIKeyboard);

         hr = __hook(&_IKeyboardEvents::KeyDown, m_pIKeyboard, 
                        &CCubeWindow::OnKeyDown);

         hr = __hook(&_IKeyboardEvents::KeyUp, m_pIKeyboard, 
                        &CCubeWindow::OnKeyUp);

         // Connect to the driver
         _3DxSimpleDevice->Connect();
      }
      _3DxDevice.Release();
   }
   else
      ::MessageBox (NULL, 
                     _T("Could not create Device"),
                     _T("CoCreateInstance failed"), 
                     MB_ICONERROR|MB_OK);


   return hr;
}

HRESULT CCubeWindow::OnDeviceChange(long reserved )
{
   HRESULT result = S_OK;
   AtlTrace (_T("CCubeWindow::OnDeviceChange(reserved =0x%x)\n"), reserved);

   HRESULT hr;
   ISimpleDevicePtr _p3DxDevice;
   hr = m_pIKeyboard->get_Device((IDispatch**)&_p3DxDevice);
   if (SUCCEEDED(hr))
   {
      long device;
      _p3DxDevice->get_Type(&device);
      AtlTrace (_T("Attached device=%d\n"), device);
      _p3DxDevice.Release();

      long keys, programmableKeys;
      m_pIKeyboard->get_Keys(&keys);
      m_pIKeyboard->get_ProgrammableKeys(&programmableKeys);
      AtlTrace (_T("Number of Keys=%d\tProgrammable keys=%d\n"), keys, programmableKeys);
      for (long i=1; i<=keys; i++)
      {
         BSTR bstrLabel;
         BSTR bstrName;
         m_pIKeyboard->GetKeyLabel(i, &bstrLabel);
         m_pIKeyboard->GetKeyName(i, &bstrName);
         CString strLabel(bstrLabel);
         CString strName(bstrName);

         AtlTrace (_T("Key Label=%s\tKey Name=%s\n"), (const TCHAR *)strLabel, (const TCHAR *)strName);
      }
   }
   return result;
}

HRESULT CCubeWindow::OnKeyDown(int keyCode )
{
   HRESULT result = S_OK;
   AtlTrace (_T("CCubeWindow::OnKeyDown(keyCode =%d)\n"), keyCode);
   switch (keyCode)
   {
   case 31: // Fit
      MathFrameTranslation( &FrameCube, 0.0, 0.0, -10. );
      MathFrameRotation( &FrameCube, 0., 0., 0. );
      DisplayCube( (HWND)*this, &FrameCube, &VGAVideo );
      break;
   }
   return result;
}

HRESULT CCubeWindow::OnKeyUp(int keyCode )
{
   HRESULT result = S_OK;
   AtlTrace (_T("CCubeWindow::OnKeyUp(keyCode =%d)\n"), keyCode);
   return result;
}

#define ScaleRotation		1024.0
#define ScaleTranslation	512.0
#define NearPosition		   -2.0
#define Sensitivity        1.0

HRESULT CCubeWindow::OnSensorInput(void)
{
   HRESULT result = S_OK;
   static DWORD s_dwLastDraw = 0;

   AtlTrace (_T("CCubeWindow::OnSensorInput()\n"));
   try {
      CComPtr<IAngleAxis> pRotation;
      HRESULT hr = m_pISensor->get_Rotation(&pRotation);

      double angle;
      pRotation->get_Angle(&angle);


      CComPtr<IVector3D> pTranslation;
      hr = m_pISensor->get_Translation(&pTranslation);
      
 
      double length;
      pTranslation->get_Length(&length);

      if (angle > 0. || length > 0.)
      {
         MathFrame FrameTransRot;
         double x, y, z;
         double timeFactor = 1.;

         DWORD dwNow = ::GetTickCount();
         if (s_dwLastDraw)
         {
            double period;
            m_pISensor->get_Period(&period);
            timeFactor = (double)(dwNow-s_dwLastDraw)/period;
         }
         s_dwLastDraw = dwNow;



         length *= timeFactor;
         length /= ScaleTranslation*Sensitivity;
         pTranslation->put_Length(length);
         pTranslation->get_X(&x);
         pTranslation->get_Y(&y);
         pTranslation->get_Z(&z);
         MathFrameTranslation(&FrameTransRot, x, y, z);


         pRotation->get_X(&x);
         pRotation->get_Y(&y);
         pRotation->get_Z(&z);

         angle *= timeFactor;
         angle /= ScaleRotation*Sensitivity;
         MathFrameRotation( &FrameTransRot, 
                     x * angle, 
                     y * angle,
                     z * angle );

         MathFrameMultiplication( &FrameTransRot, &FrameCube, &FrameCube );
         if ( FrameCube.MathTranslation[2] > NearPosition )
            FrameCube.MathTranslation[2] = NearPosition;

         DisplayCube( (HWND)*this, &FrameCube, &VGAVideo );
      }
      else
         s_dwLastDraw = 0;

      pRotation.Release();
      pTranslation.Release();
   }
   catch (...)
   {
      // Some sort of exception handling
   }

   return result;
}

HRESULT CCubeWindow::UninitializeCOM()
{
   HRESULT hr = E_FAIL;

   CComPtr<IDispatch> _3DxDevice;
   if (m_pISensor)
      hr = m_pISensor->get_Device(&_3DxDevice);

   else if (m_pIKeyboard)
      hr = m_pIKeyboard->get_Device(&_3DxDevice);

   if (SUCCEEDED(hr))
   {
      CComPtr<ISimpleDevice> _3DxSimpleDevice;
      hr = _3DxDevice.QueryInterface(&_3DxSimpleDevice);
      if (SUCCEEDED(hr))
         _3DxSimpleDevice->Disconnect();
   }

   if (m_pISensor)
   {
      // unhook (unadvise) the sensor event sink
      __unhook(&_ISensorEvents::SensorInput, m_pISensor, 
                     &CCubeWindow::OnSensorInput);

      m_pISensor.Release();
   }

   if (m_pIKeyboard)
   {
      __unhook(&_IKeyboardEvents::KeyDown, m_pIKeyboard, 
                     &CCubeWindow::OnKeyDown);

      m_pIKeyboard.Release();
   }

   return hr;
}
