// 
// Notice Regarding Standards.  AMD does not provide a license or sublicense to
// any Intellectual Property Rights relating to any standards, including but not
// limited to any audio and/or video codec technologies such as MPEG-2, MPEG-4;
// AVC/H.264; HEVC/H.265; AAC decode/FFMPEG; AAC encode/FFMPEG; VC-1; and MP3
// (collectively, the "Media Technologies"). For clarity, you will pay any
// royalties due for such third party technologies, which may include the Media
// Technologies that are owed as a result of AMD providing the Software to you.
// 
// MIT license 
// 
// Copyright (c) 2016 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//

#include <stdlib.h>
#include <string.h>
#include <iostream>

#include "Settings.h"

#ifndef _WIN32
#define _stricmp strcasecmp
#define _strnicmp strncasecmp
#define strcpy_s strcpy
#endif



Settings::Settings()
{
    // initialize default value
#ifdef _WIN32
    m_ApiExt = API_EXT_DX; 
#else
    m_ApiExt = API_EXT_OPENGL; 
#endif

    m_Width = 3840;
    m_Height = 2160;

    m_FrameRate = 9999;
    m_frameOffset = 0;

    m_Format = VIDEO_FORMAT::FORMAT_YUV420_8;

    m_NumBuffers = 2;
    m_NumThreads = 2;
    m_Loop = 1;

    m_Sync = true;
    m_VSync = false;

    m_NumFileStreams = 0;
}

bool Settings::Parse(int argc, char** argv)
{
    bool valid = true;

    for (int i = 1; i < argc; i++)
    {
        char * arg = argv[i];

        if (arg[0] == '-')
        {
            // option
            char * optionValue = 0;
            char * optionName = arg + 1;
            size_t optionNameLen = 0;

            // option name and value
            char * equal = strchr(arg, '=');
            if (equal != 0) {
                optionValue = equal + 1;
                optionNameLen = equal - arg - 1;
            }
            else
                optionNameLen = strlen(arg);

            // option name
            if (_strnicmp(optionName, "w", optionNameLen) == 0 ||
                _strnicmp(optionName, "width", optionNameLen) == 0)
            {
                // width
                m_Width = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "h", optionNameLen) == 0 ||
                _strnicmp(optionName, "height", optionNameLen) == 0)
            {
                // width
                m_Height = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "4k", optionNameLen) == 0)
            {
                // 4k resolution
                m_Width = 3840;
                m_Height = 2160;
            }
            else if (_strnicmp(optionName, "8k", optionNameLen) == 0)
            {
                // 8k resolution
                m_Width = 7680;
                m_Height = 4320;
            }
            else if (_strnicmp(optionName, "f", optionNameLen) == 0 ||
                _strnicmp(optionName, "framerate", optionNameLen) == 0)
            {
                m_FrameRate = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "d", optionNameLen) == 0 ||
                _strnicmp(optionName, "dataformat", optionNameLen) == 0)
            {
                // format
				if (_stricmp(optionValue, "rgb10") == 0)
                    m_Format = FORMAT_RGB101010;
                else if (_stricmp(optionValue, "yuv420_8") == 0)
                    m_Format = FORMAT_YUV420_8;
            }
            else if (_strnicmp(optionName, "foff", optionNameLen) == 0)
            {
                // frame offset
                m_frameOffset = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "b", optionNameLen) == 0 ||
                _strnicmp(optionName, "buffers", optionNameLen) == 0)
            {
                // buffers
                m_NumBuffers = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "t", optionNameLen) == 0 ||
                _strnicmp(optionName, "threads", optionNameLen) == 0)
            {
                // threads
                m_NumThreads = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "l", optionNameLen) == 0 ||
                _strnicmp(optionName, "loop", optionNameLen) == 0)
            {
                // loop
                m_Loop = atoi(optionValue);
            }
            else if (_strnicmp(optionName, "n", optionNameLen) == 0 ||
                _strnicmp(optionName, "nosync", optionNameLen) == 0)
            {
                // sync
                m_Sync = false;
            }
            else if (_strnicmp(optionName, "v", optionNameLen) == 0 ||
                _strnicmp(optionName, "vsync", optionNameLen) == 0)
            {
                // sync
                m_VSync = true;
            }
            else if (_strnicmp(optionName, "api", optionNameLen) == 0)
            {
                // platform
                if (_stricmp(optionValue, "ogl") == 0)
                {
                    m_ApiExt = API_EXT_OPENGL;
                }
                else if (_stricmp(optionValue, "ocl") == 0)
                {
                    m_ApiExt = API_EXT_OPENCL;
                }
#ifdef _WIN32
                else if (_stricmp(optionValue, "dx") == 0)
                {
                    m_ApiExt = API_EXT_DX;
                }
#endif
                else
                    valid = false;
            }
        }
        else
        {
            // file
            strcpy_s(m_Files[m_NumFileStreams], arg);
            m_NumFileStreams++;
        }
    }

    // validate
    if (m_NumFileStreams <= 0 ||
        m_NumBuffers < 1 || 
        m_NumThreads < 0 ||
        m_Loop <= 0)
        valid = false;

    if (!valid)
    {
        std::cout << "Uncompressed video player with support for direct I/O." << std::endl;
        std::cout << "Program usage: RadeonProSsgPlayer.exe <options> filename1" << std::endl;
        std::cout << std::endl;
        std::cout << "Options are specified in the format -name=value. Available options:" << std::endl;
        std::cout << " -w, -width: the width of the video in pixels" << std::endl;
        std::cout << " -h, -height: the height of the video in pixels" << std::endl;
        std::cout << " -4k: 4k resolution (3840 x 2160)" << std::endl;
        std::cout << " -8k: 8k resolution (7680 x 4320)" << std::endl;
        std::cout << " -f, -framerate: the frame rate of the video (default is 9999)" << std::endl;
        std::cout << " -d, -dataformat: the format of the video:" << std::endl;
        std::cout << "  '-d=rgb10' is 10-bit RGB (R10G10B10A2 from DPX only - big endian)" << std::endl;
        std::cout << "  '-d=yuv420_8' is 8-bit YUV 4:2:0 (default)" << std::endl;
        std::cout << " -b, -buffers: number of buffers for single/double/triple/n buffering (default is 2, min is 1)" << std::endl;
        std::cout << " -t, -slices: number of parallel requests to split each frame into (default is 2)" << std::endl;
        std::cout << " -l, -loop: number of times to repeat (default is 1)" << std::endl;
        std::cout << " -n, -nosync: run without resource synchronization" << std::endl;
        std::cout << " -v, -vsync: sync to VSYNC (default is off)" << std::endl;
        std::cout << " -api: determine graphics API:" << std::endl;
#ifdef _WIN32
        std::cout << "  '-api=dx' use DirectX (default on Windows)" << std::endl;
#endif
        std::cout << "  '-api=ogl' use OpenGL (default on Linux)" << std::endl;
        std::cout << "  '-api=ocl' use OpenCL" << std::endl;

        return false;
    }

    return true;
}


int64_t  Settings::FrameSize() const
{
    switch (Format())
    {
    case VIDEO_FORMAT::FORMAT_YUV420_8:
        return Width() * Height() * 3 / 2;

    case VIDEO_FORMAT::FORMAT_RGB101010:
        return Width() * Height() * 4;

    default:
        return Width() * Height() * 3;
    }

    return 0;
}
