// DXApplication.cpp

#include <iostream>
#include <vector>

#include <float.h>
#include <math.h>
#include <windowsx.h>
#include <D2d1.h>
#include <comutil.h>

#include "DXApplication.h"
#include "Timer.h"
#include "resource.h"


struct PS_CONSTANT_BUFFER
{
    int frameWidth;
    int frameHeight;
    int a;
    int b;
};



DXApplication::DXApplication()
    : m_hWnd(NULL)
{
    m_Paused = false;
#ifdef _DEBUG
    m_displayText = true;
#else
    m_displayText = false;
#endif // DEBUG
    m_IsFirstFrame = true;

    // extensions
    m_DxExt = NULL;
    m_DxSsgExt = NULL;

    m_sectorSize = 0;
    m_SsgFileSize.QuadPart = 0;
    m_hDxSsgFileHandle = NULL;

    m_readBuffer = 0;
    m_waitBuffer = 0;
}

DXApplication::~DXApplication()
{
    Term();
}


//
//
// specific Windows functionality
//
//

LRESULT DXApplication::WndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
    DXApplication *pApp = (DXApplication *)GetWindowLongPtr(hWnd, GWLP_USERDATA);

    switch (msg)
    {
    case WM_CREATE:
        SetWindowLongPtr(
            hWnd,
            GWLP_USERDATA,
            (LONG_PTR)(((LPCREATESTRUCT)lParam)->lpCreateParams));
        break;
    case WM_DESTROY:
    case WM_CLOSE:
        PostQuitMessage(0);
        break;
    case WM_KEYDOWN:
        pApp->OnKey((int)wParam);
        break;
    case WM_LBUTTONDOWN:
        pApp->OnMouseDown(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
        break;
    case WM_MOUSEMOVE:
        pApp->OnMouseMove(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
        break;
    case WM_LBUTTONUP:
        pApp->OnMouseUp(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
        break;
    }
    
    return ::DefWindowProc(hWnd, msg, wParam, lParam);
}

void DXApplication::TranslateCoordinates(int & x, int & y)
{
    RECT rect;
    GetClientRect(m_hWnd, &rect);

    x = x * m_Settings->Width() / rect.right;
    y = y * m_Settings->Height() / rect.bottom;
}

void DXApplication::OnKey(int keyCode)
{
    int translatedCode;

    switch (keyCode)
    {
    case VK_ESCAPE:
        translatedCode = KEY_ESCAPE;
        break;
    case VK_SPACE:
        translatedCode = KEY_SPACE;
        break;
    case 'F':
        translatedCode = KEY_FULLSCREEN;
        break;
    case 'V':
        translatedCode = KEY_FPSCONTROL;
        break;
    case 'T':
    case 't':
        m_displayText = !m_displayText;
        return;

    default:
        return;
    }

    m_Application->HandleKey(translatedCode);
}

void DXApplication::OnMouseDown(int x, int y)
{
    // translate to video coordinates
    TranslateCoordinates(x, y);

    m_Application->HandleMouseDown(x, y);
}

void DXApplication::OnMouseMove(int x, int y)
{
    // translate to video coordinates
    TranslateCoordinates(x, y);

    m_Application->HandleMouseMove(x, y);
}

void DXApplication::OnMouseUp(int x, int y)
{
    // translate to video coordinates
    TranslateCoordinates(x, y);

    m_Application->HandleMouseUp(x, y);
}


//
//
// Initialize
//
//

bool DXApplication::LoadShader(UINT id, char** ppSource, LPDWORD pSize)
{
    // check passed in paramters
    if (!ppSource || !pSize)
        return false;


    HMODULE hModule = GetModuleHandle(NULL);
    if (hModule == NULL)
        return false;

    HRSRC hResource = FindResource(hModule, MAKEINTRESOURCE(id), _T("SHADER"));
    if (hResource == NULL)
        return false;

    *pSize = SizeofResource(hModule, hResource);

    HGLOBAL hGlobal = LoadResource(hModule, hResource);
    if (hGlobal == NULL)
        return false;

    *ppSource = (char *)LockResource(hGlobal);

    return true;
}

bool DXApplication::InitWindow(int width, int height)
{
    WNDCLASSEX wc;
    memset(&wc, 0, sizeof(WNDCLASSEX));

    LPCTSTR className = _T("SsgPlayer");
    HINSTANCE hInstance = GetModuleHandle(NULL);

    // Setup the windows class with default settings.
    wc.style = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc = WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = LoadIcon(NULL, IDI_WINLOGO);
    wc.hIconSm = wc.hIcon;
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH)GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = className;
    wc.cbSize = sizeof(WNDCLASSEX);

    // Register the window class.
    RegisterClassEx(&wc);

    // Create the window with the screen settings and get the handle to it.
    m_hWnd = CreateWindowEx(WS_EX_APPWINDOW, className, _T("SsgPlayer"),
        WS_OVERLAPPEDWINDOW,
        0, 0, width * 3 / 3, height * 3 / 3, NULL, NULL, hInstance, this);
    if (m_hWnd == NULL)
        return false;

    // Bring the window up on the screen and set it as main focus.
    ShowWindow(m_hWnd, SW_SHOW);
    SetForegroundWindow(m_hWnd);
    SetFocus(m_hWnd);

    return true;
}

HRESULT DXApplication::InitExtensions()
{
    HMODULE hDrvDll = GetModuleHandleW(L"atidxx64.dll");
    if (hDrvDll == NULL)
    {
        hDrvDll = GetModuleHandleW(L"atidxx32.dll");
        if (hDrvDll == NULL)
            return E_FAIL;
    }

    PFNAmdDxExtCreate11 pAmdDxExtCreate = reinterpret_cast<PFNAmdDxExtCreate11>(GetProcAddress(hDrvDll, "AmdDxExtCreate11"));
    if (pAmdDxExtCreate == NULL)
        return E_FAIL;

    HRESULT hr = pAmdDxExtCreate(m_Device, &m_DxExt);
    if (FAILED(hr))
        return hr;

    if (m_DxExt == NULL)
        return E_FAIL;

    return S_OK;
}

HRESULT DXApplication::InitDXXSsg()
{
    if (m_DxExt == NULL)
        return E_FAIL;

    m_DxSsgExt = static_cast<IAmdDxExtSSG*>((m_DxExt)->GetExtInterface(AmdDxExtSsgID));
    if (m_DxSsgExt == NULL)
    {
        std::cout << "DirectX: Unable to get Ssg extension!" << std::endl;
        return E_FAIL;
    }

    return S_OK;
}

bool  DXApplication::InitFontWrapper()
{
    CComPtr<IFW1Factory> spFW1Factory;
    HRESULT hRes = FW1CreateFactory(FW1_VERSION, &spFW1Factory);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create font factory!" << std::endl;
        return false;
    }

    hRes = spFW1Factory->CreateFontWrapper(m_Device, L"Arial", &m_FontWrapper);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create font wrapper!" << std::endl;
        return false;
    }

    return true;
}

bool  DXApplication::GetMonitorRefreshRate(DXGI_RATIONAL& refreshRate)
{
    // obtain information
    const int screenWidth  = ::GetSystemMetrics(SM_CXSCREEN);
    const int screenHeight = ::GetSystemMetrics(SM_CYSCREEN);

    // Create a DirectX graphics interface factory.
    CComPtr<IDXGIFactory> factory;
    HRESULT hRes = CreateDXGIFactory(__uuidof(IDXGIFactory), (void**)&factory);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create DXGI factory!" << std::endl;
        return false;
    }

    // Use the factory to create an adapter for the primary graphics interface (video card).
    CComPtr<IDXGIAdapter> adapter;
    hRes = factory->EnumAdapters(0, &adapter);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to enumerate adapters!" << std::endl;
        return false;
    }

    // Enumerate the primary adapter output (monitor).
    CComPtr<IDXGIOutput> adapterOutput;
    hRes = adapter->EnumOutputs(0, &adapterOutput);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to enumerate outputs!" << std::endl;
        return false;
    }

    // Get the number of modes that fit the DXGI_FORMAT_R8G8B8A8_UNORM display format for the adapter output (monitor).
    UINT numModes = 0;
    hRes = adapterOutput->GetDisplayModeList(DXGI_FORMAT_R8G8B8A8_UNORM, DXGI_ENUM_MODES_INTERLACED, &numModes, NULL);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to get number of display modes!" << std::endl;
        return false;
    }

    // Create a list to hold all the possible display modes for this monitor/video card combination.
    std::vector<DXGI_MODE_DESC>  modeDesc(numModes);
    if (!modeDesc.data() || (modeDesc.size() != numModes))
    {
        std::cout << "DirectX: Unable to allocate memory for display modes!" << std::endl;
        return false;
    }

    // Now fill the display mode list structures.
    hRes = adapterOutput->GetDisplayModeList(DXGI_FORMAT_R8G8B8A8_UNORM, DXGI_ENUM_MODES_INTERLACED, &numModes, modeDesc.data());
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to get display modes!" << std::endl;
        return false;
    }

    // Now go through all the display modes and find the one that matches the screen width and height.
    // When a match is found store the numerator and denominator of the refresh rate for that monitor.
    for (UINT i = 0; i < numModes; i++)
    {
        if ((modeDesc[i].Width == (unsigned int)screenWidth) && 
            (modeDesc[i].Height == (unsigned int)screenHeight))
        {
            refreshRate = modeDesc[i].RefreshRate;
            break;
        }
    }

    return true;
}

bool  DXApplication::InitSwapChain(int width, int height)
{
    DXGI_SWAP_CHAIN_DESC swapChainDesc;
    memset(&swapChainDesc, 0, sizeof(DXGI_SWAP_CHAIN_DESC));

    if (!GetMonitorRefreshRate(swapChainDesc.BufferDesc.RefreshRate))
    {
        // it's possible we're connected remotely and no monitor is connected
        // so in that case, set a default refresh rate of 60Hz
        std::cout << "DirectX: Unable to get monitor refresh rate!" << std::endl;
        std::cout << "DirectX: Setting a default refresh rate of 60Hz!" << std::endl;
        swapChainDesc.BufferDesc.RefreshRate.Numerator = 60;
        swapChainDesc.BufferDesc.RefreshRate.Denominator = 1;
    }

    swapChainDesc.BufferDesc.Width = width;
    swapChainDesc.BufferDesc.Height = height;
    swapChainDesc.BufferDesc.Format = (m_Settings->Format() == VIDEO_FORMAT::FORMAT_RGB101010) ? DXGI_FORMAT_R10G10B10A2_UNORM
                                                                                               : DXGI_FORMAT_R8G8B8A8_UNORM;
    swapChainDesc.BufferDesc.ScanlineOrdering = DXGI_MODE_SCANLINE_ORDER_UNSPECIFIED;
    swapChainDesc.BufferDesc.Scaling = DXGI_MODE_SCALING_UNSPECIFIED;

    // NOTE: this is just an example and should be changed based on users needs
    swapChainDesc.SampleDesc.Count = 1;
    swapChainDesc.SampleDesc.Quality = 0;
    swapChainDesc.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
    swapChainDesc.BufferCount = 1;
    swapChainDesc.OutputWindow = m_hWnd;
#ifdef _DEBUG
    swapChainDesc.Windowed = TRUE;
#else
    swapChainDesc.Windowed = FALSE;
#endif
    swapChainDesc.SwapEffect = DXGI_SWAP_EFFECT_DISCARD;
    swapChainDesc.Flags = DXGI_SWAP_CHAIN_FLAG_ALLOW_MODE_SWITCH;

    D3D_FEATURE_LEVEL featureLevel = D3D_FEATURE_LEVEL_11_0;
    HRESULT hRes = D3D11CreateDeviceAndSwapChain(
        NULL,                       // adapter
        D3D_DRIVER_TYPE_HARDWARE,   // driver type
        NULL,                       // software
        0,                          // flags
        &featureLevel,              // feature levels
        1,                          // feature levels
        D3D11_SDK_VERSION,          // SDK version
        &swapChainDesc,             // swap chain desc
        &m_SwapChain,               // swap chain
        &m_Device,                  // device
        NULL,                       // feature level
        &m_DeviceContext            // immediate context
    );

    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create device and swap chain!" << std::endl;
        return false;
    }

    return true;
}

bool  DXApplication::InitRenderTargetView()
{
    CComPtr<ID3D11Texture2D> backBuffer;
    HRESULT hRes = m_SwapChain->GetBuffer(0, __uuidof(ID3D11Texture2D), (void **)&backBuffer);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to get back buffer!" << std::endl;
        return false;
    }

    hRes = m_Device->CreateRenderTargetView(backBuffer, NULL, &m_RenderTargetView);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create render target view!" << std::endl;
        return false;
    }

    return true;
}

bool  DXApplication::InitPixelShader()
{
    char* shaderSource = nullptr;
    DWORD shaderSize   = 0;
    if (!LoadShader(IDR_PIXEL_SHADER, &shaderSource, &shaderSize))
    {
        std::cout << "DirectX: Unable to load pixel shader!" << std::endl;
        return false;
    }

    LPCSTR entryPoint = nullptr;
    switch (m_Settings->Format())
    {
        case VIDEO_FORMAT::FORMAT_RGB101010:
            entryPoint = "PS_rgb10";
            break;
        case VIDEO_FORMAT::FORMAT_YUV420_8:
            entryPoint = "PS_yuv420_8";
            break;
    }

    CComPtr<ID3DBlob> shader;
    CComPtr<ID3DBlob> errorMessage;
    HRESULT hRes = D3DCompile(shaderSource, shaderSize, NULL, NULL, NULL, entryPoint, "ps_5_0", D3DCOMPILE_DEBUG, 0, &shader, &errorMessage);
    if (FAILED(hRes))
    {
        const char* pErrMsg = (const char*) errorMessage->GetBufferPointer();
        std::cout << "DirectX: Unable to compile pixel shader!" << std::endl;
        std::cout << "DirectX: Error msg:" << pErrMsg << std::endl;
        return false;
    }
    hRes = m_Device->CreatePixelShader(shader->GetBufferPointer(), shader->GetBufferSize(), NULL, &m_PixelShader);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create pixel shader!" << std::endl;
        return false;
    }

    return true;
}

bool  DXApplication::InitVertexShader()
{
    char* shaderSource = nullptr;
    DWORD shaderSize   = 0;
    if (!LoadShader(IDR_VERTEX_SHADER, &shaderSource, &shaderSize))
    {
        std::cout << "DirectX: Unable to load vertex shader!" << std::endl;
        return false;
    }

    CComPtr<ID3DBlob> shader;
    CComPtr<ID3DBlob> errorMessage;
    HRESULT hRes = D3DCompile(shaderSource, shaderSize, NULL, NULL, NULL, "Video_VS", "vs_5_0", D3DCOMPILE_DEBUG, 0, &shader, &errorMessage);
    if (FAILED(hRes))
    {
        const char* pErrMsg = (const char*) errorMessage->GetBufferPointer();
        std::cout << "DirectX: Unable to compile vertex shader!" << std::endl;
        std::cout << "DirectX: Error msg:" << pErrMsg << std::endl;
        return false;
    }
    hRes = m_Device->CreateVertexShader(shader->GetBufferPointer(), shader->GetBufferSize(), NULL, &m_VertexShader);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create vertex shader!" << std::endl;
        return false;
    }

    return true;
}

bool  DXApplication::AllocateConstBuffer()
{
    PS_CONSTANT_BUFFER PsConstData = { 0, 0, 0, 0 };
    PsConstData.frameWidth = m_Settings->Width();
    PsConstData.frameHeight = m_Settings->Height();

    D3D11_BUFFER_DESC cbDesc = { 0 };
    cbDesc.ByteWidth = sizeof(PS_CONSTANT_BUFFER);
    cbDesc.Usage = D3D11_USAGE_DYNAMIC;
    cbDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
    cbDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    cbDesc.MiscFlags = 0;
    cbDesc.StructureByteStride = 0;

    D3D11_SUBRESOURCE_DATA InitData = { 0 };
    InitData.pSysMem = &PsConstData;
    InitData.SysMemPitch = 0;
    InitData.SysMemSlicePitch = 0;

    HRESULT  hRes = m_Device->CreateBuffer(&cbDesc, &InitData, &m_ConstantBuffer);
    if (FAILED(hRes))
    {
        std::cout << "DirectX: Unable to create constant buffer!" << std::endl;
        return false;
    }

    return true;
}

bool  DXApplication::AllocateDataBuffer(int dataBufferSize)
{
    // check the passed in buffer size is valid
    // obviously a negative number wouldn't work
    if (dataBufferSize <= 0)
    {
        std::cout << "DirectX: Invalid buffer size!" << std::endl;
        return false;
    }
    
    // allocate the required buffer
    D3D11_BUFFER_DESC bufferDesc = { 0 };
    bufferDesc.ByteWidth = dataBufferSize;
	bufferDesc.Usage = D3D11_USAGE_DYNAMIC;
    bufferDesc.BindFlags = D3D11_BIND_SHADER_RESOURCE;
	bufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
    bufferDesc.MiscFlags = 0; 
    bufferDesc.StructureByteStride = 0;

    for (int i = 0; i < m_NumBuffers; i++)
    {
        HRESULT  hRes = m_Device->CreateBuffer(&bufferDesc, NULL, &m_DataBuffer[i]);
        if (FAILED(hRes))
        {
            std::cout << "DirectX: Unable to create buffer!" << std::endl;
            return false;
        }
    }

    return true;
}

bool  DXApplication::InitShaderResourceView(int dataBufferSize)
{
    D3D11_SHADER_RESOURCE_VIEW_DESC SRVDesc;
    SRVDesc.ViewDimension = D3D11_SRV_DIMENSION_BUFFER;
    if (m_Settings->Format() == VIDEO_FORMAT::FORMAT_RGB101010)
    {
        // we receive an image in 10 10 10 2 format - in the pixel 
        // shader we read each byte of the buffer, we pack it 
        // ourselves because the format information might not be exact 
        // as the R10G10B10A2 expected by the output surface, and we 
        // normalize the values, allowing the output surface to scale 
        // values properly
        SRVDesc.Format = DXGI_FORMAT_R8_UINT;
        SRVDesc.Buffer.FirstElement = 0;
        SRVDesc.Buffer.NumElements = dataBufferSize;
    }
    else
    {
        SRVDesc.Format = DXGI_FORMAT_R8_UNORM;
        SRVDesc.Buffer.FirstElement = 0;
        SRVDesc.Buffer.NumElements = dataBufferSize;
    }

    for (int i = 0; i < m_NumBuffers; i++)
    {
        HRESULT hRes = m_Device->CreateShaderResourceView(m_DataBuffer[i], &SRVDesc, &m_ShaderResourceView[i]);
        if (FAILED(hRes))
        {
            std::cout << "DirectX: Unable to create shader resource view!" << std::endl;
            return false;
        }
    }

    return true;
}

bool DXApplication::Init(IApplication *pApplication, int /*argc*/, char** /*argv*/)
{
    m_Application = pApplication;
    m_Settings = pApplication->GetSettings();
    if (!m_Settings)
    {
        std::cout << "Error: No settings available!" << std::endl;
        return false;
    }

    std::cout << "DirectX: Initializing subsystem..." << std::endl;

    m_NumBuffers = m_Settings->NumBuffers();
    m_readBuffer = 0;
    m_waitBuffer = 0;

    // reserve enough space in the buffers that depend
    // on how many of them we have
    m_hDxSsgEvent.resize(m_NumBuffers, NULL);
    m_DataBuffer.resize(m_NumBuffers, NULL);
    m_ShaderResourceView.resize(m_NumBuffers, NULL);

    // initialize DirectX
    const int screenWidth  = GetSystemMetrics(SM_CXSCREEN);
    const int screenHeight = GetSystemMetrics(SM_CYSCREEN);
    if (!InitWindow(screenWidth, screenHeight))
    {
        std::cout << "DirectX: Unable to intialize display window!" << std::endl;
        return false;
    }

    if (!InitSwapChain(screenWidth, screenHeight))
    {
        std::cout << "DirectX: Unable to create device and swap chain!" << std::endl;
        return false;
    }

    if (!InitRenderTargetView())
    {
        std::cout << "DirectX: Unable to create render target view!" << std::endl;
        return false;
    }

    m_DeviceContext->OMSetRenderTargets(1, &m_RenderTargetView.p, NULL);

    // setup viewport
    D3D11_VIEWPORT viewport;

    viewport.Width = (float)screenWidth;
    viewport.Height = (float)screenHeight;
    viewport.MinDepth = 0.0f;
    viewport.MaxDepth = 1.0f;
    viewport.TopLeftX = 0.0f;
    viewport.TopLeftY = 0.0f;

    m_DeviceContext->RSSetViewports(1, &viewport);


    // extensions
    // NOTE: OCL uses this as a base class, but we shouldn't 
    //       care about the DX SSG extension in this case as
    //       OCL should be using its SSG extension at that point
    if (m_Settings->ApiExt() == API_EXTENSION::API_EXT_DX)
    {
        HRESULT hRes = InitExtensions();
        if (FAILED(hRes))
        {
            std::cout << "DirectX: DX extensions not found!" << std::endl;
            return false;
        }

        hRes = InitDXXSsg();
        if (FAILED(hRes))
        {
            std::cout << "DirectX: DX SSG extension not found!" << std::endl;
            return false;
        }

        std::cout << "DirectX: SSG extension found - using it" << std::endl;
    }


    // shaders
    if (!InitPixelShader())
    {
        std::cout << "DirectX: Unable to create pixel shader!" << std::endl;
        return false;
    }

    if (!InitVertexShader())
    {
        std::cout << "DirectX: Unable to create vertex shader!" << std::endl;
        return false;
    }

    // constant buffer
    if (!AllocateConstBuffer())
    {
        std::cout << "DirectX: Unable to create constant buffer!" << std::endl;
        return false;
    }

    // font wrapper
    if (!InitFontWrapper())
    {
        std::cout << "DirectX: Unable to create font wrapper!" << std::endl;
        return false;
    }

    // data buffer
    const long long  buffSize = m_Settings->FrameSize();
    if (buffSize > INT_MAX)
    {
        std::cout << "DirectX: Buffer too large for allocation!" << std::endl;
        return false;
    }

    const int  dataBufferSize = (int) buffSize;
    if (!AllocateDataBuffer(dataBufferSize))
    {
        std::cout << "DirectX: Unable to allocate data buffer!" << std::endl;
        return false;
    }

    if (!InitShaderResourceView(dataBufferSize))
    {
        std::cout << "DirectX: Unable to create shader resource view!" << std::endl;
        return false;
    }

    // query
    if (m_Settings->Sync())
    {
        D3D11_QUERY_DESC queryDesc;
        queryDesc.Query = D3D11_QUERY_EVENT;
        queryDesc.MiscFlags = 0;

        m_Device->CreateQuery(&queryDesc, &m_Query);
        m_DeviceContext->End(m_Query);
    }

    return true;
}

void DXApplication::Term()
{
    // extensions
    if (m_DxSsgExt != NULL)
    {
        // event handles
        for (int i = 0; i < m_hDxSsgEvent.size(); i++)
            if (m_hDxSsgEvent[i])
            {
                m_DxSsgExt->DestroyEvent(m_hDxSsgEvent[i]);
                m_hDxSsgEvent[i] = NULL;
            }

        m_DxSsgExt->Release();
        m_DxSsgExt = NULL;
    }

    if (m_DxExt != NULL)
    {
        m_DxExt->Release();
        m_DxExt = NULL;
    }
    
    if (m_SwapChain)
    {
        m_SwapChain->SetFullscreenState(FALSE, NULL);
    }

    m_FontWrapper.Release();
    m_ConstantBuffer.Release();
    m_Query.Release();
    for (int i = 0; i < m_ShaderResourceView.size(); i++)
        m_ShaderResourceView[i].Release();
    for (int i = 0; i < m_DataBuffer.size(); i++)
        m_DataBuffer[i].Release();
    m_VertexShader.Release();
    m_PixelShader.Release();
    m_RenderTargetView.Release();
    m_DeviceContext.Release();
    m_SwapChain.Release();
    m_Device.Release();

    // UI
    if (m_hWnd != NULL)
    {
        DestroyWindow(m_hWnd);
        m_hWnd = NULL;
    }
}


//
//
// Control methods
//
//

void DXApplication::Run()
{
    while (true)
    {
        MSG msg;
        if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }

        if (msg.message == WM_QUIT)
            break;

        if (!m_Paused)
        {
            if (!m_Application->HandleFrame())
                break;
        }
    }
}

void DXApplication::Pause(bool pause)
{
    m_Paused = pause;
}

void DXApplication::Stop()
{
    // pause so we no longer read any frames...
    // PostQuitMessage is a "post" so it will
    // return immediately, which means that it is 
    // possible we can still have frame processing
    // before the Quit occurs, which is the reason 
    // for the pause at this point
    Pause(true);

    // here we have to wait for all the transfers to finish before 
    // we terminate the app
    for (int i = 0; i < m_hDxSsgEvent.size(); i++)
        if (m_hDxSsgEvent[i])
            m_DxSsgExt->WaitEvent(m_hDxSsgEvent[i]);

    PostQuitMessage(0);
}

void DXApplication::SetFullScreen(bool fullScreen)
{
    if (m_SwapChain)
    {
        m_SwapChain->SetFullscreenState(fullScreen, NULL);
    }
}



//
//
// Handle a frame
//
//

void DXApplication::BeginFrame(bool clear)
{
    if (m_Query)
    {
        while (S_OK != m_DeviceContext->GetData(m_Query, NULL, 0, 0))
        {
        }
    }

    if (clear)
    {
        float color[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
        m_DeviceContext->ClearRenderTargetView(m_RenderTargetView, color);
    }
}

void DXApplication::DrawFrame()
{
    // Currently we are reading N'th frame to the buffer, and drawing the (N-1)'s frame
    if (m_hDxSsgEvent[m_waitBuffer])
    {
        HRESULT  hRes = m_DxSsgExt->WaitEvent(m_hDxSsgEvent[m_waitBuffer]);
        if (FAILED(hRes))
            return;
    }

    m_DeviceContext->VSSetShader(m_VertexShader, NULL, 0);

    m_DeviceContext->PSSetShaderResources(0, 1, &m_ShaderResourceView[m_waitBuffer].p);
    m_DeviceContext->PSSetShader(m_PixelShader, NULL, NULL);
    m_DeviceContext->PSSetConstantBuffers(0, 1, &m_ConstantBuffer.p);

    m_DeviceContext->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);
    m_DeviceContext->Draw(3, 0);

    m_waitBuffer = (m_waitBuffer + 1) % m_NumBuffers;
}


void DXApplication::DrawStatus(char * text)
{
    if (!m_displayText)
        return;

    if (!m_FontWrapper)
        return;

    float fontSize = 40.0f;

    int height = m_Settings->Height();

    m_FontWrapper->DrawString(
        m_DeviceContext,
        CA2W(text),
        fontSize,
        0.0f,
        0.0f,
        0xffffffff,
        FW1_NOGEOMETRYSHADER
        );

}

void DXApplication::DrawSeekbar(int frameNumber, int totalNumberOfFrames)
{
    D3D11_MAPPED_SUBRESOURCE mappedSubresource;
    memset(&mappedSubresource, 0, sizeof(mappedSubresource));

    HRESULT hRes = m_DeviceContext->Map(m_ConstantBuffer, 0, D3D11_MAP_WRITE_NO_OVERWRITE, 0, &mappedSubresource);
    if (SUCCEEDED(hRes))
    {
        ((PS_CONSTANT_BUFFER *)mappedSubresource.pData)->a = frameNumber;
        ((PS_CONSTANT_BUFFER *)mappedSubresource.pData)->b = totalNumberOfFrames;
        m_DeviceContext->Unmap(m_ConstantBuffer, 0);
    }
}

void DXApplication::EndFrame()
{
    if (m_Settings->VSync())
    {
        m_SwapChain->Present(1, 0);
    }
    else
    {
        m_SwapChain->Present(0, 0);
    }

    if (m_Query)
    {
        m_DeviceContext->End(m_Query);
    }
}


//
//
// P2P file access through DX extension
//
//

bool DXApplication::P2PIsExtensionActive()
{
    return m_DxSsgExt != nullptr;
}

bool DXApplication::P2POpenFile(const char * filename)
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return false;

	HRESULT hRes = m_DxSsgExt->CreateFile(CA2W(filename), AmdDxSsgFile_Read, &m_hDxSsgFileHandle);
    if (FAILED(hRes))
    {
        P2PCloseFile();
        return false;
    }

    AmdDxSsgFileInfo fileInfo = { 0, 0 };
    hRes = m_DxSsgExt->GetFileInfo(m_hDxSsgFileHandle, &fileInfo);
    if (FAILED(hRes))
    {
        P2PCloseFile();
        return false;
    }

	m_SsgFileSize.QuadPart = fileInfo.fileSize;
    m_sectorSize = fileInfo.blockSize;

    return true;
}

unsigned long long DXApplication::P2PGetFileSize()
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return 0;

    return m_SsgFileSize.QuadPart;
}

unsigned long long DXApplication::P2PGetSectorSize()
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return 0;

    return m_sectorSize;
}

bool DXApplication::P2PReadFile(unsigned long long fileOffset, unsigned long long size)
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return false;


    const unsigned int  nrOfRegions = m_Settings->NumThreads();
    if (nrOfRegions == 0)
    {
        AmdDxSsgRegionDesc  region;
        region.bufferOffset = 0;
        region.fileOffset = fileOffset;
        region.regionSize = size;

        // read synchronous
        HRESULT hRes = m_DxSsgExt->ReadBufferFromFile(m_DataBuffer[m_readBuffer], m_hDxSsgFileHandle, 1, &region, NULL);
        if (FAILED(hRes))
            return false;
    }
    else
    {
        const unsigned long long  startFileOffset  = (fileOffset + m_sectorSize - 1) / m_sectorSize;
        const unsigned long long  numberOfSectors  = (size + m_sectorSize - 1) / m_sectorSize;
        const unsigned long long  sectorsPerRegion = (numberOfSectors + nrOfRegions - 1) / nrOfRegions;

        std::vector<AmdDxSsgRegionDesc> regions(nrOfRegions);
        for (unsigned int i = 0; i < nrOfRegions; i++)
        {
            // fill parameters
            regions[i].bufferOffset = sectorsPerRegion * i * m_sectorSize;
            regions[i].fileOffset = (startFileOffset + sectorsPerRegion * i) * m_sectorSize;

            // the last region might be smaller
            regions[i].regionSize = ((i < nrOfRegions - 1) ? sectorsPerRegion
                                                           : numberOfSectors - i * sectorsPerRegion) * m_sectorSize;
        }

        // create the sync event...
        if (!m_hDxSsgEvent[m_readBuffer])
        {
            HRESULT hRes = m_DxSsgExt->CreateEvent(&m_hDxSsgEvent[m_readBuffer]);
            if (FAILED(hRes))
                return false;
        }

        // read asynchronous
        HRESULT hRes = m_DxSsgExt->ReadBufferFromFile(m_DataBuffer[m_readBuffer], m_hDxSsgFileHandle, nrOfRegions, regions.data(), m_hDxSsgEvent[m_readBuffer]);
        if (FAILED(hRes))
            return false;
    }

    m_readBuffer = (m_readBuffer + 1) % m_NumBuffers;

    // update the first frame(s) flag - if we sent requests
    // for all frames, now we can start waiting on them to 
    // finish to start displaying them...
    //   m_NumBuffers == 1    -->  m_readBuffer = 0
    //   m_NumBuffers == 2    -->  m_readBuffer = 0
    //   m_NumBuffers == 3    -->  m_readBuffer = 2
    //   m_NumBuffers == 4    -->  m_readBuffer = 3
    // it seems that for double buffering, it's better
    // to submit read(0) and read(1) first and 
    // then go with wait(0), present(0), read(0), even though 
    // present(0), read(0) should cause a bit of serialization 
    // as we can't start reading into the buffer that's in use 
    // till present is done with it
    if (m_IsFirstFrame && 
         (  ((m_NumBuffers != 2) && (m_readBuffer == m_NumBuffers - 1)) ||
            ((m_NumBuffers == 2) && (m_readBuffer == 0))  )  )
        m_IsFirstFrame = false;

    return true;
}

void DXApplication::P2PCloseFile()
{
    // if extension is not active, P2P is not available
    if (!P2PIsExtensionActive())
        return;

    if (m_hDxSsgFileHandle)
    {
        HRESULT hRes = m_DxSsgExt->ReleaseFile(m_hDxSsgFileHandle);
        if (FAILED(hRes))
        {
            std::cout << "DirectX: Unable to close file handle!" << std::endl;
        }
        else
        {
            m_hDxSsgFileHandle = NULL;
        }
    }
}
