// 
// Notice Regarding Standards.  AMD does not provide a license or sublicense to
// any Intellectual Property Rights relating to any standards, including but not
// limited to any audio and/or video codec technologies such as MPEG-2, MPEG-4;
// AVC/H.264; HEVC/H.265; AAC decode/FFMPEG; AAC encode/FFMPEG; VC-1; and MP3
// (collectively, the "Media Technologies"). For clarity, you will pay any
// royalties due for such third party technologies, which may include the Media
// Technologies that are owed as a result of AMD providing the Software to you.
// 
// MIT license 
// 
// Copyright (c) 2016 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//

#ifdef _WIN32
#define _CRT_NONSTDC_NO_DEPRECATE
#define _CRT_SECURE_NO_WARNINGS
#endif

#include <iostream>
#include <algorithm>
#include <vector>
#include <string.h>

#ifdef _WIN32
#include <windows.h>

#include <atlbase.h>
#include <atlcom.h>

#else
#include <cassert>
#define _ASSERTE assert
#endif

#include "Application.h"


Application::Application(ISettings * pSettings)
{
    m_Settings = nullptr;

    m_API = nullptr;
    m_pAPIFrame = nullptr;
    m_pAPI_P2P = nullptr;

    Clear();
    
    m_Settings = pSettings;
    m_DataOffset = m_Settings->FrameOffset();
}

Application::~Application()
{
}

void Application::SetEngine(IGraphicsEngine* pEngine)
{
    m_API = dynamic_cast<IGraphicsAPI*>(pEngine);
    m_pAPIFrame = dynamic_cast<IGraphicsAPIFrame*>(pEngine);
    m_pAPI_P2P = dynamic_cast<IGraphicsAPI_P2P*>(pEngine);
}

void Application::Clear()
{
    m_FileSize = 0;
    m_FrameSize = 0;
    m_MouseDown = false;

    m_FrameCounter = 0;
    m_LocalFrameCounter = 0;
    m_DataOffset = 0;

    m_FPS = 0;
    m_ReadSpeed = 0;
    m_AvgReadSpeed = 0;

    m_StatsTimer.Start();
    m_FrameTimer.Start();

#ifdef _DEBUG
    m_FullScreen = false;
#else
    m_FullScreen = true;
#endif

    m_Paused = false;
    m_LoopCounter = 0;
    m_FPSOverride = false;
    m_TotalDataSize = 0;
}

bool Application::Init(int argc, char ** argv)
{
    Clear();
    m_DataOffset = m_Settings->FrameOffset();


    //////////////////////////
    // initialize graphics API
    if (!m_API->Init(this, argc, argv))
    {
        std::cerr << "Error: Unable to initialize graphics API!" << std::endl;
        return false;
    }


    //////////////////////////
    // open files   
    if (!m_pAPI_P2P->P2PIsExtensionActive())
    {
        std::cerr << "Error: No API to access files found!" << std::endl;
        return false;
    }

    // only a single file stream is supported
    if (!m_pAPI_P2P->P2POpenFile(m_Settings->Filename(0)))
    {
        std::cerr << "Error: Extension unable to open file!" << std::endl;
        return false;
    }

    m_FileSize = m_pAPI_P2P->P2PGetFileSize();


    //////////////////////////
    // calculate frame size
    m_FrameSize = m_Settings->FrameSize();
    if (m_FrameSize > m_FileSize)
    {
        std::cerr << "Error: Requested data size is larger than available file size!" << std::endl;
        return false;
    }


    //////////////////////////
    // validate alignment
    unsigned long long sectorSize = m_pAPI_P2P->P2PGetSectorSize();
    if (sectorSize <= 0)
    {
        std::cerr << "Error: Unable to get sector size!" << std::endl;
        return false;
    }

    if (m_FrameSize % sectorSize != 0)
    {
        std::cerr << "Error: Frame size is not a multiple of sector size!" << std::endl;
        return false;
    }

    if (m_DataOffset % sectorSize != 0)
    {
        std::cerr << "Error: Frame offset will force reads not alligned on sector size boundary!" << std::endl;
        return false;
    }


    //////////////////////////
    // set full screen mode
	m_API->SetFullScreen(m_FullScreen);

    return true;
}

void Application::Run()
{
	m_API->Run();
}

void Application::Term()
{
    m_pAPI_P2P->P2PCloseFile();
    m_API->Term();
}

void  Application::CalculateStats()
{
    int statisticsWindow = 60; // frames
    if ((m_FrameCounter % statisticsWindow) == 0)
    {
        unsigned long long elapsedTime = m_StatsTimer.GetElapsedMicroseconds();
        if (elapsedTime != 0)
        {
            // compute FPS
            double d = 1000000.0 / elapsedTime * statisticsWindow;
            m_FPS = (int)(d + 0.5);

            // compute read speed
            m_ReadSpeed = (int)(m_TotalDataSize / elapsedTime);
        }

        m_StatsTimer.Start();

        m_TotalDataSize = 0;
        if (m_FrameCounter > 300 && m_ReadSpeed > 100)
        {
            m_AvgReadSpeed = m_AvgReadSpeed + (float)(m_ReadSpeed - m_AvgReadSpeed) / (m_LocalFrameCounter + 1);
            m_LocalFrameCounter++;
        }
    }
}


void  Application::GenerateFpsText(char* pTxt)
{
    if (!pTxt)
    {
        _ASSERTE(pTxt);
        return;
    }

    const char* platform = nullptr;
    switch (m_Settings->ApiExt())
    {
    case API_EXT_DX:
        platform = "DirectX";
        break;
    case API_EXT_OPENGL:
        platform = "OpenGL";
        break;
    case API_EXT_OPENCL:
        platform = "OpenCL";
        break;
    default:
        platform = "Unknown";
        break;
    }

    const char* format = "RGB";
    switch (m_Settings->Format())
    {
    case VIDEO_FORMAT::FORMAT_YUV420_8:
        format = "YUV 4:2:0 8-bit";
        break;
    }

    sprintf(pTxt,
        "%s %s %dx%d FPS: %d Rate: %d MB/sec Avg: %.2f MB/sec\n",
        platform,
        format,
        m_Settings->Width(),
        m_Settings->Height(),
        m_FPS,
        m_ReadSpeed,
        m_AvgReadSpeed);

    // //Raw 8K video, FPS = xx, Rate = xxxx MB / s

    char resolution[64] = { 0 };
    if (m_Settings->Width() == 7680 && m_Settings->Height() == 4320)
    {
        strcpy(resolution, "8K");
    }
    else if (m_Settings->Width() == 3840 && m_Settings->Height() == 2160)
    {
        strcpy(resolution, "4K");
    }
    else
    {
        sprintf(resolution, "%dx%d", m_Settings->Width(), m_Settings->Height());
    }

    sprintf(pTxt,
        "Raw %s video, FPS = %d, Rate = %d MB / s\n",
        resolution,
        m_FPS,
        m_ReadSpeed);
}


bool Application::HandleFrame()
{
    // determine if we will display the frame or skip it...
    if (!m_FPSOverride)
    {
        if (m_FrameCounter == 0)
        {
            m_FrameTimer.Start();
        }
        else
        {
            unsigned long long  elapsedTime = m_FrameTimer.GetElapsedMicroseconds();
            if (elapsedTime <= (m_FrameCounter * (1000000L / m_Settings->FrameRate())))
                return true;
        }
    }


    m_pAPIFrame->BeginFrame(false);

    m_pAPI_P2P->P2PReadFile(m_DataOffset, m_FrameSize);

    const unsigned long long  numFrames   = m_FileSize / (m_FrameSize + m_Settings->FrameOffset());
    const unsigned long long  frameNumber = m_DataOffset / (m_FrameSize + m_Settings->FrameOffset());

    // Read the first frame without drawing, when read the second frame, 
    // we will start to draw the first frame, then we will read the N'th frame, 
    // and draw the (N-1)'s frame
    if (m_pAPI_P2P && (m_pAPI_P2P->IsFirstFrame() == false))
    {
        m_pAPIFrame->DrawFrame();
    }


    // compute statistics
    m_TotalDataSize += m_FrameSize;
    CalculateStats();

    // generate the text string to be displayed for FPS, etc...
    char txt[256];
    GenerateFpsText(txt);


    if (m_pAPI_P2P)
    {
        if (!m_pAPI_P2P->IsFirstFrame())
        {
            m_pAPIFrame->DrawStatus(txt);
            m_pAPIFrame->DrawSeekbar((int)frameNumber, (int)numFrames);
            m_pAPIFrame->EndFrame();
        }
    }

    m_DataOffset += m_FrameSize + m_Settings->FrameOffset();

    if (m_DataOffset < m_FileSize)
    {
        m_FrameCounter++;
    }
    else
    {
        // rewind
        m_LoopCounter++;
        if (m_LoopCounter >= m_Settings->Loop())
        {
            return false;
        }

        m_DataOffset = m_Settings->FrameOffset();
        m_FrameCounter = 0;
    }

    return true;
}

void Application::HandleKey(int keyCode)
{
    switch (keyCode)
    {
    case KEY_ESCAPE:
        m_API->Stop();
        break;
    case KEY_SPACE:
        m_Paused = !m_Paused;
        m_API->Pause(m_Paused);
        break;
    case KEY_FULLSCREEN:
        m_FullScreen = !m_FullScreen;
        m_API->SetFullScreen(m_FullScreen);
        break;
    case KEY_FPSCONTROL:
        m_FPSOverride = !m_FPSOverride;
        m_FrameCounter = 0;
        break;
    }
}

void Application::HandleMouseDown(int x, int y)
{
    m_MouseDown = true;

    if (IsSeekBar(x, y))
    {
        Seek(x, y);
    }
}

void Application::HandleMouseMove(int x, int y)
{
    if (m_Paused && m_MouseDown && IsSeekBar(x, y))
    {
        Seek(x, y);
    }
}

void Application::HandleMouseUp(int x, int y)
{
    m_MouseDown = false;
}

bool Application::IsSeekBar(int x, int y)
{
    return ((y * 100 / m_Settings->Height()) > 90);
}

void Application::Seek(int x, int y)
{
    // seek
    unsigned long long numFrames = m_FileSize / (m_Settings->FrameOffset() + m_FrameSize);
    unsigned long long frameNumber = x * numFrames / m_Settings->Width();
    m_DataOffset = frameNumber * (m_Settings->FrameOffset() + m_FrameSize) + m_Settings->FrameOffset();

    if (m_Paused)
    {
        HandleFrame();
    }

    m_FrameCounter = 0;
}